/*
 * <<< p_sharedbus.cc >>>
 *
 * --- Shared bus architecture simulator with probabilistic simulation
 *     'isis_p_sharedbus'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <iomanip>
#include <isis/isis.h>

using namespace std;

typedef unsigned int word;

int main(int, char** argv)
{
	bool verbose_flag = 0;
	word top = 0, size = 0x100;
	int punum = 1, delay = 0;
	double access_prob = .1, read_prob = .8;
	argument_parser arg((const char* const*)(argv + 1));
	if (arg.defined('z')) verbose_flag = true;
	if (arg['a'] != NULL) access_prob = atof(arg['a']);
	if (arg['d'] != NULL) delay = atoi(arg['d']);
	if (arg['p'] != NULL) punum = atoi(arg['p']);
	if (arg['r'] != NULL) read_prob = atof(arg['r']);
	if (arg['s'] != NULL) size = atoi(arg['s']);
	if (arg['w'] != NULL) read_prob = atof(arg['w']) - 1;
	access_prob = (access_prob < 0) ? 0 : (access_prob > 1) ? 1 : access_prob;
	read_prob = (read_prob < 0) ? 0 : (read_prob > 1) ? 1 : read_prob;
	const int iter_count = (arg.argument()[0] != NULL) ?
		atoi(arg.argument()[0]) : 100;
	{
		cout << "number of processors: " << punum		<< endl
			 << "access probability:   " << access_prob << endl
			 << "read probability:     " << read_prob	<< endl
			 << "memory delay:         " << delay		<< endl
			 << "iteration counts:     " << iter_count	<< endl << endl;
	}
	random_access_processor<word> *pu;
	pu = new random_access_processor<word>[punum];
	memory_control_unit<word> mem_ctl;
	mapped_memory<word> mem_buf(top, size);
	bus_port<word> bus_if;
	int* stall_count = new int[punum];
	int* read_count = new int[punum];
	int* write_count = new int[punum];
	int bususe_count = 0;
	mem_ctl.port_ref().connect(bus_if);
	mem_ctl.connect_memory(mem_buf);
	mem_ctl.set_read_wait(delay);
	mem_ctl.set_write_wait(delay);
	{
		for (int i = 0; i < punum; i++) {
			pu[i].set_access_probability(access_prob);
			pu[i].set_read_probability(read_prob);
			pu[i].set_address(mem_buf.top(), mem_buf.size());
			pu[i].port_ref().connect(bus_if);
			pu[i].port_ref().set_id(i);
			stall_count[i] = read_count[i] = write_count[i] = 0;
		}
	}
	for (int iter = 0; iter < iter_count; iter++) {
		int i;
		if (verbose_flag) {
			cerr << "--- clk:" << iter << " ---" << endl;
			cerr << "in:" << endl;
			for (i = 0; i < punum; i++) {
				cerr << "pu" << i << ":  " << pu[i] << endl;
			}
			cerr << "bus: " << bus_if << endl;
		}
		if (!bus_if.is_ready() || bus_if.is_owned()) bususe_count++;
		for (i = 0; i < punum; i++) {
			if (pu[i].is_stall()) stall_count[i]++;
			if (pu[i].is_reading()) read_count[i]++;
			if (pu[i].is_writing()) write_count[i]++;
			if (pu[i].is_writing()) write_count[i]++;
			pu[i].clock_in();
		}
		mem_ctl.clock_in();
		if (verbose_flag) {
			cerr << "out:" << endl;
			for (i = 0; i < punum; i++) {
				cerr << "pu" << i << ":  " << pu[i] << endl;
			}
			cerr << "bus: " << bus_if << endl;
		}
		for (i = 0; i < punum; i++) pu[i].clock_out();
		mem_ctl.clock_out();
	}
	if (verbose_flag) {
		cerr << mem_buf << endl;
	}
	{
		cout << "puid:    calc  stall   read  write" << endl;
		for (int i = 0; i < punum; i++) {
			int st_c = stall_count[i],
				ca_c = iter_count - st_c,
				rd_c = read_count[i],
				wr_c = write_count[i],
				tt_c = iter_count;
			cout << setw(4) << i << ": "
				 << setw(7) << setprecision(5) << 100. * ca_c / tt_c
				 << setw(7) << setprecision(5) << 100. * st_c / tt_c
				 << setw(7) << setprecision(5) << 100. * rd_c / tt_c
				 << setw(7) << setprecision(5) << 100. * wr_c / tt_c << endl;
		}
		cout << "bus use: "
			 << setw(7) << setprecision(5) << 100. * bususe_count / iter_count
			 << endl;
	}
	delete[] pu;
	delete[] stall_count;
	delete[] read_count;
	delete[] write_count;
	return 0;
}
