/*
 * <<< p_hypercube.cc >>>
 *
 * --- Probabilistic hypercube simulator
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <algorithm>
#include <cassert>
#include <cstdlib>
#include <vector>
#include <iostream>
#include <iomanip>
#include <string>
#include <isis/isis.h>
#include "hypercube_ecube_router.h"
#include "hypercube_duato_router.h"

using namespace std;

#define rnd() ((unsigned int)((seed = 1566083941UL * seed + 1) >> 16))
static unsigned long seed = 1;

// typedefs
typedef size_t address_type;
typedef network_packet_base<address_type> packet_type;
typedef router<packet_type> router_base_type;
typedef dummy_network_interface<packet_type> network_interface_type;
typedef packet_type::timestamp_type timestamp_type;

static const char* to_basename(const char* s)
{
	const char* head = s;
	while (*s != '\0') {
		if (*s == '/') head = s + 1;
		s++;
	}
	return head;
}

void usage(const char* name)
{
	cout
	<< "usage: " << name << " [options]" << endl
	<< "options:" << endl
	<< "  -b<n>, --buffer=<n>    set channel buffer size."				<< endl
	<< "  -c<n>, --cluster=<n>   set size of packet cluster."			<< endl
	<< "  -C<n>, --cooldown=<n>  set last cooldown count."				<< endl
	<< "  -i<n>, --idling=<n>    set beginning idling count."			<< endl
	<< "  -l<n>, --length=<n>    set packet length."					<< endl
	<< "  -n<n>, --iteration=<n> set iteration count."					<< endl
	<< "  -r<n>, --rate=<n>      set access rate of each node."			<< endl
	<< "  -x<n>, --dimension=<n> set dimension of hypercube."			<< endl
	<< "         --routing=<str> set routing algorithm:"				<< endl
	<< "                           ecube:  ecube routing (default)"		<< endl
	<< "                           duato:  Duato's protocol"			<< endl
	<< "         --pattern=<str> set access pattern:"					<< endl
	<< "                           random:    uniform random (default)"	<< endl
	<< "                           reversal:  bit-reversal"				<< endl
	<< "                           transpose: matrix transpose"			<< endl
	<< "  -h,    --help          print this message."					<< endl
	<< "  -v,    --verbose       verbosely output."						<< endl;
}

int main(int, char** argv)
{
	// set default values
	size_t dim_size = 6, buf_size = 8, pkt_len = 8, cluster_size = 1,
		   iter_count = 2000, idling_count = 0, cooldown_count = 0;
	double access_rate = .1;
	enum { ECUBE, DUATO } routing = ECUBE;
	enum { RANDOM, REVERSAL, TRANSPOSE } pattern = RANDOM;
	string routing_name = "ecube routing";
	string pattern_name = "uniform random";
	vector<address_type> dst_tbl;
	vector<size_t> send_count_tbl;
	bool verbose_flag = false;

	// read command-line option
	argument_parser arg((const char* const*)(argv + 1));
	if (arg.defined('h') || arg.defined("help"))
		{ usage(to_basename(argv[0])); return 0; }
	if (arg.defined('v') || arg.defined("verbose")) verbose_flag = true;
	if (arg['b'] != NULL)		  buf_size = size_t(atoi(arg['b']));
	if (arg["buffer"] != NULL)	  buf_size = size_t(atoi(arg["buffer"]));
	if (arg['c'] != NULL)		  cluster_size = size_t(atoi(arg['c']));
	if (arg["cluster"] != NULL)	  cluster_size = size_t(atoi(arg["cluster"]));
	if (arg['C'] != NULL)		  cooldown_count = size_t(atoi(arg['C']));
	if (arg["cooldown"] != NULL)
		cooldown_count = size_t(atoi(arg["cooldown"]));
	if (arg['i'] != NULL)		  idling_count = size_t(atoi(arg['i']));
	if (arg["idling"] != NULL)	  idling_count = size_t(atoi(arg["idling"]));
	if (arg['l'] != NULL)		  pkt_len = size_t(atoi(arg['l']));
	if (arg["length"] != NULL)	  pkt_len = size_t(atoi(arg["length"]));
	if (arg['n'] != NULL)		  iter_count = size_t(atoi(arg['n']));
	if (arg["iteration"] != NULL) iter_count = size_t(atoi(arg["iteration"]));
	if (arg['r'] != NULL)		  access_rate = double(atof(arg['r']));
	if (arg["rate"] != NULL)	  access_rate = double(atof(arg["rate"]));
	if (arg['x'] != NULL)		  dim_size = size_t(atoi(arg['x']));
	if (arg["dimension"] != NULL) dim_size = size_t(atoi(arg["dimension"]));
	if (arg["routing"] != NULL)	{
		if (string(arg["routing"]) == "duato") {
			routing = DUATO;
			routing_name = "duato's routing";
		}
	}
	if (arg["pattern"] != NULL) {
		if (string(arg["pattern"]) == "reversal") {
			pattern = REVERSAL;
			pattern_name = "bit-reversal";
		} else if (string(arg["pattern"]) == "transpose") {
			pattern = TRANSPOSE;
			pattern_name = "matrix transpose";
		}
	}

	// check network size
	if (dim_size <= 0) {
		cerr << "Network size is too small." << endl;
		exit(1);
	}

	// create routers, injectors, receivers and connect each other
	vector<router_base_type*> rt(1 << dim_size);
	vector<network_interface_type> ni(rt.size());
	{
		size_t i, j;
		for (i = 0; i < rt.size(); i++) {
			switch (routing) {
			case ECUBE:
				{
					typedef hypercube_ecube_router<packet_type> RT;
					RT* p = new RT;
					p->set_dimension(dim_size);
					rt[i] = p;
				}
				break;
			case DUATO:
				{
					typedef hypercube_duato_router<packet_type> RT;
					RT* p = new RT;
					p->set_dimension(dim_size);
					rt[i] = p;
				}
				break;
			}
			rt[i]->set_node_address(i);
			ni[i].set_node_address(i);
			for (j = 0; j < rt[i]->input_size(); j++) {
				rt[i]->set_buffer_size(j, buf_size);
			}
		}
		for (i = 0; i < rt.size(); i++) {
			for (j = 0; j < dim_size; j++) {
				size_t me = i, you = (i ^ (1 << j));
				rt[me]->output_channel(j).connect(rt[you]->input_channel(j));
			}
			rt[i]->output_channel(dim_size).connect(ni[i].input_channel());
			rt[i]->input_channel(dim_size).connect(ni[i].output_channel());
		}
		for (i = 0; i < rt.size(); i++) rt[i]->setup();
	}

	// show parameters
	cout << "--- condition ---" << endl
		 << "network size:   " << "2^" << dim_size
		 					  << " (" << rt.size() << ')' << endl
		 << "buffer size:    " << buf_size << endl
		 << "packet length:  " << pkt_len << endl
		 << "cluster size:   " << cluster_size << endl
		 << "access rate:    " << access_rate << endl
		 << "iterations:     " << iter_count
		 					   << " (idling:" << idling_count
							   << ", cooldown:" << cooldown_count << ')' << endl
		 << "routing:        " << routing_name << endl
		 << "access pattern: " << pattern_name << endl
		 << endl;

	// set destination table, send count table
	dst_tbl.resize(ni.size());
	send_count_tbl.resize(ni.size());
	if (pattern == REVERSAL || pattern == TRANSPOSE) {
		size_t equal_count = 0;
		for (size_t i = 0; i < dst_tbl.size(); i++) {
			address_type dst;
			switch (pattern) {
			case REVERSAL:
				{
					address_type tmp = i, max = (ni.size() - 1);
					dst = 0;
					while (max != 0) {
						dst = (dst << 1) | (tmp & 1);
						tmp >>= 1;
						max >>= 1;
					}
				}
				break;
			case TRANSPOSE:
				dst = i ^ (~((~0) << dim_size));
				break;
			default:
				assert(false);
				break;
			}
			dst = dst % dst_tbl.size();
			if (dst == i) equal_count++;
			dst_tbl[i] = dst;
		}
		if (equal_count > 0) {
			access_rate *= double(ni.size()) / (ni.size() - equal_count);
		}
	}
	fill(send_count_tbl.begin(), send_count_tbl.end(), 0);

	// main loop
	for (size_t iter = 0; ; iter++) {
		const unsigned long gen_threshold
			= (unsigned long)
			  (access_rate / (pkt_len * cluster_size) * 0x1000000U);
		size_t i;
		// at the end of idling, reset all counters
		if (iter == idling_count) {
			for (i = 0; i < rt.size(); i++) rt[i]->reset_counter();
			for (i = 0; i < ni.size(); i++) ni[i].reset_counter();
		}
		// show current status if verbose
		if (verbose_flag) {
			cout << "--- clk:" << iter << " ---" << endl;
			for (i = 0; i < rt.size(); i++) {
				cout << ' ' << setw(2) << i << ": " << *rt[i] << endl;
			}
			cout << endl;
		}
		if (iter >= iter_count) break;
		// generate packet and inject to NI
		if (iter < iter_count - cooldown_count) {
			for (i = 0; i < ni.size(); i++) {
				if (send_count_tbl[i] == 0) {
					unsigned long gen_rnd
						= ((unsigned long)(rnd() & 0x7ff8) << 9)
						| ((unsigned long)(rnd() & 0x7ff8) >> 3);
					if (gen_rnd < gen_threshold) {
						if (pattern == RANDOM) {
							address_type dst;
							do {
								dst = rnd() % ni.size();
							} while (dst == i);
							dst_tbl[i] = dst;
						}
						send_count_tbl[i] = cluster_size;
					}
				}
				if (ni[i].is_ready() && send_count_tbl[i] > 0) {
					ni[i].inject_packet(dst_tbl[i], pkt_len);
					send_count_tbl[i]--;
				}
			}
		}
		// clock_in
		for (i = 0; i < rt.size(); i++) {
			rt[i]->clock_in();
			ni[i].clock_in();
		}
		// clock_out
		for (i = 0; i < rt.size(); i++) {
			rt[i]->clock_out();
			ni[i].clock_out();
		}
	}

	// show results
	{
		size_t send_count = 0, recv_count = 0, total_hops = 0;
		timestamp_type total_latency = 0;
		double ave_hops, throughput, latency;
		for (size_t i = 0; i < rt.size(); i++) {
			total_hops += rt[i]->total_hop_count();
		}
		for (size_t i = 0; i < ni.size(); i++) {
			send_count += ni[i].send_packet_count();
			recv_count += ni[i].receive_packet_count();
			total_latency += ni[i].total_latency();
		}
		ave_hops = double(total_hops) / recv_count;
		throughput = double(recv_count) * pkt_len / ni.size()
					 / (iter_count - idling_count);
		latency = double(total_latency) / recv_count;
		cout << "--- results ---" << endl
			 << "sent:       " << setw(8) << send_count << endl
			 << "received:   " << setw(8) << recv_count << endl
			 << "ave. hops:  " << setw(8) << ave_hops << endl
			 << "throughput: " << setw(8) << throughput << endl
			 << "latency:    " << setw(8) << latency << endl;
	}

	// exit
	{
		for (size_t i = 0; i < rt.size(); i++) delete rt[i];
	}
	return 0;
}
