/*
 * <<< hypercube_ecube_router.h >>>
 *
 * --- Router class for Hypercube, with ecube routing
 *     Copyright (C) 2000-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef HYPERCUBE_ECUBE_ROUTER_H
#define HYPERCUBE_ECUBE_ROUTER_H

#include <cstddef>
#include <isis/isis.h>

template <class P>
class hypercube_ecube_router : public deterministic_router<P>
{
private:
	typedef hypercube_ecube_router<P> thisclass;
	typedef deterministic_router<P> inherited;
public:
	typedef typename inherited::packet_type packet_type;
	typedef typename inherited::node_address_type node_address_type;
private:
	size_t dim_, size_;
protected:
	typedef typename inherited::channel_controller channel_controller;
	virtual void routing(const packet_type&, channel_controller&);
public:
	hypercube_ecube_router(void);
	hypercube_ecube_router(const thisclass&);
	virtual ~hypercube_ecube_router() {}
	void set_dimension(size_t);
};

template <class P>
hypercube_ecube_router<P>::hypercube_ecube_router(void)
	: inherited(),
	  dim_(0),
	  size_(1)
{
	set_input_size(dim_ + 1);
	set_output_size(dim_ + 1);
	set_channel_size(dim_, 1);
	set_buffer_size(dim_, 1);
}

template <class P>
hypercube_ecube_router<P>::hypercube_ecube_router
	(const hypercube_ecube_router<P>& a)
	: inherited(a),
	  dim_(a.dim_),
	  size_(a.size_)
{}

template <class P>
void hypercube_ecube_router<P>::set_dimension(size_t a)
{
	size_t buf_size = buffer_size(0);
	dim_ = a;
	size_ = (1 << a);
	set_input_size(dim_ + 1);
	set_output_size(dim_ + 1);
	for (size_t i = 0; i < dim_ + 1; i++) {
		set_channel_size(i, 1);
		set_buffer_size(i, buf_size);
	}
}

template <class P>
void hypercube_ecube_router<P>::routing
	(const typename hypercube_ecube_router<P>::packet_type& pkt,
	 typename hypercube_ecube_router<P>::channel_controller& ctl)
{
	if (pkt.destination() == node_address()) {
		// destination is here, transmit to PE
		ctl.set_destination(dim_, 0);
	} else {
		size_t diff = (pkt.destination() ^ node_address());
		size_t next_ch;
		for (next_ch = 0; ; next_ch++) {
			if ((diff & 1) != 0) break;
			diff >>= 1;
		}
		assert(next_ch < dim_ + 1);
		ctl.set_destination(next_ch, 0);
	}
}

#endif /* HYPERCUBE_ECUBE_ROUTER_H */
