/* -*- C++ -*-
 *
 * <<< gdb_stub.h >>>
 *
 * --- Communication with gdb class 'gdb_stub'
 *     Copyright (C) 1999-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef GDB_STUB_H
#define GDB_STUB_H 1

#include <cassert>
#include <cctype>
#include <cstring>
#include <iostream>
#include <sstream>
#include <vector>
#include <isis/synchronous_unit.h>
#include <isis/single_processing_element.h>
#include <isis/multiprocessor.h>
#include <isis/gdb_port.h>

template <class Type>
class gdb_stub
{
private:
	static int char_to_int(char);
	static char int_to_char(int);
	synchronous_unit* sync_unit_;
	std::vector<single_processing_element<Type, Type, sizeof(Type), char>*>
		pu_table_;
	int puid_;
	gdb_port port_;
	std::ostream* debug_output_;
	bool verbose_flag_;
	const char* version_string_;
	const single_processing_element<Type, Type, sizeof(Type), char>& current_pe(void) const
		{ return *pu_table_[puid_]; }
	single_processing_element<Type, Type, sizeof(Type), char>& current_pe(void)
		{ return *pu_table_[puid_]; }
	void debug_output(const char*);
	void verbose_output(const char*);
	void manage_set_thread(const char*);
	void manage_last_signal(void);
	void manage_query(const char*);
	void manage_read_register(void);
	void manage_write_register(const char*);
	void manage_read_memory(const char*);
	void manage_write_memory(const char*);
	void manage_continue(const char*);
	void manage_kill(void);
	void manage_reset(void);
	void manage_echo(const char*);
	void manage_version(void);
	void manage_unknown_packet(const char*);
public:
	gdb_stub(void);
	virtual ~gdb_stub();
	void set_singleprocessor(single_processing_element<Type, Type, sizeof(Type), char>&);
	template <class P> void set_multiprocessor(multiprocessor<P>&);
	void use_tcp(void) { port_.use_tcp(); }
	void use_udp(void) { port_.use_udp(); }
	void set_port(int a) { port_.set_port(a); }
	void set_debug_output(std::ostream& os) { debug_output_ = &os; }
	void set_verbose(void) { verbose_flag_ = true; }
	void set_version_string(const char*);
	void service(void);
};

template <class Type>
int gdb_stub<Type>::char_to_int(char c)
{
	if (isdigit(c)) {
		return int(c - '0');
	} else if (isupper(c)) {
		return int(c - 'A') + 10;
	} else {
		return int(c - 'a') + 10;
	}
}

template <class Type>
char gdb_stub<Type>::int_to_char(int a)
{
	if (a < 10) {
		return char(a) + '0';
	} else {
		return char(a - 10) + 'a';
	}
}

template <class Type>
gdb_stub<Type>::gdb_stub(void)
	: sync_unit_(NULL),
	  puid_(0),
	  debug_output_(NULL),
	  verbose_flag_(false),
	  version_string_(NULL)
{}

template <class Type>
gdb_stub<Type>::~gdb_stub()
{
	if (version_string_ != NULL) delete[] version_string_;
}

template <class Type>
void gdb_stub<Type>::set_singleprocessor(single_processing_element<Type>& a)
{
	pu_table_.resize(1);
	puid_ = 0;
	pu_table_[puid_] = &a;
	sync_unit_ = &a;
}

template <class Type> template <class P>
void gdb_stub<Type>::set_multiprocessor(multiprocessor<P>& a)
{
	pu_table_.resize(a.processor_number());
	for (size_t i = 0; i < pu_table_.size(); i++) {
		pu_table_[i] = &(a.processing_element(i));
	}
	sync_unit_ = &a;
}

template <class Type>
void gdb_stub<Type>::debug_output(const char* s)
{
	if (debug_output_ == NULL) return;
	*debug_output_ << s << flush;
}

template <class Type>
void gdb_stub<Type>::verbose_output(const char* s)
{
	if (debug_output_ == NULL || !verbose_flag_) return;
	*debug_output_ << s << flush;
}

template <class Type>
void gdb_stub<Type>::manage_set_thread(const char* s)
{
	verbose_output("set thread: \"");
	verbose_output(s);
	verbose_output("\"\n");
	if (strcmp(s, "c-1") == 0) {
		port_.send("OK");
	} else if (strcmp(s, "g0") == 0) {
		port_.send("OK");
	} else {
		port_.send("ENN");
	}
}

template <class Type>
void gdb_stub<Type>::manage_last_signal(void)
{
	verbose_output("last signal: not implemented.\n");
	port_.send("S00");
}

template <class Type>
void gdb_stub<Type>::manage_query(const char* s)
{
	verbose_output("query: \"");
	verbose_output(s);
	verbose_output("\"\n");
	if (strcmp(s, "Offsets") == 0) {
		port_.send("Text=0;Data=0;Bss=0");
	} else {
		port_.send("ENN");
	}
}

template <class Type>
void gdb_stub<Type>::manage_read_register(void)
{
	verbose_output("read register: not implemented.\n");
	port_.send("00000000");
}

template <class Type>
void gdb_stub<Type>::manage_write_register(const char*)
{
	verbose_output("write register: not implemented.\n");
	port_.send("ENN");
}

template <class Type>
void gdb_stub<Type>::manage_read_memory(const char* s)
{
	// read memory packet: "m<adr>,<len>"
	Type adr = 0;
	unsigned int len = 0;
	// read adr
	while (isxdigit(*s)) adr = ((adr << 4) | Type(char_to_int(*s++)));
	s++; // skip ','
	while (isxdigit(*s)) len = ((len << 4) | char_to_int(*s++));
	// debug output
	{
		using namespace std;
		ostringstream ostr;
		ostr << hex << "read memory: adr:0x" << adr << ", length:0x" << len
			 << endl << ends;
		verbose_output(ostr.str().c_str());
	}
	// read memory and send reply
	{
		if (len > 1024) len = 1024;
		char* buf = new char[len * 2 + 1];
		assert(buf != NULL);
		char* p = buf;
		for (unsigned int i = 0; i < len; i++) {
			int c = int(current_pe().memory_map().read_char(adr + i));
			*(p++) = int_to_char((c >> 4) & 0xf);
			*(p++) = int_to_char(c & 0xf);
		}
		*p = '\0';
		port_.send(buf);
		delete[] buf;
	}
}

template <class Type>
void gdb_stub<Type>::manage_write_memory(const char* s)
{
	// write memory packet: "m<adr>,<len>:<value>"
	Type adr = 0;
	unsigned int len = 0;
	// read adr
	while (isxdigit(*s)) adr = ((adr << 4) | Type(char_to_int(*s++)));
	s++; // skip ','
	while (isxdigit(*s)) len = ((len << 4) | char_to_int(*s++));
	s++; // skip ':'
	// debug output
	{
		using namespace std;
		ostringstream ostr;
		ostr << hex << "write memory: adr:0x" << adr << ", length:0x" << len
			 << endl << ends;
		verbose_output(ostr.str().c_str());
	}
	// write memory
	for (unsigned int i = 0; i < len; i++) {
		if (*s == '\0') break;
		int c;
		c = char_to_int(*(s++));
		c = (c << 4) | char_to_int(*(s++));
		current_pe().memory_map().write_char(adr + i, char(c));
	}
	// send reply
	port_.send("OK");
}

template <class Type>
void gdb_stub<Type>::manage_continue(const char*)
{
	// check whether current pe is already halt
	if (current_pe().is_halt()) {
		debug_output("continue: current pe is already halted.\n");
		port_.send("Wff");
		return;
	}
	// continue the execution
	verbose_output("continue...\n");
	while (!current_pe().is_halt()) {
		sync_unit_->clock_in();
		sync_unit_->clock_out();
	}
	verbose_output("end.\n");
	// send exit status
	{
		char buf[] = "W00";
		int return_value = current_pe().commandline_status();
		buf[1] = int_to_char((return_value >> 4) & 0xf);
		buf[2] = int_to_char(return_value & 0xf);
		port_.send(buf);
	}
}

template <class Type>
void gdb_stub<Type>::manage_kill(void)
{
	verbose_output("kill\n");
	port_.send("OK");
}

template <class Type>
void gdb_stub<Type>::manage_reset(void)
{
	verbose_output("reset\n");
	sync_unit_->reset();
}

template <class Type>
void gdb_stub<Type>::manage_echo(const char* s)
{
	verbose_output("echo: \"");
	verbose_output(s);
	verbose_output("\"\n");
	port_.send(s);
}

template <class Type>
void gdb_stub<Type>::manage_version(void)
{
	if (version_string_ != NULL) {
		verbose_output("version: \"");
		verbose_output(version_string_);
		verbose_output("\"\n");
		port_.send(version_string_);
	} else {
		port_.send("");
	}
}

template <class Type>
void gdb_stub<Type>::manage_unknown_packet(const char* s)
{
	debug_output("unknown packet: \"");
	debug_output(s);
	debug_output("\"\n");
	port_.send("ENN");
}

template <class Type>
void gdb_stub<Type>::service(void)
{
	if (sync_unit_ == NULL) return;
	if (!port_.setup()) return;
	if (port_.is_tcp_mode()) {
		verbose_output("wait for connection ...");
		port_.connect();
		verbose_output(" connected.\n");
	}
	verbose_output("service start.\n");
	while (1) {
		char buf[65536];
		int len;
		while (1) {
			len = port_.receive(buf, sizeof(buf));
			if (len != 0) break;
		}
		if (len < 0) break;
		switch (buf[0]) {
		case '?':
			manage_last_signal();
			break;
		case 'c':
			manage_continue(buf + 1);
			break;
		case 'e':
			manage_echo(buf + 1);
			break;
		case 'g':
			manage_read_register();
			break;
		case 'G':
			manage_write_register(buf + 1);
			break;
		case 'H':
			manage_set_thread(buf + 1);
			break;
		case 'k':
			manage_kill();
			break;
		case 'm':
			manage_read_memory(buf + 1);
			break;
		case 'M':
			manage_write_memory(buf + 1);
			break;
		case 'q':
			manage_query(buf + 1);
			break;
		case 'r':
			manage_reset();
			break;
		case 'v':
			manage_version();
			break;
		default:
			manage_unknown_packet(buf);
			break;
		}
	}
	if (port_.is_tcp_mode()) port_.disconnect();
	port_.shutdown();
}

#endif /* GDB_STUB_H */
