/* -*- C++ -*-
 *
 * <<< tracewriter_unit.h >>>
 *
 * --- Access trace writer unit class 'tracewriter_unit'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TRACEWRITER_UNIT_H
#define TRACEWRITER_UNIT_H 1

#include <iostream>
#include <isis/single_port_synchronous_unit.h>
#include <isis/bus_port.h>

template <class A, class D = A>
class tracewriter_unit : public single_port_synchronous_unit
{
private:
	typedef tracewriter_unit thisclass;
	typedef single_port_synchronous_unit inherited;
public:
	typedef bus_port<A, D> port_type;
	typedef typename port_type::address_type address_type;
	typedef typename port_type::data_type data_type;
	typedef typename port_type::address_type size_type;
private:
	port_type bus_if;
	std::ostream* out_stream;
	int id_;
	unsigned long clock_value;
	address_type top_, old_adr;
	size_type size_;
public:
	tracewriter_unit(void);
	tracewriter_unit(const thisclass&);
	virtual ~tracewriter_unit();
	inline virtual void clock_in(void);
	inline virtual void clock_out(void);
	virtual void reset(void);
	virtual const port& port_ref(void) const { return bus_if; }
	virtual port& port_ref(void) { return bus_if; }
	int id(void) const { return id_; }
	bool is_valid_address(address_type a) const
		{ return a >= top_ && a < top_ + size_; }
	void set_id(int a) { id_ = a; }
	void reset_id(void) { id_ = -1; }
	void set_valid_address(address_type, size_type);
	void connect_stream(std::ostream& a) { out_stream = &a; }
	void disconnect_stream(void) { out_stream = NULL; }
};

template <class A, class D>
tracewriter_unit<A, D>::tracewriter_unit(void)
	: out_stream(NULL),
	  id_(-1),
	  clock_value(0),
	  top_(address_type(0)),
	  old_adr(address_type(0)),
	  size_(size_type(0))
{}

template <class A, class D>
tracewriter_unit<A, D>::tracewriter_unit
	(const tracewriter_unit<A, D>& a)
	: out_stream(a.out_stream),
	  id_(a.id_),
	  clock_value(a.clock_value),
	  top_(a.top_),
	  old_adr(a.old_adr),
	  size_(a.size_)
{}

template <class A, class D>
tracewriter_unit<A, D>::~tracewriter_unit()
{}

template <class A, class D>
inline void tracewriter_unit<A, D>::clock_in(void)
{
	using namespace std;
	address_type adr;
	bool read_flag = false;
	if (out_stream != NULL && bus_if.is_connected() &&
		((read_flag = bus_if.is_read_request()) || bus_if.is_write_request()) &&
		(adr = bus_if.address()) != old_adr && is_valid_address(adr)) {
		const ios::fmtflags flags = out_stream->flags();
		(*out_stream)
			<< hex << setw(8) << clock_value
			<< setw(4)
			<< ((id_ >= 0) ? id() : (bus_if.is_owned() ? bus_if.owner() : 0))
			<< (read_flag ? " r " : " w ")
			<< setw(8) << adr << '\n';
		out_stream->flags(flags);
		old_adr = adr;
	}
	clock_value++;
}

template <class A, class D>
inline void tracewriter_unit<A, D>::clock_out(void)
{}

template <class A, class D>
void tracewriter_unit<A, D>::reset(void)
{
	clock_value = 0;
	old_adr = address_type(0);
}

template <class A, class D>
void tracewriter_unit<A, D>::set_valid_address
	(tracewriter_unit<A, D>::address_type a,
	 tracewriter_unit<A, D>::size_type b)
{
	if (b > 0) top_ = a, size_ = b;
}

#endif /* TRACEWRITER_UNIT_H */
