/*
 * <<< torus_2d_router_test.cc >>>
 *
 * --- Test program for router class, 2d-torus version
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cassert>
#include <cstdlib>
#include <list>
#include <vector>
#include <iostream>
#include <iomanip>
#include <string>
#include <isis/network_packet_base.h>
#include "torus_2d_ecube_router_for_test.h"
#include "dummy_network_interface.h"

using namespace std;

#define rnd() ((unsigned int)((seed = 1566083941UL * seed + 1) >> 16))
static unsigned long seed = 1;

// typedefs
typedef size_t address_type;
typedef network_packet_base<address_type> packet_type;
typedef torus_2d_ecube_router<packet_type> router_type;
typedef dummy_network_interface<packet_type> network_interface_type;
typedef packet_type::timestamp_type timestamp_type;

int main(int, char** argv)
{
	// set default values
	size_t x_size = 2, y_size = 2, buf_size = 4, pkt_len = 4, iter_count = 100;
	double access_rate = .1;
	bool verbose_flag = false;

	// read command-line option
	while (*++argv != NULL && **argv == '-') {
		switch (*(*argv + 1)) {
		case 'x':
			x_size = (size_t)atoi(*argv + 2);
			break;
		case 'y':
			y_size = (size_t)atoi(*argv + 2);
			break;
		case 'b':
			buf_size = (size_t)atoi(*argv + 2);
			break;
		case 'l':
			pkt_len = (size_t)atoi(*argv + 2);
			break;
		case 'r':
			access_rate = (double)atof(*argv + 2);
			break;
		case 'n':
			iter_count = (size_t)atoi(*argv + 2);
			break;
		case 'v':
			verbose_flag = true;
			break;
		default:
			break;
		}
	}

	// create routers, injectors, receivers and connect each other
	vector<router_type> rt(x_size * y_size);
	vector<network_interface_type> ni(rt.size());
	{
		size_t i, j;
		for (i = 0; i < rt.size(); i++) {
			rt[i].set_node_address(i);
			rt[i].set_size(x_size, y_size);
			for (j = 0; j < rt[i].input_size(); j++) {
				rt[i].set_buffer_size(j, buf_size);
			}
			ni[i].set_node_address(i);
		}
		for (i = 0; i < y_size; i++) {
			for (j = 0; j < x_size; j++) {
				size_t me, right, left, down, up;
				me = i * x_size + j;
				right = i * x_size + ((j == x_size - 1) ? 0 : (j + 1));
				left  = i * x_size + ((j == 0) ? (x_size - 1) : (j - 1));
				down  = ((i == y_size - 1) ? 0 : (i + 1)) * x_size + j;
				up    = ((i == 0) ? (y_size - 1) : (i - 1)) * x_size + j;
				rt[me].output_channel(0).connect(rt[right].input_channel(1));
				rt[me].output_channel(1).connect(rt[left].input_channel(0));
				rt[me].output_channel(2).connect(rt[down].input_channel(3));
				rt[me].output_channel(3).connect(rt[up].input_channel(2));
				rt[me].input_channel(4).connect(ni[me].output_channel());
				rt[me].output_channel(4).connect(ni[me].input_channel());
			}
		}
		for (i = 0; i < rt.size(); i++) rt[i].setup();
	}

	// show parameters
	cout << "--- test condition ---" << endl
		 << "network size:  " << x_size << 'x' << y_size << endl
		 << "buffer size:   " << buf_size << endl
		 << "packet length: " << pkt_len << endl
		 << "access rate:   " << access_rate << endl
		 << "iterations:    " << iter_count << endl << endl;

	// main loop
	for (size_t iter = 0; ; iter++) {
		size_t i;
		// show current status if verbose
		if (verbose_flag) {
			cout << "--- clk:" << iter << " ---" << endl;
			for (i = 0; i < rt.size(); i++) {
				cout << ' ' << setw(2) << i << ": " << rt[i] << endl;
			}
			cout << endl;
		}
		if (iter >= iter_count) break;
		// generate packet and inject to NI
		for (i = 0; i < ni.size(); i++) {
			if (ni[i].is_ready() &&
				(double(rnd() & 0x7fff) / 0x8000) < access_rate / pkt_len) {
				address_type dst;
				do {
					dst = rnd() % (x_size * y_size);
				} while (dst == i);
				ni[i].inject_packet(dst, pkt_len);
			}
		}
		// clock_in
		for (i = 0; i < rt.size(); i++) {
			rt[i].clock_in();
			ni[i].clock_in();
		}
		// clock_out
		for (i = 0; i < rt.size(); i++) {
			rt[i].clock_out();
			ni[i].clock_out();
		}
	}

	// show results
	{
		size_t send_count = 0, recv_count = 0, total_hops = 0;
		timestamp_type total_latency = 0;
		double ave_hops, throughput, latency;
		for (size_t i = 0; i < rt.size(); i++) {
			total_hops += rt[i].total_hop_count();
		}
		for (size_t i = 0; i < ni.size(); i++) {
			send_count += ni[i].send_packet_count();
			recv_count += ni[i].receive_packet_count();
			total_latency += ni[i].total_latency();
		}
		ave_hops = double(total_hops) / recv_count;
		throughput = double(recv_count) * pkt_len / ni.size() / iter_count;
		latency = double(total_latency) / recv_count;
		cout << "--- results ---" << endl
			 << "sent:       " << setw(8) << send_count << endl
			 << "received:   " << setw(8) << recv_count << endl
			 << "ave. hops:  " << setw(8) << ave_hops << endl
			 << "throughput: " << setw(8) << throughput << endl
			 << "latency:    " << setw(8) << latency << endl;
	}

	// exit
	return 0;
}
