/* -*- C++ -*-
 *
 * <<< timer_unit.h >>>
 *
 * --- Timer Unit class 'timer_unit'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TIMER_UNIT_H
#define TIMER_UNIT_H 1

#include <iostream>
#include <isis/mapped_memory.h>
#include <isis/memory_control_unit.h>

template <class A, class D = A>
class timer_unit : public memory_control_unit<A, D, sizeof(D), char>
{
private:
	typedef timer_unit<A, D> thisclass;
	typedef memory_control_unit<A, D, sizeof(D), char> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
	typedef typename inherited::memory_type memory_type;
private:
	memory_type buf;
public:
	timer_unit(void);
	timer_unit(address_type);
	timer_unit(const thisclass&);
	virtual ~timer_unit();
	inline virtual void clock_in(void);
	virtual void reset(void);
	virtual void output(std::ostream&) const;
	const memory_type& memory(void) const { return buf; }
	memory_type& memory(void) { return buf; }
	void set_address(address_type);
	bool is_big_endian(void) const { return buf.is_big_endian(); }
	void set_big_endian(void) { buf.set_big_endian(); }
	bool is_little_endian(void) const { return buf.is_little_endian(); }
	void set_little_endian(void) { buf.set_little_endian(); }
	data_type clock_value(void) const { return buf.read(buf.top()); }
};

template <class A, class D>
timer_unit<A, D>::timer_unit(void)
	: buf(0, sizeof(data_type))
{
	inherited::connect_memory(buf);
	reset();
}

template <class A, class D>
timer_unit<A, D>::timer_unit(address_type a)
	: buf(a, sizeof(data_type))
{
	inherited::connect_memory(buf);
	reset();
}

template <class A, class D>
timer_unit<A, D>::timer_unit(const timer_unit<A, D>& a)
	: inherited(a),
	  buf(a.buf)
{
	inherited::connect_memory(buf);
}

template <class A, class D>
timer_unit<A, D>::~timer_unit()
{}

template <class A, class D>
void timer_unit<A, D>::output(std::ostream& os) const
{
	os << clock_value();
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class A, class D>
void timer_unit<A, D>::set_address(timer_unit<A, D>::address_type a)
{
	buf.set_top(a);
}

template <class A, class D>
inline void timer_unit<A, D>::clock_in(void)
{
	inherited::clock_in();
	buf.write(buf.top(), clock_value() + 1);
}

template <class A, class D>
void timer_unit<A, D>::reset(void)
{
	buf.write(buf.top(), data_type(0));
}

#endif /* TIMER_UNIT_H */
