/* -*- C++ -*-
 *
 * <<< random_access_processor.h >>>
 *
 * --- Random-access processor class 'random_access_processor'
 *     Copyright (C) 1997-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef RANDOM_ACCESS_PROCESSOR_H
#define RANDOM_ACCESS_PROCESSOR_H 1

#include <isis/random_access_processor_base.h>

template <class A, class D = A>
class random_access_processor : public random_access_processor_base<A, D>
{
private:
	typedef random_access_processor<A, D> thisclass;
	typedef random_access_processor_base<A, D> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
private:
	address_type top_, size_;
public:
	random_access_processor(void);
	random_access_processor(const thisclass&);
	virtual ~random_access_processor();
	virtual address_type get_read_address(void) const;
	virtual address_type get_write_address(void) const;
	virtual data_type get_write_data(void) const;
	address_type top(void) const { return top_; }
	address_type size(void) const { return size_; }
	address_type bottom(void) const { return top_ + size_; }
	void set_address(address_type, address_type);
};

template <class A, class D>
random_access_processor<A, D>::random_access_processor(void)
	: top_(address_type(0)),
	  size_(address_type(0))
{}

template <class A, class D>
random_access_processor<A, D>::random_access_processor
	(const random_access_processor<A, D>& a)
	: inherited(a),
	  top_(a.top_),
	  size_(a.size_)
{}

template <class A, class D>
random_access_processor<A, D>::~random_access_processor()
{}

template <class A, class D>
inline typename random_access_processor<A, D>::address_type
	random_access_processor<A, D>::get_read_address(void) const
{
	address_type tmp = address_type(double_random() * size()) + top();
	return tmp - tmp % sizeof(address_type);
}

template <class A, class D>
inline typename random_access_processor<A, D>::address_type
	random_access_processor<A, D>::get_write_address(void) const
{
	return get_read_address();
}

template <class A, class D>
inline typename random_access_processor<A, D>::data_type
	random_access_processor<A, D>::get_write_data(void) const
{
	return 1;
}

template <class A, class D>
void random_access_processor<A, D>::set_address
	(typename random_access_processor<A, D>::address_type a,
	 typename random_access_processor<A, D>::address_type b)
{
	if (b > 0) { // erase warning
		top_ = a, size_ = b;
	} else if (b == 0) {
		top_ = a, size_ = 0;
	}
}

#endif /* RANDOM_ACCESS_PROCESSOR_H */
