/*
 * <<< network_interface.h >>>
 *
 * --- Network interface class 'network_interface'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef NETWORK_INTERFACE_H
#define NETWORK_INTERFACE_H

#include <cassert>
#include <cstddef>
#include <iostream>
#include <isis/synchronous_unit.h>
#include <isis/network_packet_sender.h>
#include <isis/network_packet_receiver.h>

template <class P>
class network_interface : public synchronous_unit
{
private:
	typedef network_interface<P> thisclass;
	typedef synchronous_unit inherited;
public:
	typedef P packet_type;
	typedef typename P::address_type node_address_type;
	typedef typename P::timestamp_type timestamp_type;
private:
	network_packet_sender<packet_type> sender_;
	network_packet_receiver<packet_type> receiver_;
	node_address_type adr_;
	bool send_flag_, recv_flag_;
	timestamp_type time_;
	size_t send_pkt_count_, send_flit_count_, recv_pkt_count_, recv_flit_count_;
	timestamp_type total_latency_;
protected:
	inline bool is_ready_to_push(void) const { return sender_.empty(); }
	inline bool is_ready_to_snoop(size_t a) const { return receiver_.snoop(a); }
	inline bool is_ready_to_pop(void) const { return receiver_.full(); }
	void push_send_packet(packet_type* pkt) {
		pkt->set_timestamp(time_);
		send_pkt_count_++;
		send_flit_count_ += pkt->length();
		sender_.put(pkt);
	}
	packet_type* look_received_packet(void) { return receiver_.look(); }
	void pop_received_packet(void) { receiver_.get(); }
public:
	network_interface(void);
	network_interface(const thisclass&);
	virtual ~network_interface();
	virtual void output(std::ostream&) const;
	// state transition functions
	virtual void setup(void);
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void reset(void);
	virtual void reset_counter(void);
	// channel management
	const virtual_channel_input& input_channel(void) const
		{ return receiver_.channel(); }
	const virtual_channel_output& output_channel(void) const
		{ return sender_.channel(); }
	virtual_channel_input& input_channel(void)
		{ return receiver_.channel(); }
	virtual_channel_output& output_channel(void)
		{ return sender_.channel(); }
	// address management
	node_address_type node_address(void) const { return adr_; }
	void set_node_address(node_address_type a) { adr_ = a; }
	// diagnostics
	size_t send_packet_count(void) const { return send_pkt_count_; }
	size_t send_flit_count(void) const { return send_flit_count_; }
	size_t receive_packet_count(void) const { return recv_pkt_count_; }
	size_t receive_flit_count(void) const { return recv_flit_count_; }
	timestamp_type total_latency(void) const { return total_latency_; }
};

template <class P>
network_interface<P>::network_interface(void)
	: sender_(),
	  receiver_(),
	  adr_(),
	  send_flag_(false),
	  recv_flag_(false),
	  send_pkt_count_(0),
	  send_flit_count_(0),
	  recv_pkt_count_(0),
	  recv_flit_count_(0)
{}

template <class P>
network_interface<P>::network_interface
	(const network_interface<P>& a)
	: sender_(a.sender_),
	  receiver_(a.receiver_),
	  adr_(a.adr_),
	  send_flag_(a.send_flag_),
	  recv_flag_(a.recv_flag_),
	  send_pkt_count_(a.send_pkt_count_),
	  send_flit_count_(a.send_flit_count_),
	  recv_pkt_count_(a.recv_pkt_count_),
	  recv_flit_count_(a.recv_flit_count_)
{}

template <class P>
network_interface<P>::~network_interface()
{}

template <class P>
void network_interface<P>::setup(void)
{}

template <class P>
void network_interface<P>::clock_in(void)
{
	send_flag_ = (!output_channel().full(0) && !sender_.empty());
	recv_flag_ = (!input_channel().empty(0) && !receiver_.full());
}

template <class P>
void network_interface<P>::clock_out(void)
{
	if (send_flag_) {
		sender_.clock();
	}
	if (recv_flag_) {
		receiver_.clock();
		if (receiver_.full()) {
			packet_type* pkt = receiver_.look();
			assert(pkt->destination() == adr_);
			recv_pkt_count_++;
			recv_flit_count_ += pkt->length();
			total_latency_ += (time_ - pkt->timestamp());
		}
	}
	time_++;
}

template <class P>
void network_interface<P>::reset(void)
{
	send_flag_ = recv_flag_ = false;
	time_ = 0;
}

template <class P>
void network_interface<P>::reset_counter(void)
{
	send_pkt_count_ = send_flit_count_ = recv_pkt_count_ = recv_flit_count_ = 0;
	total_latency_ = 0;
}

template <class P>
void network_interface<P>::output(std::ostream& os) const
{
	os << "ch:" << input_channel()
	   << " send:" << sender_
	   << " recv:" << receiver_;
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* NETWORK_INTERFACE_H */
