/* -*- C++ -*-
 *
 * <<< multi_io_unit.h >>>
 *
 * --- Multiple I/O device class 'multi_io_unit'
 *     Copyright (C) 2000-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MULTI_IO_UNIT_H
#define MULTI_IO_UNIT_H 1

#include <cassert>
#include <cstddef>
#include <iostream>
#include <string>
#include <isis/cyclic_queue.h>
#include <isis/sysinfo_map.h>
#include <isis/fileio_map.h>
#include <isis/argument_map.h>
#include <isis/io_protocol.h>
#include <isis/comm_io_unit.h>

// #define MULTI_IO_UNIT_DEBUG

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char>
class multi_io_unit : public memory_control_unit<A, D, Dsz, Ch>
{
private:
	typedef multi_io_unit<A, D, Dsz, Ch> thisclass;
	typedef memory_control_unit<A, D, Dsz, Ch> inherited;
public:
	typedef typename inherited::address_type address_type;
	typedef typename inherited::data_type data_type;
	typedef typename inherited::char_type char_type;
	typedef typename inherited::size_type size_type;
	typedef typename inherited::memory_type memory_type;
	static const size_t sizeof_data_type = Dsz;
private:
	static const size_t default_buffer_size = 2048;
	enum {
		READY, SEND, RECEIVE
	} state_;
	enum {
		READY_TO_FINISH,
		SYSINFO_READ_INTEGER_FETCH_NAME_SIZE,
		SYSINFO_READ_INTEGER_FETCH_NAME,
		SYSINFO_READ_STRING_FETCH_NAME_SIZE,
		SYSINFO_READ_STRING_FETCH_NAME,
		SYSINFO_WRITE_INTEGER_FETCH_NAME_SIZE,
		SYSINFO_WRITE_INTEGER_FETCH_NAME,
		SYSINFO_WRITE_INTEGER_FETCH_VALUE,
		SYSINFO_WRITE_STRING_FETCH_NAME_SIZE,
		SYSINFO_WRITE_STRING_FETCH_NAME,
		SYSINFO_WRITE_STRING_FETCH_VALUE_SIZE,
		SYSINFO_WRITE_STRING_FETCH_VALUE,
		FILE_OPEN_REQUEST_FETCH_FLAGS,
		FILE_OPEN_REQUEST_FETCH_NAME_SIZE,
		FILE_OPEN_REQUEST_FETCH_NAME,
		FILE_CLOSE_REQUEST_FETCH_FD,
		FILE_READ_REQUEST_FETCH_FD,
		FILE_READ_REQUEST_SEND_ACK,
		FILE_READ_REQUEST_FETCH_IMAGE_SIZE,
		FILE_READ_REQUEST_SEND_IMAGE,
		FILE_WRITE_REQUEST_FETCH_FD,
		FILE_WRITE_REQUEST_SEND_ACK,
		FILE_WRITE_REQUEST_FETCH_IMAGE_SIZE,
		FILE_WRITE_REQUEST_FETCH_IMAGE,
		COMMANDLINE_READ_ARGV_FETCH_INDEX,
		COMMANDLINE_WRITE_STATUS_FETCH_STATUS
	} substate_;
	// hardware register and controller
	memory_type mem_buf;
	comm_io_unit<A, D, Dsz, Ch> comm_ctl;
	// for halt detection
	bool halt_flag_;
	// for sysinfo I/O
	sysinfo_map<data_type> sysinfo_buf;
	// for file I/O
	fileio_map file_buf;
	// for timer
	unsigned long timer_clock_value_;
	address_type timer_address_;
	// for commandline argument and status
	argument_map arg_buf;
	data_type commandline_status_;
	// temporary variables for communication
	cyclic_queue<char_type> tmp_queue;
	size_t remainder_count_;
	// temporary variables for sysinfo I/O
	std::string sysinfo_value_name;
	// temporary variables for file I/O
	int fileio_curr_fd, fileio_open_flags,
		fileio_image_remainder_count, fileio_image_processed_count;
	inline data_type get_integer_from_tmp_queue(void);
	inline void get_string_from_tmp_queue(std::string&);
	inline void put_integer_to_tmp_queue(data_type);
	inline void put_string_to_tmp_queue(const std::string&);
	inline void fetch_request_phase(void);
	inline void send_phase(void);
	inline void receive_phase(void);
public:
	multi_io_unit(void);
	multi_io_unit(address_type, size_type);
	multi_io_unit(const thisclass&);
	virtual ~multi_io_unit();
	virtual void clock_in(void);
	virtual void reset(void);
	void clear(void);
	// for inner memory
	const memory_type& memory(void) const { return mem_buf; }
	memory_type& memory(void) { return mem_buf; }
	void set_address(address_type, size_type);
	// for halt detection
	bool is_halt(void) const { return halt_flag_; }
	// for sysinfo I/O
	bool is_defined_sysinfo(const std::string& a) const
		{ return sysinfo_buf.is_defined(a); }
	bool is_integer_sysinfo(const std::string& a) const
		{ return sysinfo_buf.is_integer(a); }
	bool is_string_sysinfo(const std::string& a) const
		{ return sysinfo_buf.is_string(a); }
	data_type get_integer_sysinfo(const std::string& a) const
		{ return sysinfo_buf.get_integer(a); }
	const std::string& get_string_sysinfo(const std::string& a) const
		{ return sysinfo_buf.get_string(a); }
	void set_sysinfo(const std::string& a, data_type b)
		{ sysinfo_buf.set(a, b); }
	void set_sysinfo(const std::string& a, const std::string& b)
		{ sysinfo_buf.set(a, b); }
	void unset_sysinfo(const std::string& a) { sysinfo_buf.unset(a); }
	// for file I/O
	void set_file_table_size(size_t a) { file_buf.resize(a); }
	void set_file_input_stream(size_t a, std::istream& b)
		{ file_buf.set_input_stream(a, b); }
	void set_file_output_stream(size_t a, std::ostream& b)
		{ file_buf.set_output_stream(a, b); }
	// for timer
	unsigned long timer_clock_value(void) const { return timer_clock_value_; }
	// for commandline argument and status
	int commandline_argc(void) const { return arg_buf.argc(); }
	const char* commandline_argv(size_t i) const { return arg_buf.argv(i); }
	void set_commandline_argument(const char* const* a) { arg_buf.set(a); }
	int commandline_status(void) const { return int(commandline_status_); }
};

template <class A, class D, size_t Dsz, class Ch>
multi_io_unit<A, D, Dsz, Ch>::multi_io_unit(void)
	: inherited(),
	  state_(READY),
	  mem_buf(),
	  comm_ctl(),
	  halt_flag_(false),
	  sysinfo_buf(),
	  file_buf(0),
	  timer_clock_value_(0),
	  timer_address_(0),
	  arg_buf(),
	  commandline_status_(0),
	  tmp_queue(default_buffer_size)
{
	inherited::connect_memory(mem_buf);
	comm_ctl.connect_memory(mem_buf);
	set_sysinfo("timer_address", 0);
}

template <class A, class D, size_t Dsz, class Ch>
multi_io_unit<A, D, Dsz, Ch>::multi_io_unit
	(typename multi_io_unit<A, D, Dsz, Ch>::address_type a,
	 typename multi_io_unit<A, D, Dsz, Ch>::size_type b)
	: inherited(),
	  state_(READY),
	  mem_buf(a, b),
	  comm_ctl(),
	  halt_flag_(false),
	  sysinfo_buf(),
	  file_buf(0),
	  timer_clock_value_(0),
	  timer_address_(mem_buf.top() + mem_buf.size() - sizeof_data_type),
	  arg_buf(),
	  commandline_status_(0),
	  tmp_queue(default_buffer_size)
{
	inherited::connect_memory(mem_buf);
	comm_ctl.connect_memory(mem_buf);
	comm_ctl.set_body_size(mem_buf.size() - sizeof_data_type * 4);
	comm_ctl.setup();
	set_sysinfo("timer_address", timer_address_);
}

template <class A, class D, size_t Dsz, class Ch>
multi_io_unit<A, D, Dsz, Ch>::multi_io_unit
	(const multi_io_unit<A, D, Dsz, Ch>& a)
	: inherited(a),
	  state_(a.state_),
	  mem_buf(a.mem_buf),
	  comm_ctl(a.comm_ctl),
	  halt_flag_(a.halt_flag_),
	  sysinfo_buf(a.sysinfo_buf),
	  file_buf(a.file_buf),
	  timer_clock_value_(a.timer_clock_value_),
	  timer_address_(a.timer_address_),
	  arg_buf(a.arg_buf),
	  commandline_status_(a.commandline_status_),
	  tmp_queue(a.tmp_queue)
{
	inherited::connect_memory(mem_buf);
	comm_ctl.connect_memory(mem_buf);
	comm_ctl.set_body_size(mem_buf.size() - sizeof_data_type * 4);
	comm_ctl.setup();
	set_sysinfo("timer_address", timer_address_);
}

template <class A, class D, size_t Dsz, class Ch>
multi_io_unit<A, D, Dsz, Ch>::~multi_io_unit()
{}

template <class A, class D, size_t Dsz, class Ch>
void multi_io_unit<A, D, Dsz, Ch>::set_address
	(typename multi_io_unit<A, D, Dsz, Ch>::address_type a,
	 typename multi_io_unit<A, D, Dsz, Ch>::size_type b)
{
	mem_buf.set_top(a);
	mem_buf.resize(b);
	comm_ctl.set_body_size(b - sizeof_data_type * 4);
	comm_ctl.setup();
	timer_address_ = mem_buf.top() + mem_buf.size() - sizeof_data_type;
	set_sysinfo("timer_address", timer_address_);
}

template <class A, class D, size_t Dsz, class Ch>
inline typename multi_io_unit<A, D, Dsz, Ch>::data_type
multi_io_unit<A, D, Dsz, Ch>::get_integer_from_tmp_queue(void)
{
	data_type value = 0;
	for (size_t i = 0; i < sizeof_data_type; i++) {
		value = ((value << 8) | (unsigned char)(tmp_queue.front()));
		tmp_queue.pop();
	}
	return value;
}

template <class A, class D, size_t Dsz, class Ch>
inline void multi_io_unit<A, D, Dsz, Ch>::get_string_from_tmp_queue
	(std::string& a)
{
	a.resize(0);
	a.reserve(tmp_queue.size());
	while (!tmp_queue.empty()) {
		a += tmp_queue.front();
		tmp_queue.pop();
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void multi_io_unit<A, D, Dsz, Ch>::put_integer_to_tmp_queue
	(typename multi_io_unit<A, D, Dsz, Ch>::data_type a)
{
	for (size_t i = 0; i < sizeof_data_type; i++) {
		char_type c =
			char_type((a >> ((sizeof_data_type - 1 - i) * 8)) & 0xff);
		tmp_queue.push(c);
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void multi_io_unit<A, D, Dsz, Ch>::put_string_to_tmp_queue
	(const std::string& a)
{
	size_t length = a.length();
	for (size_t i = 0; i < length; i++) {
		tmp_queue.push(char_type(a[i]));
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void multi_io_unit<A, D, Dsz, Ch>::fetch_request_phase(void)
{
	char_type c = comm_ctl.get();
#ifdef MULTI_IO_UNIT_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "multi_io_unit: get request: 0x" << hex
			 << (unsigned int)(unsigned char)(c) << endl;
		cerr.flags(flags);
	}
#endif // MULTI_IO_UNIT_DEBUG
	switch (c) {
	case IO_HALT_REQUEST:
		halt_flag_ = true;
		tmp_queue.push(IO_COMMON_ACK);
		state_ = SEND;
		substate_ = READY_TO_FINISH;
		break;
	case IO_SYSINFO_READ_INTEGER_REQUEST:
		state_ = RECEIVE;
		substate_ = SYSINFO_READ_INTEGER_FETCH_NAME_SIZE;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_SYSINFO_READ_STRING_REQUEST:
		state_ = RECEIVE;
		substate_ = SYSINFO_READ_STRING_FETCH_NAME_SIZE;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_SYSINFO_WRITE_INTEGER_REQUEST:
		state_ = RECEIVE;
		substate_ = SYSINFO_WRITE_INTEGER_FETCH_NAME_SIZE;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_SYSINFO_WRITE_STRING_REQUEST:
		state_ = RECEIVE;
		substate_ = SYSINFO_WRITE_STRING_FETCH_NAME_SIZE;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_FILE_OPEN_REQUEST:
		state_ = RECEIVE;
		substate_ = FILE_OPEN_REQUEST_FETCH_FLAGS;
		remainder_count_ = 1;
		break;
	case IO_FILE_CLOSE_REQUEST:
		state_ = RECEIVE;
		substate_ = FILE_CLOSE_REQUEST_FETCH_FD;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_FILE_READ_REQUEST:
		state_ = RECEIVE;
		substate_ = FILE_READ_REQUEST_FETCH_FD;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_FILE_WRITE_REQUEST:
		state_ = RECEIVE;
		substate_ = FILE_WRITE_REQUEST_FETCH_FD;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_COMMANDLINE_READ_ARGC_REQUEST:
		tmp_queue.push(char_type(IO_COMMON_ACK));
		put_integer_to_tmp_queue(commandline_argc());
		state_ = SEND;
		substate_ = READY_TO_FINISH;
		break;
	case IO_COMMANDLINE_READ_ARGV_REQUEST:
		state_ = RECEIVE;
		substate_ = COMMANDLINE_READ_ARGV_FETCH_INDEX;
		remainder_count_ = sizeof_data_type;
		break;
	case IO_COMMANDLINE_WRITE_STATUS_REQUEST:
		state_ = RECEIVE;
		substate_ = COMMANDLINE_WRITE_STATUS_FETCH_STATUS;
		remainder_count_ = sizeof_data_type;
		break;
	default:
		tmp_queue.push(char_type(IO_COMMON_NACK));
		state_ = SEND;
		substate_ = READY_TO_FINISH;
		break;
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void multi_io_unit<A, D, Dsz, Ch>::send_phase(void)
{
	while (comm_ctl.can_put() && !tmp_queue.empty()) {
		comm_ctl.put(char_type(tmp_queue.front()));
#ifdef MULTI_IO_UNIT_DEBUG
		const long flags = cerr.flags();
		cerr << "multi_io_unit: put 0x" << hex
			 << (unsigned int)(unsigned char)(tmp_queue.front()) << endl;
		cerr.flags(flags);
#endif // MULTI_IO_UNIT_DEBUG
		tmp_queue.pop();
	}
	if (tmp_queue.empty()) {
		switch (substate_) {
		case READY_TO_FINISH:
			state_ = READY;
			break;
		case FILE_READ_REQUEST_SEND_ACK:
			state_ = RECEIVE;
			substate_ = FILE_READ_REQUEST_FETCH_IMAGE_SIZE;
			remainder_count_ = sizeof_data_type;
			break;
		case FILE_READ_REQUEST_SEND_IMAGE:
			{
				if (fileio_image_remainder_count == 0 ||
					!file_buf.good_input_stream(fileio_curr_fd)) {
					put_integer_to_tmp_queue(0);
					substate_ = READY_TO_FINISH;
				} else {
					size_t max = tmp_queue.max_size() - sizeof_data_type;
					if (int(max) > fileio_image_remainder_count) {
						max = size_t(fileio_image_remainder_count);
					}
					char_type* buf = new char_type[max];
					size_t size = 0;
					while (size < max) {
						char_type c;
						if (file_buf.eof(fileio_curr_fd) ||
							!file_buf.good_input_stream(fileio_curr_fd)) break;
						if (!file_buf.get(fileio_curr_fd, c)) break;
						buf[size] = c;
						size++;
					}
					put_integer_to_tmp_queue(size);
					for (size_t i = 0; i < size; i++) tmp_queue.push(buf[i]);
					delete[] buf;
					fileio_image_remainder_count -= size;
					if (size == 0) substate_ = READY_TO_FINISH;
				}
			}
			break;
		case FILE_WRITE_REQUEST_SEND_ACK:
			state_ = RECEIVE;
			substate_ = FILE_WRITE_REQUEST_FETCH_IMAGE_SIZE;
			remainder_count_ = sizeof_data_type;
			break;
		default:
			assert(false);
			break;
		}
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void multi_io_unit<A, D, Dsz, Ch>::receive_phase(void)
{
	while (comm_ctl.can_get() && remainder_count_ > 0) {
		char_type c = comm_ctl.get();
		tmp_queue.push(c);
#ifdef MULTI_IO_UNIT_DEBUG
		const long flags = cerr.flags();
		cerr << "multi_io_unit: get 0x" << hex
			 << (unsigned int)(unsigned char)(c) << endl;
		cerr.flags(flags);
#endif // MULTI_IO_UNIT_DEBUG
		remainder_count_--;
	}
	if (remainder_count_ == 0) {
		switch (substate_) {
		case SYSINFO_READ_INTEGER_FETCH_NAME_SIZE:
			remainder_count_ = get_integer_from_tmp_queue();
			substate_ = SYSINFO_READ_INTEGER_FETCH_NAME;
			break;
		case SYSINFO_READ_INTEGER_FETCH_NAME:
			get_string_from_tmp_queue(sysinfo_value_name);
			sysinfo_buf.get_integer(sysinfo_value_name);
				// dummy for gcc-2.95.2 optimize bug
			if (is_integer_sysinfo(sysinfo_value_name)) {
				const data_type& value =
					get_integer_sysinfo(sysinfo_value_name);
				tmp_queue.push(char_type(IO_COMMON_ACK));
				put_integer_to_tmp_queue(value);
			} else {
				tmp_queue.push(char_type(IO_COMMON_NACK));
			}
			state_ = SEND;
			substate_ = READY_TO_FINISH;
			break;
		case SYSINFO_READ_STRING_FETCH_NAME_SIZE:
			remainder_count_ = get_integer_from_tmp_queue();
			substate_ = SYSINFO_READ_STRING_FETCH_NAME;
			break;
		case SYSINFO_READ_STRING_FETCH_NAME:
			get_string_from_tmp_queue(sysinfo_value_name);
			if (is_string_sysinfo(sysinfo_value_name)) {
				const std::string& value =
					get_string_sysinfo(sysinfo_value_name);
				size_t length = value.length();
				tmp_queue.push(char_type(IO_COMMON_ACK));
				put_integer_to_tmp_queue(length);
				put_string_to_tmp_queue(value);
			} else {
				tmp_queue.push(char_type(IO_COMMON_NACK));
			}
			state_ = SEND;
			substate_ = READY_TO_FINISH;
			break;
		case SYSINFO_WRITE_INTEGER_FETCH_NAME_SIZE:
			remainder_count_ = get_integer_from_tmp_queue();
			substate_ = SYSINFO_WRITE_INTEGER_FETCH_NAME;
			break;
		case SYSINFO_WRITE_INTEGER_FETCH_NAME:
			get_string_from_tmp_queue(sysinfo_value_name);
			remainder_count_ = sizeof_data_type;
			substate_ = SYSINFO_WRITE_INTEGER_FETCH_VALUE;
			break;
		case SYSINFO_WRITE_INTEGER_FETCH_VALUE:
			{
				data_type value = get_integer_from_tmp_queue();
				set_sysinfo(sysinfo_value_name, value);
				tmp_queue.push(char_type(IO_COMMON_ACK));
				state_ = SEND;
				substate_ = READY_TO_FINISH;
			}
			break;
		case SYSINFO_WRITE_STRING_FETCH_NAME_SIZE:
			remainder_count_ = get_integer_from_tmp_queue();
			substate_ = SYSINFO_WRITE_STRING_FETCH_NAME;
			break;
		case SYSINFO_WRITE_STRING_FETCH_NAME:
			get_string_from_tmp_queue(sysinfo_value_name);
			remainder_count_ = sizeof_data_type;
			substate_ = SYSINFO_WRITE_STRING_FETCH_VALUE_SIZE;
			break;
		case SYSINFO_WRITE_STRING_FETCH_VALUE_SIZE:
			remainder_count_ = get_integer_from_tmp_queue();
			substate_ = SYSINFO_WRITE_STRING_FETCH_VALUE;
			break;
		case SYSINFO_WRITE_STRING_FETCH_VALUE:
			{
				std::string value;
				get_string_from_tmp_queue(value);
				set_sysinfo(sysinfo_value_name, value);
				tmp_queue.push(char_type(IO_COMMON_ACK));
				state_ = SEND;
				substate_ = READY_TO_FINISH;
			}
			break;
		case FILE_OPEN_REQUEST_FETCH_FLAGS:
			fileio_open_flags = tmp_queue.front();
			tmp_queue.pop();
			remainder_count_ = sizeof_data_type;
			substate_ = FILE_OPEN_REQUEST_FETCH_NAME_SIZE;
			break;
		case FILE_OPEN_REQUEST_FETCH_NAME_SIZE:
			remainder_count_ = get_integer_from_tmp_queue();
			substate_ = FILE_OPEN_REQUEST_FETCH_NAME;
			break;
		case FILE_OPEN_REQUEST_FETCH_NAME:
			{
				std::string name;
				size_t fd;
				get_string_from_tmp_queue(name);
				for (fd = 0; fd < file_buf.size(); fd++) {
					if (!file_buf.is_opened(fd) &&
						!file_buf.is_readable(fd) &&
						!file_buf.is_writable(fd)) break;
				}
				if (fd < file_buf.size() &&
					file_buf.open(fd, name.c_str(), fileio_open_flags)) {
					tmp_queue.push(char_type(IO_COMMON_ACK));
					put_integer_to_tmp_queue(fd);
				} else {
					tmp_queue.push(char_type(IO_COMMON_NACK));
				}
				state_ = SEND;
				substate_ = READY_TO_FINISH;
			}
			break;
		case FILE_CLOSE_REQUEST_FETCH_FD:
			{
				size_t fd = get_integer_from_tmp_queue();
				if (fd < file_buf.size() && file_buf.close(fd)) {
					tmp_queue.push(char_type(IO_COMMON_ACK));
				} else {
					tmp_queue.push(char_type(IO_COMMON_NACK));
				}
				state_ = SEND;
				substate_ = READY_TO_FINISH;
			}
			break;
		case FILE_READ_REQUEST_FETCH_FD:
			{
				size_t fd = get_integer_from_tmp_queue();
				if (fd < file_buf.size() && file_buf.is_readable(fd)) {
					fileio_curr_fd = fd;
					tmp_queue.push(char_type(IO_COMMON_ACK));
					state_ = SEND;
					substate_ = FILE_READ_REQUEST_SEND_ACK;
				} else {
					tmp_queue.push(char_type(IO_COMMON_NACK));
					state_ = SEND;
					substate_ = READY_TO_FINISH;
				}
			}
			break;
		case FILE_READ_REQUEST_FETCH_IMAGE_SIZE:
			fileio_image_remainder_count = get_integer_from_tmp_queue();
			state_ = SEND;
			substate_ = FILE_READ_REQUEST_SEND_IMAGE;
			break;
		case FILE_WRITE_REQUEST_FETCH_FD:
			{
				size_t fd = get_integer_from_tmp_queue();
				if (fd < file_buf.size() && file_buf.is_writable(fd)) {
					fileio_curr_fd = fd;
					tmp_queue.push(char_type(IO_COMMON_ACK));
					state_ = SEND;
					substate_ = FILE_WRITE_REQUEST_SEND_ACK;
				} else {
					tmp_queue.push(char_type(IO_COMMON_NACK));
					state_ = SEND;
					substate_ = READY_TO_FINISH;
				}
			}
			break;
		case FILE_WRITE_REQUEST_FETCH_IMAGE_SIZE:
			{
				size_t t = get_integer_from_tmp_queue();
				size_t max = tmp_queue.max_size();
				fileio_image_remainder_count = t;
				fileio_image_processed_count = 0;
				remainder_count_ = (t < max) ? t : max;
				substate_ = FILE_WRITE_REQUEST_FETCH_IMAGE;
			}
			break;
		case FILE_WRITE_REQUEST_FETCH_IMAGE:
			{
				size_t size = 0;
				bool break_flag = false;
				while (!tmp_queue.empty()) {
					if (!file_buf.good_output_stream(fileio_curr_fd)) {
						break_flag = true;
						break;
					}
					file_buf.put(fileio_curr_fd, tmp_queue.front());
					tmp_queue.pop();
					size++;
				}
				fileio_image_remainder_count -= size;
				fileio_image_processed_count += size;
				if (!break_flag && fileio_image_remainder_count > 0) {
					size_t t = fileio_image_remainder_count;
					size_t max = tmp_queue.max_size();
					remainder_count_ = (t < max) ? t : max;
					fileio_image_remainder_count = t;
				} else {
					file_buf.flush(fileio_curr_fd);
					put_integer_to_tmp_queue(fileio_image_processed_count);
					state_ = SEND;
					substate_ = READY_TO_FINISH;
				}
			}
			break;
		case COMMANDLINE_READ_ARGV_FETCH_INDEX:
			{
				data_type index = get_integer_from_tmp_queue();
				if (index < size_t(commandline_argc())) {
					std::string value = commandline_argv(index);
					data_type length = value.length();
					tmp_queue.push(char_type(IO_COMMON_ACK));
					put_integer_to_tmp_queue(length);
					put_string_to_tmp_queue(value);
				} else {
					tmp_queue.push(char_type(IO_COMMON_NACK));
				}
				state_ = SEND;
				substate_ = READY_TO_FINISH;
			}
			break;
		case COMMANDLINE_WRITE_STATUS_FETCH_STATUS:
			commandline_status_ = get_integer_from_tmp_queue();
			tmp_queue.push(char_type(IO_COMMON_ACK));
			state_ = SEND;
			substate_ = READY_TO_FINISH;
			break;
		default:
			assert(false);
			break;
		}
	}
}

template <class A, class D, size_t Dsz, class Ch>
void multi_io_unit<A, D, Dsz, Ch>::clock_in(void)
{
	inherited::clock_in();
	comm_ctl.clock_in();
	timer_clock_value_++;
	mem_buf.write(timer_address_, timer_clock_value_);
	switch (state_) {
	case READY:
		if (comm_ctl.can_get()) fetch_request_phase();
		break;
	case SEND:
		send_phase();
		break;
	case RECEIVE:
		receive_phase();
		break;
	}
}

template <class A, class D, size_t Dsz, class Ch>
void multi_io_unit<A, D, Dsz, Ch>::reset(void)
{
	inherited::reset();
	comm_ctl.reset();
	clear();
}

template <class A, class D, size_t Dsz, class Ch>
void multi_io_unit<A, D, Dsz, Ch>::clear(void)
{
	sysinfo_buf.clear();
	arg_buf.clear();
	timer_clock_value_ = 0;
	halt_flag_ = false;
	tmp_queue.clear();
	state_ = READY;
}

#endif /* MULTI_IO_UNIT_H */
