/*
 * <<< mp_network_interface_test.cc >>>
 *
 * --- Test program for mp_network_interface class
 *     Copyright (C) 2001-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <isis/argument_parser.h>
#include <isis/network_packet.h>
#include <isis/r3081_processing_element.h>
#include "loopback_router.h"

#define MP_NETWORK_INTERFACE_DEBUG

#include "mp_network_interface.h"

using namespace std;

typedef r3000_word word_type;
typedef size_t node_address_type;
typedef r3081_processing_element processing_element_type;
typedef network_packet<node_address_type, word_type> packet_type;
typedef loopback_router<packet_type> router_type;
typedef mp_network_interface<packet_type, word_type> network_interface_type;
typedef packet_type::timestamp_type timestamp_type;

int main(int, char** argv)
{
	const word_type local_memory_address = 0;
	const word_type local_memory_size = 0x20000000;
	const word_type network_interface_address = 0xbfe00000;
	unsigned int mem_delay = 0;
	size_t buf_size = 64;
	bool verbose_flag = false;

	// read commandline argument for simulator
	argument_parser arg((const char* const*)(argv + 1));
	if (arg['b'] != NULL)	   buf_size = atoi(arg['b']);
	if (arg["buffer"] != NULL) buf_size = atoi(arg["buffer"]);
	if (arg['d'] != NULL)	   mem_delay = atoi(arg['d']);
	if (arg["delay"] != NULL)  mem_delay = atoi(arg["delay"]);
	if (arg.defined('v') || arg.defined("verbose")) verbose_flag = true;
	if (arg.argument()[0] == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}

	processing_element_type pe;
	router_type rt;
	network_interface_type ni;

	// setup processing element
	pe.set_local_memory_area(local_memory_address, local_memory_size);
	pe.set_local_memory_read_wait(mem_delay);
	pe.set_local_memory_write_wait(mem_delay);
	pe.set_file_table_size(16);
	pe.set_standard_input_stream(cin);
	pe.set_standard_output_stream(cout);
	pe.set_standard_error_stream(cerr);
	if (!pe.load(arg.argument()[0])) {
		cerr << arg.argument()[0]
			 << ": No such file or directory." << endl;
		return 1;
	}
	pe.set_commandline_argument(arg.argument());
	pe.set_sysinfo("network_interface_address", network_interface_address);

	// setup router
	{
		for (size_t i = 0; i < rt.input_size(); i++) {
			rt.set_buffer_size(i, buf_size);
		}
	}

	// setup network interface
	ni.set_node_address(0);
	ni.bus_port_ref().connect(pe.processor().port_ref());
	ni.set_memory_address(network_interface_address);

	// last setup: router and network interface
	rt.input_channel(0).connect(ni.output_channel());
	rt.output_channel(0).connect(ni.input_channel());
	rt.setup();
	ni.setup();

	// execute simulation
	while (!pe.is_halt()) {
		if (verbose_flag) {
			cout << "--- clk:" << pe.timer_clock_value() << " ---" << endl
				 << "router:" << endl << rt << endl
				 << "network interface:" << endl << ni << endl
				 << "bus: " << pe.processor().port_ref() << endl
				 << endl;
		}
		pe.clock_in();
		ni.clock_in();
		rt.clock_in();
		pe.clock_out();
		ni.clock_out();
		rt.clock_out();
	}

	// exit
	return pe.commandline_status();
}
