/* -*- C++ -*-
 *
 * <<< mp_network_interface.h >>>
 *
 * --- Network interface unit for message-passing class
 *     'mp_network_interface'
 *     Copyright (C) 2000-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MP_NETWORK_INTERFACE_H
#define MP_NETWORK_INTERFACE_H 1

#include <cassert>
#include <iostream>
#include <vector>
#include <isis/cyclic_queue.h>
#include <isis/network_interface.h>
#include <isis/io_protocol.h>

#define MP_NETWORK_INTERFACE_DEBUG

template <class P, class A, class D = A, size_t Dsz = sizeof(D),
		  class Ch = char>
class mp_network_interface : public network_interface<P>
{
private:
	typedef mp_network_interface<P, A, D, Dsz, Ch> thisclass;
	typedef network_interface<P> inherited;
public:
	typedef typename inherited::packet_type packet_type;
	typedef typename inherited::node_address_type node_address_type;
	typedef A address_type;
	typedef D data_type;
	typedef Ch char_type;
	typedef A size_type;
	typedef bus_port<A, D> bus_port_type;
	typedef mapped_memory<A, D, Dsz, Ch> memory_type;
	typedef typename bus_port_type::transaction_id_type transaction_id_type;
	static const size_t sizeof_data_type = Dsz;
private:
	enum state_type {
		READY,
		REG_SINGLE_READ_WAIT,
		REG_SINGLE_READ_ACK,
		REG_SINGLE_READ_TRANS,
		REG_SINGLE_WRITE_WAIT,
		REG_SINGLE_WRITE_ACK,
		REG_SINGLE_WRITE_TRANS,
		REG_MULTI_READ_WAIT,
		REG_MULTI_READ_ACK,
		REG_MULTI_READ_TRANS,
		REG_MULTI_WRITE_WAIT,
		REG_MULTI_WRITE_ACK,
		REG_MULTI_WRITE_TRANS,
		SEND_DMA_READ_REQ_OWNER,
		SEND_DMA_READ_SEND,
		SEND_DMA_READ_WAIT,
		SEND_DMA_READ_TRANS,
		SEND_DMA_READ_END
	};
	struct send_request_type {
		size_t req_id;
		data_type pkt_id;
		address_type data_adr;
		size_t data_len; // in flit
		packet_type* pkt;
		bool progressive_flag;
		bool send_ready_flag;
		size_t rest_flits;
		send_request_type(void) : pkt(NULL) {}
		inline void output(std::ostream&) const;
	};
	struct receive_request_type {
		bool used_flag;
		bool anysrc_flag;
		bool anysize_flag;
		bool progressive_flag;
		node_address_type src;
		data_type pkt_id;
		address_type data_adr;
		size_t data_len; // in flit
		// ...
		receive_request_type(void) : used_flag(false) {}
		inline void output(std::ostream&) const;
	};
	struct receive_pool_entry_type {
		packet_type* pkt;
		// data_type pkt_id;
		bool recv_end_flag;
		// bool matched_flag;
		// size_t recvreqbuf_index;
		receive_pool_entry_type(void) : pkt(NULL) {}
		inline void output(std::ostream&) const;
	};
	const static size_t default_packet_header_size = 1; // in flits
	const static size_t default_max_packet_data_size = 256; // in flits
	const static size_t default_sendreqbuf_size = 256;
	const static size_t default_recvreqbuf_size = 256;
	const static size_t default_recvpool_size = 256;
	const static size_t request_table_size = 16; // in words
	const static size_t fixed_parameter_size = 8; // in words
	const static size_t variable_parameter_size = 8; // in words
	const static size_t fixed_hardware_register_size
		= request_table_size + fixed_parameter_size + variable_parameter_size;
	// state, bus i/f, hardware reg.
	state_type state_;
	bus_port_type bus_if_;
	memory_type mem_buf_;
	// send/receive request buffers, receive pools
	cyclic_queue<send_request_type> sendreq_buf_;
	std::vector<receive_request_type> recvreq_buf_;
	std::vector<receive_pool_entry_type> recv_pool_;
	// fixed parameters: packet size
	size_t packet_header_size_, max_packet_data_size_;
	// fixed parameters: memory access wait
	unsigned int mem_read_wait_, mem_write_wait_;
	// request buffer states: fetched send/receive
	bool fetched_sendreq_flag_, fetched_recvreq_flag_;
	// network access states: send/receive
	size_t sendreq_restflit_count_;
	bool receiving_flag_;
	size_t recvreq_count_, recvpool_count_, ready_to_recvreq_count_;
	// current processed request states for DMA trans.
	send_request_type* curr_sendreq_dmatrans_;
	// current processed receive states
	size_t receiving_pool_index_;
	// bus access states: memory access counter, address
	unsigned int wait_count_;
	address_type access_address_;
	size_t total_packet_count_, packet_index_;
	// inner utilities
	size_t sendflags_offset(void) const
		{ return fixed_hardware_register_size; }
	size_t recvflags_offset(void) const
		{ return fixed_hardware_register_size + send_request_buffer_size(); }
	size_t hardware_register_size(void) const
		{ return fixed_hardware_register_size + send_request_buffer_size() +
				 receive_request_buffer_size(); }
	void update_fixed_parameters(void);
	void update_variable_parameters(void);
	// fetch each command from processor
	inline void fetch_command_sense(void);
	inline void fetch_command_send(bool progressive_flag);
	inline void fetch_command_recv(bool progressive_flag);
	void fetch_command(int);
	// accept fetched command
	void accept_command(void);
	// self-check inner status consistency: for debug
	bool selfcheck(void) const;
	// receive buffer and receive pool deadlock check: for debug
	bool deadlock(void) const;
public:
	mp_network_interface(void);
	mp_network_interface(const thisclass&);
	virtual ~mp_network_interface();
	virtual void output(std::ostream&) const;
	// state transition
	virtual void setup(void);
	inline virtual void clock_in(void);
	inline virtual void clock_out(void);
	virtual void reset(void);
	// bus port
	const bus_port_type& bus_port_ref(void) const { return bus_if_; }
	bus_port_type& bus_port_ref(void) { return bus_if_; }
	// inner memory
	unsigned int memory_read_wait(void) const { return mem_read_wait_; }
	unsigned int memory_write_wait(void) const { return mem_write_wait_; }
	void set_memory_read_wait(unsigned int a) { mem_read_wait_ = a; }
	void set_memory_write_wait(unsigned int a) { mem_write_wait_ = a; }
	const memory_type& memory(void) const { return mem_buf_; }
	memory_type& memory(void) { return mem_buf_; }
	void set_memory_address(address_type a) { mem_buf_.set_top(a); }
	// buffer size functions: sendreqbuf(=sendpool), recvreqbuf, recvpool
	size_t send_request_buffer_size(void) const
		{ return sendreq_buf_.max_size(); }
	void set_send_request_buffer_size(size_t a)
		{ sendreq_buf_.reserve(a); send_pool_.resize(a); }
	size_t receive_request_buffer_size(void) const
		{ return recvreq_buf_.size(); }
	void set_receive_request_buffer_size(size_t a)
		{ recvreq_buf_.resize(a); }
	size_t receive_pool_size(void) const
		{ return recv_pool_.size(); }
	void set_receive_pool_size(size_t a)
		{ recv_pool_.resize(a); }
	// packet size functions: header, data
	size_t packet_header_size(void) const { return packet_header_size_; }
	void set_packet_header_size(size_t a) { packet_header_size_ = a; }
	size_t max_packet_data_size(void) const { return max_packet_data_size_; }
	void set_max_packet_data_size(size_t a) { max_packet_data_size_ = a; }
	// get variable state functions
	size_t pending_send_request_count(void) const
		{ return sendreq_buf_.size(); }
	size_t pending_receive_request_count(void) const
		{ return recvreq_count_; }
	size_t pending_receive_pool_count(void) const
		{ return recvpool_count_; }
};

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::update_fixed_parameters(void)
{
	const address_type top = mem_buf_.top();
	mem_buf_.write(
		top + MPNI_IO_OFFSET_MAX_PACKET_DATA_SIZE * sizeof_data_type,
		data_type(max_packet_data_size()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_SENDREQBUF_SIZE * sizeof_data_type,
		data_type(send_request_buffer_size()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_RECVREQBUF_SIZE * sizeof_data_type,
		data_type(receive_request_buffer_size()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_RECVPOOL_SIZE * sizeof_data_type,
		data_type(receive_pool_size()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_SENDFLAGS_OFFSET * sizeof_data_type,
		data_type(sendflags_offset()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_RECVFLAGS_OFFSET * sizeof_data_type,
		data_type(recvflags_offset()));
}

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::update_variable_parameters(void)
{
	const address_type top = mem_buf_.top();
	mem_buf_.write(
		top + MPNI_IO_OFFSET_PENDING_SENDREQ_COUNT * sizeof_data_type,
		data_type(pending_send_request_count()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_PENDING_RECVREQ_COUNT * sizeof_data_type,
		data_type(pending_receive_request_count()));
	mem_buf_.write(
		top + MPNI_IO_OFFSET_PENDING_RECVPOOL_COUNT * sizeof_data_type,
		data_type(pending_receive_pool_count()));
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void mp_network_interface<P, A, D, Dsz, Ch>::fetch_command_sense(void)
{
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "mp_network_interface(0x" << hex << node_address() << "): "
			 << "sense(): done." << endl
			 << *this << endl;
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
	update_variable_parameters();
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void mp_network_interface<P, A, D, Dsz, Ch>::fetch_command_send
	(bool progressive_flag)
{
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "mp_network_interface(0x" << hex << node_address() << "): "
			 << (progressive_flag ? "psend" : "send");
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
	const address_type top = mem_buf_.top();
	if (sendreq_buf_.full()) {
		// the send buffer is full, the request is rejected
		mem_buf_.write(top + MPNI_IO_OFFSET_REQUEST_ID * sizeof_data_type,
					   send_request_buffer_size());
#ifdef MP_NETWORK_INTERFACE_DEBUG
		{
			using namespace std;
			cerr << "rejected" << endl;
		}
#endif // MP_NETWORK_INTERFACE_DEBUG
		return;
	}
	// read request information
	node_address_type dst;
	data_type pkt_id;
	address_type data_adr;
	size_t data_len;
	{
		dst = node_address_type(mem_buf_.read(
			top + MPNI_IO_OFFSET_PARTNER_ADDRESS * sizeof_data_type));
		pkt_id = mem_buf_.read(
			top + MPNI_IO_OFFSET_PACKET_ID * sizeof_data_type);
		data_adr = address_type(mem_buf_.read(
			top + MPNI_IO_OFFSET_DATA_ADDRESS * sizeof_data_type));
		data_len = size_t(mem_buf_.read(
			top + MPNI_IO_OFFSET_DATA_SIZE * sizeof_data_type));
	}
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		// show all request arguments
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << hex
			 << "(dst:0x" << dst
			 << ", pktid:0x" << pkt_id
			 << ", adr:0x" << data_adr
			 << ", len:" << dec << data_len
			 << "): ";
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
	assert(data_len <= max_packet_data_size());
	// allocate id for the request
	size_t req_id;
	{
		const address_type base = top + sendflags_offset() * sizeof_data_type;
		size_t i;
		for (i = 0; i < send_request_buffer_size(); i++) {
			if (mem_buf_.read(base + i * sizeof_data_type) == data_type(0)) {
				break;
			}
		}
		assert(i < send_request_buffer_size());
		req_id = i;
	}
	// create a request entry and push it to the send request buffer
	{
		{
			send_request_type req;
			sendreq_buf_.push(req);
		}
		send_request_type& req(sendreq_buf_.back());
		packet_type* p = new packet_type;
		assert(p != NULL);
		const size_t data_flit_count = data_len + 1;
		const size_t all_flit_count = packet_header_size() + data_flit_count;
		req.req_id = req_id;
		req.pkt_id = pkt_id;
		req.data_adr = data_adr;
		req.data_len = data_len;
		req.pkt = p;
		req.progressive_flag = progressive_flag;
		req.send_ready_flag = false;
		req.rest_flits = all_flit_count;
		p->set_source(node_address());
		p->set_destination(dst);
		p->set_length(all_flit_count);
		p->set_data_size(data_flit_count);
		p->set_data(0, pkt_id);
	}
	// the request is now fetched, wait for accept...
	fetched_sendreq_flag_ = true;
	// update status information in the hardware register
	{
		mem_buf_.write(top + (sendflags_offset() + req_id) * sizeof_data_type,
					   data_type(1));
		mem_buf_.write(top + MPNI_IO_OFFSET_REQUEST_ID * sizeof_data_type,
					   data_type(req_id));
	}
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "fetched, id:" << dec << req_id << endl;
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void mp_network_interface<P, A, D, Dsz, Ch>::fetch_command_recv
	(bool progressive_flag)
{
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "mp_network_interface(0x" << hex << node_address() << "): "
			 << (progressive_flag ? "precv" : "recv");
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
	const address_type top = mem_buf_.top();
	if (pending_receive_request_count() >= receive_request_buffer_size()) {
		// the receive request buffer is full, the request is rejected
		mem_buf_.write(top + MPNI_IO_OFFSET_REQUEST_ID * sizeof_data_type,
					   receive_request_buffer_size());
#ifdef MP_NETWORK_INTERFACE_DEBUG
		{
			using namespace std;
			cerr << "rejected" << endl;
		}
#endif // MP_NETWORK_INTERFACE_DEBUG
		return;
	}
	// read request information
	node_address_type src;
	data_type pkt_id;
	address_type data_adr;
	size_t data_len;
	bool anysrc_flag, anysize_flag;
	{
		data_type flags;
		src = node_address_type(mem_buf_.read(
			top + MPNI_IO_OFFSET_PARTNER_ADDRESS * sizeof_data_type));
		pkt_id = mem_buf_.read(
			top + MPNI_IO_OFFSET_PACKET_ID * sizeof_data_type);
		data_adr = address_type(mem_buf_.read(
			top + MPNI_IO_OFFSET_DATA_ADDRESS * sizeof_data_type));
		data_len = size_t(mem_buf_.read(
			top + MPNI_IO_OFFSET_DATA_SIZE * sizeof_data_type));
		flags = size_t(mem_buf_.read(
			top + MPNI_IO_OFFSET_FLAGS * sizeof_data_type));
		anysrc_flag = ((flags & MPNI_IO_FLAGS_MASK_ANYSRC) != 0);
		anysize_flag = ((flags & MPNI_IO_FLAGS_MASK_ANYSIZE) != 0);
	}
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		// show all request arguments
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << hex
			 << "(src:0x" << src
			 << ", pktid:0x" << pkt_id
			 << ", adr:0x" << data_adr
			 << ", len:" << dec << data_len
			 << ", anysrc:" << (anysrc_flag ? "yes" : "no")
			 << ", anysize:" << (anysize_flag ? "yes" : "no")
			 << "): ";
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
	assert(data_len <= max_packet_data_size());
	// allocate id for the request
	size_t req_id;
	{
		size_t i;
		for (i = 0; i < receive_request_buffer_size(); i++) {
			if (!recvreq_buf_[i].used_flag) {
				break;
			}
		}
		assert(i < receive_request_buffer_size());
		req_id = i;
	}
	// put the request information into the receive request buffer
	{
		const size_t data_flit_count = data_len + 1;
		const size_t all_flit_count = packet_header_size() + data_flit_count;
		receive_request_type& req(recvreq_buf_[req_id]);
		req.used_flag = true;
		req.src = src;
		req.pkt_id = pkt_id;
		req.data_adr = data_adr;
		req.data_len = data_len;
		req.anysrc_flag = anysrc_flag;
		req.anysize_flag = anysize_flag;
		req.progressive_flag = progressive_flag;
		recvreq_count_++;
	}
	// the request is now fetched, wait for accept...
	fetched_recvreq_flag_ = true;
	// update status information in the hardware register
	{
		mem_buf_.write(top + (recvflags_offset() + req_id) * sizeof_data_type,
					   data_type(1));
		mem_buf_.write(top + MPNI_IO_OFFSET_REQUEST_ID * sizeof_data_type,
					   data_type(req_id));
	}
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "fetched, id:" << dec << req_id << endl;
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
}

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::fetch_command(int cmd)
{
	switch (cmd) {
	case MPNI_IO_SENSE:
		fetch_command_sense();
		break;
	case MPNI_IO_SEND:
		fetch_command_send(false);
		break;
	case MPNI_IO_PSEND:
		fetch_command_send(true);
		break;
	case MPNI_IO_RECV:
		fetch_command_recv(false);
		break;
	case MPNI_IO_PRECV:
		fetch_command_recv(true);
		break;
	default:
		assert(false);
		break;
	}
}

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::accept_command(void)
{
	if (!fetched_sendreq_flag_ && !fetched_recvreq_flag_) return;
	fetched_sendreq_flag_ = false;
	fetched_recvreq_flag_ = false;
#ifdef MP_NETWORK_INTERFACE_DEBUG
	{
		using namespace std;
		const ios::fmtflags flags = cerr.flags();
		cerr << "mp_network_interface(0x" << hex << node_address() << "): "
			 << "accept_command(): done." << endl
			 << *this << endl;
		cerr.flags(flags);
	}
#endif // MP_NETWORK_INTERFACE_DEBUG
}

template <class P, class A, class D, size_t Dsz, class Ch>
bool mp_network_interface<P, A, D, Dsz, Ch>::selfcheck(void) const
{
	if (fetched_sendreq_flag_ && fetched_recvreq_flag_) return false;
	return true;
}

template <class P, class A, class D, size_t Dsz, class Ch>
bool mp_network_interface<P, A, D, Dsz, Ch>::deadlock(void) const
{
	return false;
}

template <class P, class A, class D, size_t Dsz, class Ch>
mp_network_interface<P, A, D, Dsz, Ch>::mp_network_interface(void)
	:
	inherited(),
	state_(READY),
	bus_if_(),
	mem_buf_(address_type(0),
			 sizeof_data_type * 
			 (fixed_hardware_register_size + default_sendreqbuf_size +
			  default_recvreqbuf_size)),
	sendreq_buf_(default_sendreqbuf_size),
	recvreq_buf_(default_recvreqbuf_size),
	recv_pool_(default_recvpool_size ),
	packet_header_size_(default_packet_header_size),
	max_packet_data_size_(default_max_packet_data_size),
	mem_read_wait_(0),
	mem_write_wait_(0),
	fetched_sendreq_flag_(false),
	fetched_recvreq_flag_(false),
	sendreq_restflit_count_(0),
	receiving_flag_(false),
	recvreq_count_(0),
	recvpool_count_(0),
	ready_to_recvreq_count_(0)
{}

template <class P, class A, class D, size_t Dsz, class Ch>
mp_network_interface<P, A, D, Dsz, Ch>::mp_network_interface
	(const mp_network_interface<P, A, D, Dsz, Ch>& a)
	:
	inherited(a),
	state_(a.state_),
	bus_if_(),
	mem_buf_(a.mem_buf_),
	sendreq_buf_(a.sendreqbuf_.max_size()),
	recvreq_buf_(a.recvreqbuf_.size()),
	recv_pool_(a.recv_pool_.size()),
	packet_header_size_(a.packet_header_size_),
	max_packet_data_size_(a.max_packet_data_size_),
	mem_read_wait_(a.mem_read_wait_),
	mem_write_wait_(a.mem_write_wait_),
	fetched_sendreq_flag_(false),
	fetched_recvreq_flag_(false),
	sendreq_restflit_count_(0),
	receiving_flag_(false),
	recvreq_count_(0),
	recvpool_count_(0),
	ready_to_recvreq_count_(0)
{}

template <class P, class A, class D, size_t Dsz, class Ch>
mp_network_interface<P, A, D, Dsz, Ch>::~mp_network_interface()
{}

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::setup(void)
{
	inherited::setup();
	mem_buf_.resize(hardware_register_size() * sizeof_data_type);
	update_fixed_parameters();
	thisclass::reset();
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void mp_network_interface<P, A, D, Dsz, Ch>::clock_in(void)
{
	assert(bus_if_.is_connected());
	switch (state_) {
	case READY:
		if (!bus_if_.is_request()) {
			if (bus_if_.is_ready()) {
				if (ready_to_recvreq_count_ > 0 && !fetched_recvreq_flag_) {
					// recv ...
				} else if (pending_send_request_count() > 0 &&
						   !fetched_sendreq_flag_) {
					size_t i;
					for (i = 0; i < pending_send_request_count(); i++) {
						send_request_type& req(sendreq_buf_[i]);
						if (!req.send_ready_flag) {
							curr_sendreq_dmatrans_ = &req;
							state_ = SEND_DMA_READ_REQ_OWNER;
							break;
						}
					}
				}
			}
		} else {
			const address_type adr = bus_if_.address();
			if (!mem_buf_.is_valid(adr)) break;
			if (bus_if_.is_single_read_request()) {
				accept_command();
				wait_count_ = memory_read_wait();
				access_address_ = adr;
				state_ = (wait_count_ > 1) ?
							 REG_SINGLE_READ_WAIT : REG_SINGLE_READ_ACK;
			} else if (bus_if_.is_single_write_request()) {
				accept_command();
				wait_count_ = memory_write_wait();
				access_address_ = adr;
				state_ = (wait_count_ > 1) ?
							 REG_SINGLE_WRITE_WAIT : REG_SINGLE_WRITE_ACK;
			} else if (bus_if_.is_multi_read_request()) {
				accept_command();
				wait_count_ = memory_read_wait();
				access_address_ = adr;
				total_packet_count_ = bus_if_.total_packet_count();
				packet_index_ = 0;
				state_ = (wait_count_ > 1) ?
							 REG_MULTI_READ_WAIT : REG_MULTI_READ_ACK;
			} else if (bus_if_.is_multi_write_request()) {
				accept_command();
				wait_count_ = memory_write_wait();
				access_address_ = adr;
				total_packet_count_ = bus_if_.total_packet_count();
				state_ = (wait_count_ > 1) ?
							 REG_MULTI_WRITE_WAIT : REG_MULTI_WRITE_ACK;
			}
		}
		break;
	case REG_SINGLE_READ_ACK:
	case REG_SINGLE_WRITE_ACK:
	case REG_MULTI_READ_ACK:
	case REG_MULTI_WRITE_ACK:
	case SEND_DMA_READ_SEND:
	case SEND_DMA_READ_END:
		assert(false);
		break;
	case REG_SINGLE_READ_TRANS:
	case REG_MULTI_READ_TRANS:
		break;
	case REG_SINGLE_READ_WAIT:
		assert(wait_count_ > 1);
		wait_count_--;
		if (wait_count_ == 1) state_ = REG_SINGLE_READ_ACK;
		break;
	case REG_SINGLE_WRITE_WAIT:
		assert(wait_count_ > 1);
		wait_count_--;
		if (wait_count_ == 1) state_ = REG_SINGLE_WRITE_ACK;
		break;
	case REG_SINGLE_WRITE_TRANS:
		if (bus_if_.is_single_write_data()) {
			data_type dat = bus_if_.data();
			if (access_address_
				== mem_buf_.top() + MPNI_IO_OFFSET_REQUEST_TYPE) {
				fetch_command(int(dat));
			} else {
				mem_buf_.write(access_address_, dat);
			}
			state_ = READY;
		}
		break;
	case REG_MULTI_READ_WAIT:
		assert(wait_count_ > 1);
		wait_count_--;
		if (wait_count_ == 1) state_ = REG_MULTI_READ_ACK;
		break;
	case REG_MULTI_WRITE_WAIT:
		assert(wait_count_ > 1);
		wait_count_--;
		if (wait_count_ == 1) state_ = REG_MULTI_WRITE_ACK;
		break;
	case REG_MULTI_WRITE_TRANS:
		if (bus_if_.is_multi_write_data()) {
			data_type dat = bus_if_.data();
			if (access_address_ ==
				mem_buf_.top() + MPNI_IO_OFFSET_REQUEST_TYPE) {
				fetch_command(int(dat));
			} else {
				mem_buf_.write(access_address_, dat);
			}
			if (++packet_index_ < total_packet_count_) {
				access_address_ += sizeof_data_type;
			} else {
				state_ = READY;
			}
		}
		break;
	case SEND_DMA_READ_REQ_OWNER:
		state_ = (bus_if_.is_owner() ? SEND_DMA_READ_SEND : READY);
		break;
	case SEND_DMA_READ_WAIT:
		if (bus_if_.is_multi_read_ack()) {
			packet_index_ = 0;
			state_ = SEND_DMA_READ_TRANS;
		}
		break;
	case SEND_DMA_READ_TRANS:
		if (bus_if_.is_multi_read_data()) {
			send_request_type& req(*curr_sendreq_dmatrans_);
			const size_t req_id = req.req_id;
			const size_t data_len = req.data_len;
			req.pkt->set_data(packet_index_ + 1, bus_if_.data());
			packet_index_++;
			if ((req.progressive_flag && packet_index_ == 1) ||
				(!req.progressive_flag && packet_index_ == data_len)) {
				req.send_ready_flag = true;
			}
			if (packet_index_ == data_len) {
				state_ = SEND_DMA_READ_END;
			}
		}
		break;
	}
	inherited::clock_in();
	if (pending_send_request_count() > 0 && sendreq_restflit_count_ == 0) {
		send_request_type& req(sendreq_buf_.front());
		if (req.send_ready_flag && is_ready_to_push()) {
			const address_type top = mem_buf_.top();
			mem_buf_.write(
				top + (sendflags_offset() + req.req_id) * sizeof_data_type,
				data_type(0)
			);
			push_send_packet(req.pkt);
			sendreq_restflit_count_ = req.rest_flits;
		}
	}
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void mp_network_interface<P, A, D, Dsz, Ch>::clock_out(void)
{
	assert(bus_if_.is_connected());
	inherited::clock_out();
	if (!receiving_flag_ && is_ready_to_snoop(packet_header_size_) &&
		pending_receive_pool_count() < receive_pool_size()) {
		size_t i;
		for (i = 0; i < recv_pool_.size(); i++) {
			if (recv_pool_[i].pkt == NULL) break;
		}
		assert(i < recv_pool_.size());
		receive_pool_entry_type& entry(recv_pool_[i]);
		packet_type* p = look_received_packet();
		entry.pkt = p;
		entry.recv_end_flag = false;
		receiving_pool_index_ = i;
		receiving_flag_ = true;
		recvpool_count_++;
	} else if (receiving_flag_ && is_ready_to_pop()) {
		receive_pool_entry_type& entry(recv_pool_[receiving_pool_index_]);
		pop_received_packet();
		entry.recv_end_flag = true;
		receiving_flag_ = false;
#ifdef MP_NETWORK_INTERFACE_DEBUG
		{
			using namespace std;
			const ios::fmtflags flags = cerr.flags();
			cerr << "mp_network_interface(0x" << hex << node_address() << "): "
				 << "receive_packet_from_network: done." << endl
				 << *this << endl;
			cerr.flags(flags);
		}
#endif // MP_NETWORK_INTERFACE_DEBUG
	}
	switch (state_) {
	case READY:
	case REG_SINGLE_READ_WAIT:
	case REG_SINGLE_WRITE_WAIT:
	case REG_SINGLE_WRITE_TRANS:
	case REG_MULTI_READ_WAIT:
	case REG_MULTI_WRITE_WAIT:
	case REG_MULTI_WRITE_TRANS:
	case SEND_DMA_READ_WAIT:
	case SEND_DMA_READ_TRANS:
		break;
	case REG_SINGLE_READ_ACK:
		bus_if_.send_single_read_ack();
		state_ = REG_SINGLE_READ_TRANS;
		break;
	case REG_SINGLE_READ_TRANS:
		bus_if_.send_single_read_data(mem_buf_.read(access_address_));
		state_ = READY;
		break;
	case REG_SINGLE_WRITE_ACK:
		bus_if_.send_single_write_ack();
		state_ = REG_SINGLE_WRITE_TRANS;
		break;
	case REG_MULTI_READ_ACK:
		bus_if_.send_multi_read_ack();
		state_ = REG_MULTI_READ_TRANS;
		break;
	case REG_MULTI_READ_TRANS:
		bus_if_.send_multi_read_data(mem_buf_.read(access_address_),
									 total_packet_count_, packet_index_);
		if (++packet_index_ < total_packet_count_) {
			access_address_ += sizeof_data_type;
		} else {
			state_ = READY;
		}
		break;
	case REG_MULTI_WRITE_ACK:
		bus_if_.send_multi_write_ack();
		state_ = REG_MULTI_WRITE_TRANS;
		break;
	case SEND_DMA_READ_REQ_OWNER:
		bus_if_.request_ownership();
		break;
	case SEND_DMA_READ_SEND:
		{
			send_request_type& req(*curr_sendreq_dmatrans_);
			bus_if_.send_multi_read_request(req.data_adr, req.data_len);
			state_ = SEND_DMA_READ_WAIT;
		}
		break;
	case SEND_DMA_READ_END:
		bus_if_.clear();
		bus_if_.release_ownership();
		state_ = READY;
		break;
	}
	if (sendreq_restflit_count_ > 0) {
		if (--sendreq_restflit_count_ == 0) {
			sendreq_buf_.pop();
		}
	}
}

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::reset(void)
{
	inherited::reset();
	state_ = READY;
	wait_count_ = 0;
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void
mp_network_interface<P, A, D, Dsz, Ch>::send_request_type::output
	(std::ostream& os) const
{
	using namespace std;
	if (pkt == NULL) {
		os << "(invalid)";
		return;
	}
	const ios::fmtflags flags = os.flags();
	os << "(reqid:" << dec << req_id
	   << ",pktid:" << dec << pkt_id
	   << ",dst:0x" << hex << pkt->destination()
	   << ",adr:0x" << hex << data_adr
	   << ",len:" << dec << data_len
	   << ";prgrs:" << (progressive_flag ? "yes" : "no")
	   << ",ready:" << (send_ready_flag ? "yes" : "no")
	   << ",rest:" << rest_flits
	   << ";dat:[" << hex;
	{
		const size_t n = pkt->data_size();
		for (size_t i = 0; i < n; i++) {
			os << "0x" << pkt->data(i);
			if (i < n - 1) os << ',';
		}
	}
	os << "],route:" << *pkt << ')';
	os.flags(flags);
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void
mp_network_interface<P, A, D, Dsz, Ch>::receive_request_type::output
	(std::ostream& os) const
{
	if (!used_flag) {
		os << "(invalid)";
		return;
	}
	using namespace std;
	const ios::fmtflags flags = os.flags();
	os << "(pktid:" << dec << pkt_id
	   << ",src:0x" << hex << src
	   << ",adr:0x" << hex << data_adr
	   << ",len:" << dec << data_len
	   << ";prgrs:" << (progressive_flag ? "yes" : "no")
	   << ",anysrc:" << (anysrc_flag ? "yes" : "no")
	   << ",anysize:" << (anysize_flag ? "yes" : "no")
	   << ')';
	os.flags(flags);
}

template <class P, class A, class D, size_t Dsz, class Ch>
inline void
mp_network_interface<P, A, D, Dsz, Ch>::receive_pool_entry_type::output
	(std::ostream& os) const
{
	using namespace std;
	if (pkt == NULL) {
		os << "(invalid)";
		return;
	}
	const ios::fmtflags flags = os.flags();
	// os << "(pktid:" << dec << pkt->...;
	os << "(src:0x" << hex << pkt->source()
	   << ",len:" << dec << pkt->data_size()
	   << ";recv:" << (recv_end_flag ? "done" : "not");
	//if (matched_flag) {
	//	os << ",matched:yes,idx:" << dec << recvreqbuf_index;
	//} else {
	//	os << ",matched:no,idx:-";
	//}
	os << ";dat:[" << hex;
	{
		const size_t n = pkt->data_size();
		for (size_t i = 0; i < n; i++) {
			os << "0x" << pkt->data(i);
			if (i < n - 1) os << ',';
		}
	}
	os << "],route:" << *pkt << ')';
	os.flags(flags);
}

template <class P, class A, class D, size_t Dsz, class Ch>
void mp_network_interface<P, A, D, Dsz, Ch>::output(std::ostream& os) const
{
	size_t i;
	bool first_flag;
	inherited::output(os);
	os << "\nsendreq:  [";
	for (i = 0; i < pending_send_request_count(); i++) {
		sendreq_buf_[i].output(os);
		if (i < pending_send_request_count() - 1) os << ',';
	}
	os << ']';
	os << "\nrecvpool: [";
	{
		bool first_flag = true;
		for (i = 0; i < receive_pool_size(); i++) {
			if (recv_pool_[i].pkt != NULL) {
				if (!first_flag) os << ',';
				os << i << ':';
				recv_pool_[i].output(os);
				first_flag = false;
			}
		}
	}
	os << ']';
	os << "\nrecvreq:  [";
	{
		bool first_flag = true;
		for (i = 0; i < receive_request_buffer_size(); i++) {
			if (recvreq_buf_[i].used_flag) {
				if (!first_flag) os << ',';
				os << i << ':';
				recvreq_buf_[i].output(os);
				first_flag = false;
			}
		}
	}
	os << ']';
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* MP_NETWORK_INTERFACE_H */
