/* -*- C++ -*-
 *
 * <<< memory_control_unit.h >>>
 *
 * --- Memory control unit class 'memory_control_unit'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MEMORY_CONTROL_UNIT_H
#define MEMORY_CONTROL_UNIT_H 1

#include <cstddef>
#include <iostream>
#include <isis/single_port_synchronous_unit.h>
#include <isis/bus_port.h>
#include <isis/mapped_memory.h>

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char>
class memory_control_unit : public single_port_synchronous_unit
{
private:
	typedef memory_control_unit<A, D, Dsz, Ch> thisclass;
	typedef single_port_synchronous_unit inherited;
public:
	typedef A address_type;
	typedef D data_type;
	typedef Ch char_type;
	typedef A size_type;
	typedef bus_port<A, D> port_type;
	typedef mapped_memory<A, D, Dsz, Ch> memory_type;
	typedef typename port_type::transaction_id_type transaction_id_type;
	static const size_t sizeof_data_type = Dsz;
private:
	enum state_type {
		READY,
		SINGLE_READ_GRANT,
		SINGLE_READ_SPLIT_WAIT,
		SINGLE_READ_SPLIT_REQ_OWNER,
		SINGLE_READ_SPLIT_OWNER_WAIT,
		SINGLE_READ_WAIT,
		SINGLE_READ_ACK,
		SINGLE_READ_TRANS,
		SINGLE_WRITE_GRANT,
		SINGLE_WRITE_SPLIT_WAIT,
		SINGLE_WRITE_SPLIT_REQ_OWNER,
		SINGLE_WRITE_SPLIT_OWNER_WAIT,
		SINGLE_WRITE_WAIT,
		SINGLE_WRITE_ACK,
		SINGLE_WRITE_TRANS,
		MULTI_READ_GRANT,
		MULTI_READ_SPLIT_WAIT,
		MULTI_READ_SPLIT_REQ_OWNER,
		MULTI_READ_SPLIT_OWNER_WAIT,
		MULTI_READ_WAIT,
		MULTI_READ_ACK,
		MULTI_READ_TRANS,
		MULTI_WRITE_GRANT,
		MULTI_WRITE_SPLIT_WAIT,
		MULTI_WRITE_SPLIT_REQ_OWNER,
		MULTI_WRITE_SPLIT_OWNER_WAIT,
		MULTI_WRITE_WAIT,
		MULTI_WRITE_ACK,
		MULTI_WRITE_TRANS,
		SINGLE_READ_REJECT,
		SINGLE_WRITE_REJECT,
		MULTI_READ_REJECT,
		MULTI_WRITE_REJECT
	};
	state_type state;
	state_type back_state; // for nack within split-wait
	port_type bus_if;
	memory_type* mem_buf;
	unsigned int read_wait_, write_wait_, wait_count;
	size_t total_packet_count, packet_number;
	address_type base_address, address;
	bool split_trans_flag;
	transaction_id_type trans_id;
protected:
	virtual data_type read(address_type a) { return mem_buf->read(a); }
	virtual void write(address_type a, data_type b) { mem_buf->write(a, b); }
public:
	memory_control_unit(void);
	memory_control_unit(const thisclass&);
	virtual void output(std::ostream&) const;
	inline virtual void clock_in(void);
	inline virtual void clock_out(void);
	virtual void reset(void);
	virtual const port& port_ref(void) const { return bus_if; }
	virtual port& port_ref(void) { return bus_if; }
	unsigned int read_wait(void) const { return read_wait_; }
	unsigned int write_wait(void) const { return write_wait_; }
	void set_read_wait(unsigned int);
	void set_write_wait(unsigned int);
	bool is_connected_to_memory(void) const { return mem_buf != NULL; }
	void connect_memory(memory_type& a) { mem_buf = &a; }
	void disconnect_memory(void) { mem_buf = NULL; }
	void enable_split_transaction(void) { split_trans_flag = true; }
	void disable_split_transaction(void) { split_trans_flag = false; }
};

template <class A, class D, size_t Dsz, class Ch>
memory_control_unit<A, D, Dsz, Ch>::memory_control_unit(void)
	: state(READY),
	  bus_if(),
	  mem_buf(NULL),
	  read_wait_(0),
	  write_wait_(0),
	  split_trans_flag(false)
{}

template <class A, class D, size_t Dsz, class Ch>
memory_control_unit<A, D, Dsz, Ch>::memory_control_unit
	(const memory_control_unit<A, D, Dsz, Ch>& a)
	: state(READY),
	  bus_if(),
	  mem_buf(NULL),
	  read_wait_(a.read_wait_),
	  write_wait_(a.write_wait_),
	  split_trans_flag(a.split_trans_flag)
{}

template <class A, class D, size_t Dsz, class Ch>
inline void memory_control_unit<A, D, Dsz, Ch>::clock_in(void)
{
	if (!is_connected_to_memory() || !bus_if.is_connected()) return;
	if (wait_count > 0) wait_count--;
	switch (state) {
	case SINGLE_READ_SPLIT_WAIT:
	case SINGLE_READ_SPLIT_OWNER_WAIT:
	case SINGLE_READ_SPLIT_REQ_OWNER:
	case SINGLE_WRITE_SPLIT_WAIT:
	case SINGLE_WRITE_SPLIT_OWNER_WAIT:
	case SINGLE_WRITE_SPLIT_REQ_OWNER:
	case MULTI_READ_SPLIT_WAIT:
	case MULTI_READ_SPLIT_OWNER_WAIT:
	case MULTI_READ_SPLIT_REQ_OWNER:
	case MULTI_WRITE_SPLIT_WAIT:
	case MULTI_WRITE_SPLIT_OWNER_WAIT:
	case MULTI_WRITE_SPLIT_REQ_OWNER:
		if (!bus_if.is_ready() && trans_id != bus_if.transaction_id() &&
			bus_if.is_owned() && !bus_if.is_owner() &&
			mem_buf->is_valid(bus_if.address())) {
			if (bus_if.is_single_read_request()) {
				back_state = state;
				state = SINGLE_READ_REJECT;
				return;
			} else if (bus_if.is_single_write_request()) {
				back_state = state;
				state = SINGLE_WRITE_REJECT;
				return;
			} else if (bus_if.is_multi_read_request()) {
				back_state = state;
				state = MULTI_READ_REJECT;
				return;
			} else if (bus_if.is_multi_write_request()) {
				back_state = state;
				state = MULTI_WRITE_REJECT;
				return;
			}
		}
		break;
	default:
		break;
	}
	switch (state) {
	case READY:
		if (bus_if.is_ready()) break;
		if (mem_buf->is_valid(bus_if.address())) {
			address = bus_if.address();
			if (bus_if.is_single_read_request()) {
				wait_count = read_wait();
				if (split_trans_flag) {
					trans_id = bus_if.transaction_id();
					state = SINGLE_READ_GRANT;
				} else {
					state = (wait_count > 1) ?
						SINGLE_READ_WAIT : SINGLE_READ_ACK;
				}
			} else if (bus_if.is_single_write_request()) {
				wait_count = write_wait();
				if (split_trans_flag) {
					trans_id = bus_if.transaction_id();
					state = SINGLE_WRITE_GRANT;
				} else {
					state = (wait_count > 1) ?
						SINGLE_WRITE_WAIT : SINGLE_WRITE_ACK;
				}
			} else if (bus_if.is_multi_read_request()) {
				base_address = address;
				total_packet_count = bus_if.total_packet_count();
				packet_number = 0;
				wait_count = read_wait();
				if (split_trans_flag) {
					trans_id = bus_if.transaction_id();
					state = MULTI_READ_GRANT;
				} else {
					state = (wait_count > 1) ?
						MULTI_READ_WAIT : MULTI_READ_ACK;
				}
			} else if (bus_if.is_multi_write_request()) {
				base_address = address;
				total_packet_count = bus_if.total_packet_count();
				packet_number = 0;
				wait_count = write_wait();
				if (split_trans_flag) {
					trans_id = bus_if.transaction_id();
					state = MULTI_WRITE_GRANT;
				} else {
					state = (wait_count > 1) ?
						MULTI_WRITE_WAIT : MULTI_WRITE_ACK;
				}
			}
		}
		break;
	case SINGLE_READ_SPLIT_WAIT:
		if (wait_count <= 2) {
			state = (bus_if.is_owned()) ?
				SINGLE_READ_SPLIT_OWNER_WAIT : SINGLE_READ_SPLIT_REQ_OWNER;
		}
		break;
	case SINGLE_READ_SPLIT_OWNER_WAIT:
		if (!bus_if.is_owned()) state = SINGLE_READ_SPLIT_REQ_OWNER;
		break;
	case SINGLE_READ_SPLIT_REQ_OWNER:
		if (bus_if.is_owner()) state = SINGLE_READ_ACK;
		break;
	case SINGLE_READ_WAIT:
		if (wait_count <= 1) state = SINGLE_READ_ACK;
		break;
	case SINGLE_WRITE_SPLIT_WAIT:
		if (wait_count <= 2) {
			state = (bus_if.is_owned()) ?
				SINGLE_WRITE_SPLIT_OWNER_WAIT : SINGLE_WRITE_SPLIT_REQ_OWNER;
		}
		break;
	case SINGLE_WRITE_SPLIT_OWNER_WAIT:
		if (!bus_if.is_owned()) state = SINGLE_WRITE_SPLIT_REQ_OWNER;
		break;
	case SINGLE_WRITE_SPLIT_REQ_OWNER:
		if (bus_if.is_owner()) state = SINGLE_WRITE_ACK;
		break;
	case SINGLE_WRITE_WAIT:
		if (wait_count <= 1) state = SINGLE_WRITE_ACK;
		break;
	case SINGLE_WRITE_TRANS:
		if (bus_if.is_single_write_data()) {
			write(address, bus_if.data());
			state = READY;
		}
		break;
	case MULTI_READ_SPLIT_WAIT:
		if (wait_count <= 2) {
			state = (bus_if.is_owned()) ?
				MULTI_READ_SPLIT_OWNER_WAIT : MULTI_READ_SPLIT_REQ_OWNER;
		}
		break;
	case MULTI_READ_SPLIT_OWNER_WAIT:
		if (!bus_if.is_owned()) state = MULTI_READ_SPLIT_REQ_OWNER;
		break;
	case MULTI_READ_SPLIT_REQ_OWNER:
		if (bus_if.is_owner()) state = MULTI_READ_ACK;
		break;
	case MULTI_READ_WAIT:
		if (wait_count <= 1) state = MULTI_READ_ACK;
		break;
	case MULTI_WRITE_SPLIT_WAIT:
		if (wait_count <= 2) {
			state = (bus_if.is_owned()) ?
				MULTI_WRITE_SPLIT_OWNER_WAIT : MULTI_WRITE_SPLIT_REQ_OWNER;
		}
		break;
	case MULTI_WRITE_SPLIT_OWNER_WAIT:
		if (!bus_if.is_owned()) state = MULTI_WRITE_SPLIT_REQ_OWNER;
		break;
	case MULTI_WRITE_SPLIT_REQ_OWNER:
		if (bus_if.is_owner()) state = MULTI_WRITE_ACK;
		break;
	case MULTI_WRITE_WAIT:
		if (wait_count <= 1) state = MULTI_WRITE_ACK;
		break;
	case MULTI_WRITE_TRANS:
		if (bus_if.is_multi_write_data()) {
			write(address, bus_if.data());
			if (++packet_number < total_packet_count) {
				address += sizeof_data_type;
			} else {
				state = READY;
			}
		}
		break;
	default:
		break;
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void memory_control_unit<A, D, Dsz, Ch>::clock_out(void)
{
	if (!is_connected_to_memory() || !bus_if.is_connected() || state == READY)
		return;
	switch (state) {
	case SINGLE_READ_GRANT:
		bus_if.send_single_read_grant();
		state = SINGLE_READ_SPLIT_WAIT;
		break;
	case SINGLE_READ_SPLIT_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case SINGLE_READ_ACK:
		if (split_trans_flag) bus_if.set_transaction_id(trans_id);
		bus_if.send_single_read_ack();
		state = SINGLE_READ_TRANS;
		break;
	case SINGLE_READ_TRANS:
		bus_if.send_single_read_data(read(address));
		state = READY;
		break;
	case SINGLE_WRITE_GRANT:
		bus_if.send_single_write_grant();
		state = SINGLE_WRITE_SPLIT_WAIT;
		break;
	case SINGLE_WRITE_SPLIT_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case SINGLE_WRITE_ACK:
		if (split_trans_flag) bus_if.set_transaction_id(trans_id);
		bus_if.send_single_write_ack();
		state = SINGLE_WRITE_TRANS;
		break;
	case MULTI_READ_GRANT:
		bus_if.send_multi_read_grant();
		state = MULTI_READ_SPLIT_WAIT;
		break;
	case MULTI_READ_SPLIT_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case MULTI_READ_ACK:
		if (split_trans_flag) bus_if.set_transaction_id(trans_id);
		bus_if.send_multi_read_ack();
		state = MULTI_READ_TRANS;
		break;
	case MULTI_READ_TRANS:
		bus_if.send_multi_read_data(read(address), total_packet_count,
			packet_number);
		if (++packet_number < total_packet_count) {
			address += sizeof_data_type;
		} else {
			state = READY;
		}
		break;
	case MULTI_WRITE_GRANT:
		bus_if.send_multi_write_grant();
		state = MULTI_WRITE_SPLIT_WAIT;
		break;
	case MULTI_WRITE_SPLIT_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case MULTI_WRITE_ACK:
		if (split_trans_flag) bus_if.set_transaction_id(trans_id);
		bus_if.send_multi_write_ack();
		state = MULTI_WRITE_TRANS;
		break;
	case SINGLE_READ_REJECT:
		bus_if.send_single_read_nack();
		state = back_state;
		break;
	case SINGLE_WRITE_REJECT:
		bus_if.send_single_write_nack();
		state = back_state;
		break;
	case MULTI_READ_REJECT:
		bus_if.send_multi_read_nack();
		state = back_state;
		break;
	case MULTI_WRITE_REJECT:
		bus_if.send_multi_write_nack();
		state = back_state;
		break;
	default:
		break;
	}
}

template <class A, class D, size_t Dsz, class Ch>
void memory_control_unit<A, D, Dsz, Ch>::reset(void)
{
	state = READY;
}

template <class A, class D, size_t Dsz, class Ch>
void memory_control_unit<A, D, Dsz, Ch>::set_read_wait(unsigned int a)
{
	read_wait_ = a;
}

template <class A, class D, size_t Dsz, class Ch>
void memory_control_unit<A, D, Dsz, Ch>::set_write_wait(unsigned int a)
{
	write_wait_ = a;
}

template <class A, class D, size_t Dsz, class Ch>
void memory_control_unit<A, D, Dsz, Ch>::output(std::ostream& os) const
{
	using namespace std;
	const ios::fmtflags flags = os.flags();
	os << hex;
	switch (state) {
	case READY:
		os << "ready";
		break;
	case SINGLE_READ_GRANT:
		os << "single_read_grant(0x" << address << ')';
		break;
	case SINGLE_READ_SPLIT_WAIT:
		os << "single_read_split_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case SINGLE_READ_SPLIT_REQ_OWNER:
		os << "single_read_split_request_ownership(0x" << address << ')';
		break;
	case SINGLE_READ_SPLIT_OWNER_WAIT:
		os << "single_read_split_ownership_wait(0x" << address << ')';
		break;
	case SINGLE_READ_WAIT:
		os << "single_read_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case SINGLE_READ_ACK:
		os << "single_read_ack(0x" << address << ')';
		break;
	case SINGLE_READ_TRANS:
		os << "single_read_trans(0x" << address << ", 0x"
		   << mem_buf->read(address) << ')';
		break;
	case SINGLE_WRITE_GRANT:
		os << "single_write_grant(0x" << address << ')';
		break;
	case SINGLE_WRITE_SPLIT_WAIT:
		os << "single_write_split_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case SINGLE_WRITE_SPLIT_REQ_OWNER:
		os << "single_write_split_request_ownership(0x" << address << ')';
		break;
	case SINGLE_WRITE_SPLIT_OWNER_WAIT:
		os << "single_write_split_ownership_wait(0x" << address << ')';
		break;
	case SINGLE_WRITE_WAIT:
		os << "single_write_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case SINGLE_WRITE_ACK:
		os << "single_write_ack(0x" << address << ')';
		break;
	case SINGLE_WRITE_TRANS:
		os << "single_write_trans(0x" << address << ')';
		break;
	case MULTI_READ_GRANT:
		os << "multi_read_grant(0x" << address << ')';
		break;
	case MULTI_READ_SPLIT_WAIT:
		os << "multi_read_split_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case MULTI_READ_SPLIT_REQ_OWNER:
		os << "multi_read_split_request_ownership(0x" << address << ')';
		break;
	case MULTI_READ_SPLIT_OWNER_WAIT:
		os << "multi_read_split_ownership_wait(0x" << address << ')';
		break;
	case MULTI_READ_WAIT:
		os << "multi_read_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case MULTI_READ_ACK:
		os << "multi_read_ack(0x" << address
		   << ", *" << total_packet_count << ')';
		break;
	case MULTI_READ_TRANS:
		os << "multi_read_trans(0x" << base_address << ", 0x"
		   << mem_buf->read(address) << ", "
		   << packet_number << '/' << total_packet_count << ')';
		break;
	case MULTI_WRITE_GRANT:
		os << "multi_write_grant(0x" << address << ')';
		break;
	case MULTI_WRITE_SPLIT_WAIT:
		os << "multi_write_split_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case MULTI_WRITE_SPLIT_REQ_OWNER:
		os << "multi_write_split_request_ownership(0x" << address << ')';
		break;
	case MULTI_WRITE_SPLIT_OWNER_WAIT:
		os << "multi_write_split_ownership_wait(0x" << address << ')';
		break;
	case MULTI_WRITE_WAIT:
		os << "multi_write_wait(0x" << address << ", "
		   << wait_count << ')';
		break;
	case MULTI_WRITE_ACK:
		os << "multi_write_ack(0x" << address
		   << ", *" << total_packet_count << ')';
		break;
	case MULTI_WRITE_TRANS:
		os << "multi_write_trans(0x" << base_address
		   << ", " << packet_number << '/' << total_packet_count << ')';
		break;
	case SINGLE_READ_REJECT:
		os << "single_read_reject";
		break;
	case SINGLE_WRITE_REJECT:
		os << "single_write_reject";
		break;
	case MULTI_READ_REJECT:
		os << "multi_read_reject";
		break;
	case MULTI_WRITE_REJECT:
		os << "multi_write_reject";
		break;
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* MEMORY_CONTROL_UNIT_H */
