/*
 * <<< dummy_network_interface.h >>>
 *
 * --- Dummy network interface class 'dummy_network_interface'
 *     Copyright (C) 2000-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef DUMMY_NETWORK_INTERFACE_H
#define DUMMY_NETWORK_INTERFACE_H

#include <iostream>
#include <list>
#include <isis/cyclic_queue.h>
#include <isis/network_interface.h>

template <class P>
class dummy_network_interface : public network_interface<P>
{
private:
	typedef dummy_network_interface<P> thisclass;
	typedef network_interface<P> inherited;
public:
	typedef typename inherited::packet_type packet_type;
	typedef typename inherited::node_address_type node_address_type;
	typedef typename inherited::timestamp_type timestamp_type;
private:
	cyclic_queue<packet_type*> inj_que_;
	bool send_flag_;
	inline packet_type* generate_injection_packet
		(node_address_type, node_address_type, size_t);
public:
	dummy_network_interface(void);
	dummy_network_interface(const thisclass&);
	virtual ~dummy_network_interface();
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void output(std::ostream&) const;
	inline bool is_ready(void) const;
	inline void inject_packet(node_address_type, size_t);
	// queue management
	size_t injection_queue_size(void) const { return inj_que_.max_size(); }
	void set_injection_queue_size(size_t a) { inj_que_.reserve(a); }
	size_t injection_queue_length(void) const { return inj_que_.size(); }
};

template <class P>
dummy_network_interface<P>::dummy_network_interface(void)
	: inherited(),
	  inj_que_(0),
	  send_flag_(false)
{}

template <class P>
dummy_network_interface<P>::dummy_network_interface
	(const dummy_network_interface<P>& a)
	: inherited(a),
	  inj_que_(a.inj_que_.max_size()),
	  send_flag_(false)
{}

template <class P>
dummy_network_interface<P>::~dummy_network_interface()
{
	while (inj_que_.size() > 0) {
		delete inj_que_.front();
		inj_que_.pop();
	}
}

template <class P>
inline typename dummy_network_interface<P>::packet_type*
	dummy_network_interface<P>::generate_injection_packet
	(typename dummy_network_interface<P>::node_address_type src,
	 typename dummy_network_interface<P>::node_address_type dst,
	 size_t len)
{
	packet_type* p = new packet_type;
	p->set_source(src);
	p->set_destination(dst);
	p->set_length(len);
	return p;
}

template <class P>
bool dummy_network_interface<P>::is_ready(void) const
{
	return is_ready_to_push() || !inj_que_.full();
}

template <class P>
void dummy_network_interface<P>::inject_packet
	(typename dummy_network_interface<P>::node_address_type dst,
	 size_t len)
{
	packet_type* p = generate_injection_packet(node_address(), dst, len);
	if (is_ready_to_push()) {
		push_send_packet(p);
	} else if (!inj_que_.full()) {
		inj_que_.push(p);
	} else {
		assert(false);
	}
}

template <class P>
void dummy_network_interface<P>::clock_in(void)
{
	send_flag_ = (inj_que_.size() > 0 && is_ready_to_push());
	inherited::clock_in();
}

template <class P>
void dummy_network_interface<P>::clock_out(void)
{
	inherited::clock_out();
	if (send_flag_) {
		push_send_packet(inj_que_.front());
		inj_que_.pop();
	}
	if (is_ready_to_pop()) {
		delete look_received_packet();
		pop_received_packet();
	}
}

template <class P>
void dummy_network_interface<P>::output(std::ostream& os) const
{
	using namespace std;
	inherited::output(os);
	// injection queue
	{
		cyclic_queue<packet_type*> tmp_que(inj_que_);
		list<const packet_type*> tmp;
		while (!tmp_que.empty()) {
			tmp.push_front(tmp_que.front());
			tmp_que.pop();
		}
		typename list<const packet_type*>::const_iterator p = tmp.begin();
		os << " injque:[";
		while (p != tmp.end()) {
			const packet* pkt = *p;
			p++;
			os << *pkt;
			if (p != tmp.end()) os << ',';
		}
		os << ']';
	}
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* DUMMY_NETWORK_INTERFACE_H */
