/*
 * <<< duato_router.h >>>
 *
 * --- Adaptive router class for Duato's routing 'duato_router'
 *     Copyright (C) 2000-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef DUATO_ROUTER_H
#define DUATO_ROUTER_H

#include <vector>
#include <isis/router.h>

template <class P>
class duato_router : public router<P>
{
private:
	typedef duato_router<P> thisclass;
	typedef router<P> inherited;
public:
	typedef typename inherited::packet_type packet_type;
	typedef typename inherited::node_address_type node_address_type;
	typedef typename inherited::channel_controller channel_controller;
protected:
	// routing and arbitration functions
	virtual void adaptive_routing(packet_type&, channel_controller&) = 0;
	virtual void escape_routing(packet_type&, channel_controller&) = 0;
	virtual void arbitrate_output_channel_for_adaptive(void);
	virtual void select_output_virtual_channel(void);
	virtual void arbitrate_output_virtual_channel(void);
	virtual void arbitrate_output_physical_channel(void);
	virtual void transmit_packet(void);
private:
	size_t adapt_ph_master_, alloc_ph_master_, trans_ph_master_;
	std::vector<size_t> adapt_vr_master_, alloc_vr_master_, trans_vr_master_;
public:
	duato_router(void);
	duato_router(const thisclass&);
	virtual ~duato_router();
	// state transition functions
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void setup(void);
	virtual void reset(void);
};

template <class P>
duato_router<P>::duato_router(void)
	: inherited(),
	  adapt_ph_master_(0),
	  alloc_ph_master_(0),
	  trans_ph_master_(0),
	  adapt_vr_master_(0),
	  alloc_vr_master_(0),
	  trans_vr_master_(0)
{}

template <class P>
duato_router<P>::duato_router
	(const duato_router<P>& a)
	: inherited(a),
	  adapt_ph_master_(a.adapt_ph_master_),
	  alloc_ph_master_(a.alloc_ph_master_),
	  trans_ph_master_(a.trans_ph_master_),
	  adapt_vr_master_(a.adapt_vr_master_),
	  alloc_vr_master_(a.alloc_vr_master_),
	  trans_vr_master_(a.trans_vr_master_)
{}

template <class P>
duato_router<P>::~duato_router()
{}

template <class P>
void duato_router<P>::clock_in(void)
{
	arbitrate_output_channel_for_adaptive();
	select_output_virtual_channel();
	arbitrate_output_virtual_channel();
	arbitrate_output_physical_channel();
}

template <class P>
void duato_router<P>::clock_out(void)
{
	transmit_packet();
}

template <class P>
void duato_router<P>::arbitrate_output_channel_for_adaptive(void)
{
	const size_t physical_channel_size = input_size();
	size_t next_ph_i = physical_channel_size;
	for (size_t i = 0; i < physical_channel_size; i++) {
		size_t pid = (adapt_ph_master_ + i) % physical_channel_size;
		const size_t vr_size = input_channel(pid).channel_size();
		size_t next_vr_j = vr_size;
		for (size_t j = 0; j < vr_size; j++) {
			size_t vid = (adapt_vr_master_[pid] + j) % vr_size;
			channel_controller& ctl(internal_controller(pid, vid));
			if (ctl.is_ready() && !input_channel(pid).empty(vid)) {
				packet_type* p = (packet_type*)(input_channel(pid).look(vid));
				adaptive_routing(*p, ctl);
				if (ctl.destination_number() > 0) {
					ctl.set_packet_length(p->length());
					ctl.become_busy();
					if (i < next_ph_i) next_ph_i = i;
					if (j < next_vr_j) next_vr_j = j;
				}
			}
		}
		adapt_vr_master_[pid] = (adapt_vr_master_[pid] + next_vr_j) % vr_size;
	}
	adapt_ph_master_ = (adapt_ph_master_ + next_ph_i) % physical_channel_size;
}

template <class P>
void duato_router<P>::select_output_virtual_channel(void)
{
	for (size_t i = 0; i < input_size(); i++) {
		for (size_t j = 0; j < input_channel(i).channel_size(); j++) {
			channel_controller& ctl(internal_controller(i, j));
			if (ctl.is_ready() && !input_channel(i).empty(j)) {
				packet_type* p = (packet_type*)(input_channel(i).look(j));
				escape_routing(*p, ctl);
				ctl.set_packet_length(p->length());
				ctl.become_wait();
			}
		}
	}
}

template <class P>
void duato_router<P>::arbitrate_output_virtual_channel(void)
{
	const size_t physical_channel_size = input_size();
	size_t next_ph_i = physical_channel_size;
	for (size_t i = 0; i < physical_channel_size; i++) {
		size_t pid = (alloc_ph_master_ + i) % physical_channel_size;
		const size_t vr_size = input_channel(pid).channel_size();
		size_t next_vr_j = vr_size;
		for (size_t j = 0; j < vr_size; j++) {
			size_t vid = (alloc_vr_master_[pid] + j) % vr_size;
			channel_controller& ctl(internal_controller(pid, vid));
			if (ctl.is_wait()) {
				size_t dst_phch = ctl.destination().first;
				size_t dst_vrch = ctl.destination().second;
				if (!is_locked_output_virtual_channel(dst_phch, dst_vrch)) {
					lock_output_virtual_channel(dst_phch, dst_vrch);
					ctl.become_busy();
					if (i < next_ph_i) next_ph_i = i;
					if (j < next_vr_j) next_vr_j = j;
				}
			}
		}
		alloc_vr_master_[pid] = (alloc_vr_master_[pid] + next_vr_j) % vr_size;
	}
	alloc_ph_master_ = (alloc_ph_master_ + next_ph_i) % physical_channel_size;
}

template <class P>
void duato_router<P>::arbitrate_output_physical_channel(void)
{
	const size_t physical_channel_size = input_size();
	size_t next_ph_i = physical_channel_size;
	for (size_t i = 0; i < physical_channel_size; i++) {
		size_t pid = (trans_ph_master_ + i) % physical_channel_size;
		const size_t vr_size = input_channel(pid).channel_size();
		size_t next_vr_j = vr_size;
		for (size_t j = 0; j < vr_size; j++) {
			size_t vid = (trans_vr_master_[pid] + j) % vr_size;
			channel_controller& ctl(internal_controller(pid, vid));
			if (ctl.is_busy()) {
				size_t dst_phch = ctl.destination().first;
				size_t dst_vrch = ctl.destination().second;
				if (!input_channel(pid).empty(vid) &&
					!output_channel(dst_phch).full(dst_vrch) &&
					!internal_crossbar().is_connected_output_channel(dst_phch)){
					internal_crossbar().connect_crosspoint(pid, dst_phch);
					ctl.activate();
					if (i < next_ph_i) next_ph_i = i;
					if (j < next_vr_j) next_vr_j = j;
					break;
				}
			}
		}
		trans_vr_master_[pid] = (trans_vr_master_[pid] + next_vr_j) % vr_size;
	}
	trans_ph_master_ = (trans_ph_master_ + next_ph_i) % physical_channel_size;
}

template <class P>
void duato_router<P>::transmit_packet(void)
{
	crossbar& cb(internal_crossbar());
	for (size_t i = 0; i < input_size(); i++) {
		if (cb.is_connected_input_channel(i)) {
			for (size_t j = 0; j < input_channel(i).channel_size(); j++) {
				// i: input physical channel id
				// j: input virtual channel id
				channel_controller& ctl(internal_controller(i, j));
				if (ctl.is_activated()) {
					size_t out_ph = ctl.destination().first;
					size_t out_vr = ctl.destination().second;
					output_channel(out_ph).put(out_vr, input_channel(i).get(j));
					ctl.clock();
					if (ctl.is_ready()) {
						unlock_output_virtual_channel(out_ph, out_vr);
						increment_total_hop_count();
					}
				}
			}
		}
	}
	cb.clear();
}

template <class P>
void duato_router<P>::setup(void)
{
	inherited::setup();
	adapt_vr_master_.resize(input_size());
	alloc_vr_master_.resize(input_size());
	trans_vr_master_.resize(input_size());
	thisclass::reset();
}

template <class P>
void duato_router<P>::reset(void)
{
	inherited::reset();
	adapt_ph_master_ = 0;
	alloc_ph_master_ = 0;
	trans_ph_master_ = 0;
	fill(adapt_vr_master_.begin(), adapt_vr_master_.end(), 0);
	fill(alloc_vr_master_.begin(), alloc_vr_master_.end(), 0);
	fill(trans_vr_master_.begin(), trans_vr_master_.end(), 0);
}

#endif /* DUATO_ROUTER_H */
