/* -*- C++ -*-
 *
 * <<< comm_io_unit.h >>>
 *
 * --- Pseudo communication I/O class 'comm_io_unit'
 *     Copyright (C) 1996-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef COMM_IO_UNIT_H
#define COMM_IO_UNIT_H 1

#include <cstddef>
#include <iostream>
#include <isis/cyclic_queue.h>
#include <isis/mapped_memory.h>
#include <isis/io_protocol.h>
#include <isis/memory_control_unit.h>

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char>
class comm_io_unit : public synchronous_unit
{
private:
	typedef comm_io_unit<A, D, Dsz, Ch> thisclass;
	typedef synchronous_unit inherited;
public:
	typedef A address_type;
	typedef D data_type;
	typedef Ch char_type;
	typedef A size_type;
	static const size_t sizeof_data_type = Dsz;
	typedef mapped_memory<A, D, Dsz, Ch> memory_type;
private:
	static const size_t default_buffer_size = 512;
	memory_type* buf;
	cyclic_queue<char_type> send_queue, recv_queue;
	address_type header_address, packet_size_address, packet_address;
	size_type body_size;
	void send_to_target(void);
	void receive_from_target(void);
public:
	comm_io_unit(void);
	comm_io_unit(const thisclass&);
	virtual ~comm_io_unit();
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void reset(void);
	virtual void output(std::ostream&) const;
	bool can_put(void) const { return !send_queue.full(); }
	bool can_get(void) const { return !recv_queue.empty(); }
	void put(char_type c) { send_queue.push(c); }
	char_type get(void)
		{ char_type t = recv_queue.front(); recv_queue.pop(); return t; }
	bool is_connected_to_memory(void) const { return mem_buf != NULL; }
	void connect_memory(memory_type&);
	void disconnect_memory(void);
	void set_body_size(size_type);
	void setup(void);
};

template <class A, class D, size_t Dsz, class Ch>
comm_io_unit<A, D, Dsz, Ch>::comm_io_unit(void)
	: send_queue(default_buffer_size),
	  recv_queue(default_buffer_size)
{}

template <class A, class D, size_t Dsz, class Ch>
comm_io_unit<A, D, Dsz, Ch>::comm_io_unit(const comm_io_unit<A, D, Dsz, Ch>& a)
	: inherited(a),
	  send_queue(a.send_queue),
	  recv_queue(a.recv_queue)
{}

template <class A, class D, size_t Dsz, class Ch>
comm_io_unit<A, D, Dsz, Ch>::~comm_io_unit()
{}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::output(std::ostream& os) const
{
	os << "send:" << send_queue.size() << " receive:" << recv_queue.size();
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::send_to_target(void)
{
	int count = buf->read(packet_size_address);
	int i;
	for (i = 0; i < count; i++) {
		if (send_queue.empty()) break;
		buf->write_char(packet_address + i, send_queue.front());
		send_queue.pop();
	}
	buf->write(packet_size_address, data_type(i));
	buf->write(header_address, data_type(COMM_IO_ACK));
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::receive_from_target(void)
{
	int count = buf->read(packet_size_address);
	int i;
	for (i = 0; i < count; i++) {
		if (recv_queue.full()) break;
		recv_queue.push(buf->read_char(packet_address + i));
	}
	buf->write(packet_size_address, data_type(i));
	buf->write(header_address, data_type(COMM_IO_ACK));
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::clock_in(void)
{
	if (buf == NULL) return;
	switch (buf->read(header_address)) {
	case COMM_IO_SEND_REQUEST:
		receive_from_target();
		break;
	case COMM_IO_RECEIVE_REQUEST:
		send_to_target();
		break;
	default:
		break;
	}
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::clock_out(void)
{}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::reset(void)
{
	send_queue.clear();
	recv_queue.clear();
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::connect_memory
	(comm_io_unit<A, D, Dsz, Ch>::memory_type& a)
{
	buf = &a;
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::disconnect_memory(void)
{
	buf = NULL;
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::set_body_size
	(comm_io_unit<A, D, Dsz, Ch>::size_type a)
{
	body_size = a;
}

template <class A, class D, size_t Dsz, class Ch>
void comm_io_unit<A, D, Dsz, Ch>::setup(void)
{
	header_address = buf->top() + sizeof_data_type;
	packet_size_address = header_address + sizeof_data_type;
	packet_address = packet_size_address + sizeof_data_type;
	buf->write(buf->top(), body_size);
	buf->write(header_address, COMM_IO_READY);
}

#endif /* COMM_IO_UNIT_H */
