/* -*- C++ -*-
 *
 * <<< bus_error_detector.h >>>
 *
 * --- Bus error detector unit class 'bus_error_detector'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef BUS_ERROR_DETECTOR_H
#define BUS_ERROR_DETECTOR_H 1

#include <iostream>
#include <isis/single_port_synchronous_unit.h>
#include <isis/bus_port_base.h>

template <class A, class D = A>
class bus_error_detector : public single_port_synchronous_unit
{
private:
	typedef bus_error_detector<A, D> thisclass;
	typedef single_port_synchronous_unit inherited;
public:
	typedef bus_port_base<A, D> port_type;
	typedef typename port_type::address_type address_type;
	typedef typename port_type::data_type data_type;
private:
	static const unsigned int default_timeout = 5000;
	port_type bus_if;
	bool req_flag, error_flag, read_flag;
	address_type adr;
	unsigned int timeout, count;
public:
	bus_error_detector(void);
	bus_error_detector(const thisclass&);
	virtual ~bus_error_detector();
	virtual void output(std::ostream&) const;
	virtual void clock_in(void);
	virtual void clock_out(void);
	virtual void reset(void);
	virtual const port& port_ref(void) const { return bus_if; }
	virtual port& port_ref(void) { return bus_if; }
	bool is_bus_error(void) const { return error_flag; }
	address_type bus_error_address(void) const { return adr; }
	void set_timeout(unsigned int);
};

template <class A, class D>
bus_error_detector<A, D>::bus_error_detector(void)
	: req_flag(false),
	  error_flag(false),
	  timeout(default_timeout)
{}

template <class A, class D>
bus_error_detector<A, D>::bus_error_detector(const bus_error_detector<A, D>& a)
	: req_flag(a.req_flag),
	  error_flag(a.error_flag),
	  timeout(a.timeout)
{}

template <class A, class D>
bus_error_detector<A, D>::~bus_error_detector()
{}

template <class A, class D>
void bus_error_detector<A, D>::clock_in(void)
{
	if (!req_flag) {
		if (error_flag) return;
		if (bus_if.is_request()) {
			adr = bus_if.address();
			count = 0;
			req_flag = true;
		}
	} else {
		if (bus_if.is_request() && bus_if.address() == adr) {
			if (++count == timeout) {
				error_flag = true;
				req_flag = false;
				read_flag = bus_if.is_read_request();
			}
		} else {
			req_flag = false;
		}
	}
}

template <class A, class D>
void bus_error_detector<A, D>::clock_out(void)
{}

template <class A, class D>
void bus_error_detector<A, D>::reset(void)
{
	error_flag = req_flag = false;
}

template <class A, class D>
void bus_error_detector<A, D>::set_timeout(unsigned int a)
{
	timeout = a;
}

template <class A, class D>
void bus_error_detector<A, D>::output(std::ostream& os) const
{
	using namespace std;
	if (is_bus_error()) {
		const ios::fmtflags flags = os.flags();
		os << "bus error(0x" << hex << adr << ' '
		   << (read_flag ? 'r' : 'w') << ')';
		os.flags(flags);
	} else {
		os << "no bus error";
	}
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* BUS_ERROR_DETECTOR_H */
