/* -*- C++ -*-
 *
 * <<< limited_counter.h >>>
 *
 * --- Limited counter class 'limited_counter'
 *     Copyright (C) 1998-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef LIMITED_COUNTER_H
#define LIMITED_COUNTER_H 1

#include <cassert>
#include <iostream>

template <typename value_type = unsigned int>
class limited_counter;

template <typename T>
class limited_counter
{
private:
	typedef limited_counter self_type;
public:
	typedef T value_type;
	//
	explicit limited_counter(value_type a = default_max_())
		: max_(a), value_(0) {}
	limited_counter(const self_type& a)
		: max_(a.max()), value_(a.value()) {}
	~limited_counter()
		{}
	self_type& operator=(const self_type& a)
		{ max_ = a.max(), value_ = a.value(); return *this; }
	void output(std::ostream&) const;
	//
	value_type max(void) const
		{ return max_; }
	value_type value(void) const
		{ return value_; }
	bool is_max(void) const
		{ return value_ == max_; }
	void increment()
		{ if (value_ < max_) value_++; }
	void decrement()
		{ if (value_ > 0) value_--; }
	void clear()
		{ value_ = 0; }
	inline void set_max(value_type);
	operator value_type() const
		{ return value_; }
	const self_type& operator++()
		{ increment(); return *this; }
	const self_type operator++(int)
		{ self_type tmp(*this); increment(); return tmp; }
	const self_type& operator--()
		{ decrement(); return *this; }
	const self_type operator--(int)
		{ self_type tmp(*this); decrement(); return tmp; }
	bool check_invariant() const
		{ return value_ >= 0 && value_ <= max_; }
private:
	static value_type default_max_()
		{ return static_cast<value_type>(1); }
	value_type max_, value_;
};

template <typename T>
std::ostream& operator<<(std::ostream&, const limited_counter<T>&);

template <typename T>
inline void limited_counter<T>::set_max(limited_counter<T>::value_type a)
{
	assert(a >= 0);
	max_ = a;
	if (value_ > max_) value_ = max_;
}

template <typename T>
void limited_counter<T>::output(std::ostream& os) const
{
	if (!os) return;
	os << value();
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <typename T>
std::ostream& operator<<(std::ostream& os, const limited_counter<T>& a)
{
	a.output(os);
	return os;
}

#endif // LIMITED_COUNTER_H
