/*
 * <<< gdb_port.cc >>>
 *
 * --- Communication with gdb port class 'gdb_port'
 *     Copyright (C) 1999-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <isis_config.h>

#if USE_SOCKET

#if HAVE_CONFIG_H
# include "config.h"
#endif /* HAVE_CONFIG_H */

#ifndef _GNU_SOURCE /* for Linux */
# define _GNU_SOURCE
#endif /* _GNU_SOURCE */
#ifndef _HPUX_SOURCE /* for HPUX */
# define _HPUX_SOURCE
#endif /* _HPUX_SOURCE */
extern "C" {
#if HAVE_NETDB_H
# include <netdb.h>
#endif /* HAVE_NETDB_H */
#if HAVE_UNISTD_H
# include <unistd.h>
#endif /* HAVE_UNISTD_H */
#if HAVE_NETINET_IN_H
# include <netinet/in.h>
#endif /* HAVE_NETINET_IN_H */
#if HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif /* HAVE_SYS_PARAM_H */
/* for HPUX */
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif /* HAVE_SYS_TYPES_H */
#if HAVE_SYS_SOCKET_H
# include <sys/socket.h>
#endif /* HAVE_SYS_SOCKET_H */
};
#include <cctype>
#include <cstring>
#include <iostream>
#include "gdb_port.h"

const int default_port = 12120;

// --- htons function ---

#if !HAVE_HTONS

#include <climits>

static unsigned short htons(unsigned short a)
{
#if WORDS_BIGENDIAN
	return a;
#else
	unsigned short val = 0;
	for (unsigned int i = 0; i < (unsigned int)(sizeof(a)); i++) {
		val = (val << CHAR_BIT) | (a & (~((~0) << CHAR_BIT)));
		a >>= CHAR_BIT;
	}
	return val;
#endif // WORDS_BIGENDIAN
}

#endif // !HAVE_HTONS

// --- accept function ---

inline static int accept_hook(int a, struct sockaddr* b, int* c)
{
#if ACCEPT_THIRD_ARG_IS_UNSIGNED
	return accept(a, b, (unsigned int*)c);
#else // ACCEPT_THIRD_ARG_IS_UNSIGNED
	return accept(a, b, c);
#endif // ACCEPT_THIRD_ARG_IS_UNSIGNED
}

// --- recvfrom function ---

#if RECVFROM_SIXTH_ARG_IS_UNSIGNED
# define recvfrom_hook(a, b, c, d, e, f) \
	recvfrom((a), (b), (c), (d), (e), (unsigned int*)(f))
#else // RECVFROM_SIXTH_ARG_IS_UNSIGNED
# define recvfrom_hook(a, b, c, d, e, f) \
	recvfrom((a), (b), (c), (d), (e), (f))
#endif // RECVFROM_SIXTH_ARG_IS_UNSIGNED

int gdb_port::char_to_int(char c)
{
	if (isdigit(c)) {
		return int(c - '0');
	} else if (isupper(c)) {
		return int(c - 'A') + 10;
	} else {
		return int(c - 'a') + 10;
	}
}

char gdb_port::int_to_char(int a)
{
	if (a < 10) {
		return char(a) + '0';
	} else {
		return char(a - 10) + 'a';
	}
}

gdb_port::gdb_port(void)
	: udp_flag_(false),
	  port_(default_port),
	  socket_(-1),
	  client_(-1),
	  client_adr_(NULL)
{}

gdb_port::~gdb_port()
{
	shutdown();
}

bool gdb_port::setup(void)
{
	char hostname[64];
	if (gethostname(hostname, sizeof(hostname)) != 0) return false;
	struct hostent* host = gethostbyname(hostname);
	if (host == NULL) return false;
	if (is_tcp_mode()) {
		// make tcp socket
		socket_ = socket(AF_INET, SOCK_STREAM, 0);
	} else {
		// make udp socket
		socket_ = socket(AF_INET, SOCK_DGRAM, 0);
	}
	if (socket_ == -1) return false;
	struct sockaddr_in adr;
	memset(&adr, 0, sizeof(adr));
	adr.sin_family = AF_INET;
	adr.sin_port = htons(port_);
	memcpy(&adr.sin_addr, host->h_addr, host->h_length);
	if (bind(socket_, (const sockaddr*)(&adr), sizeof(adr)) < 0) {
		disconnect();
		return false;
	}
	if (is_tcp_mode()) {
		if (listen(socket_, 1) < 0) {
			shutdown();
			return false;
		}
	}
	return true;
}

void gdb_port::shutdown(void)
{
	if (is_tcp_mode()) {
		disconnect();
	} else {
		if (client_adr_ != NULL) delete[] (char*)(client_adr_);
	}
	if (socket_ >= 0) {
		close(socket_);
		socket_ = -1;
	}
}

void gdb_port::connect(void)
{
	if (is_udp_mode()) return;
	struct sockaddr dummy1;
	int dummy2;
	client_ = accept_hook(socket_, &dummy1, &dummy2);
	if (client_ < 0) disconnect();
}

void gdb_port::disconnect(void)
{
	if (is_udp_mode()) return;
	if (socket_ >= 0 && client_ >= 0) {
		close(client_);
		client_ = -1;
	}
}

int gdb_port::getchar_tcp(void)
{
	if (!is_ready()) return EOF;
	char c;
	while (1) {
		int val = read(client_, &c, 1);
		if (val < 0) {
			close(client_);
			client_ = -1;
			return EOF;
		}
		if (val > 0) break;
	}
	return c;
}

int gdb_port::send_tcp(const char* mes, int len)
{
	const int max_retry = 5;
	for (int i = 0; i < max_retry; i++) {
		if (write(client_, mes, len) < len) return -1;
		while (1) {
			int ack = getchar_tcp();
			if (ack == EOF) return -1;
			if (ack == '+') return len;
			if (ack == '-') break;
		}
	}
	return -1;
}

int gdb_port::send_udp(const char* mes, int len)
{
	if (client_adr_ == NULL) return -1;
	const int max_retry = 5;
	for (int i = 0; i < max_retry; i++) {
		if (sendto(socket_, mes, len, 0, (const struct sockaddr*)(client_adr_),
				   client_adr_len_) < len) {
			return -1;
		}
		while (1) {
			struct sockaddr adr;
			int adr_len = sizeof(adr);
			char ack;
			int val = recvfrom_hook(socket_, &ack, 1, 0, &adr, &adr_len);
			if (val < 0) return -1;
			if (val == 0) continue;
			// if (adr_len != client_adr_len_ ||
			// 	memcmp(&adr, client_adr_, adr_len) != 0) continue;
			// not implemented in pot...
			if (ack == '+') return len;
			if (ack == '-') break;
		}
	}
	return -1;
}

int gdb_port::send(const char* mes)
{
	if (!is_ready()) return -1;
	unsigned int len = strlen(mes);
	char buf[4096];
	if (len > int(sizeof(buf)) - 5) return -1; // 5: '$' + '#' + sum[0,1]
	char* p = buf;
	*(p++) = '$';
	int sum = 0;
	for (unsigned int i = 0; i < len; i++) {
		char c = mes[i];
		if (c == '\\' || c == '$' || c == '#') *(p++) = '\\';
		*(p++) = c;
		sum = (sum + c) & 0xff;
	}
	*(p++) = '#';
	*(p++) = int_to_char(sum >> 4);
	*(p++) = int_to_char(sum & 0xf);
	return (is_tcp_mode()) ?  send_tcp(buf, p - buf) : send_udp(buf, p - buf);
}

int gdb_port::receive_tcp(char* mes, int len)
{
	// skip head
	while (1) {
		int c = getchar_tcp();
		if (c == EOF) return -1;
		if (c == '$') break;
	}
	// get body
	int count = 0, sum = 0;
	while (1) {
		int c = getchar_tcp();
		if (c == EOF) return -1;
		if (c == '#') break;
		if (c == '\\') {
			int cc = getchar_tcp();
			if (cc == EOF) return -1;
			c = cc;
		}
		*(mes++) = c;
		sum = (sum + c) & 0xff;
		count++;
		if (count >= len - 1) return 0;
	}
	*mes = '\0';
	// get checksum
	int upper = getchar_tcp();
	if (upper == EOF) return -1;
	int lower = getchar_tcp();
	if (lower == EOF) return -1;
	if (!isxdigit(upper) || !isxdigit(lower)) return 0;
	// calc checksum, and check value
	if (sum != ((char_to_int(upper) << 4) | char_to_int(lower))) {
		return 0;
	}
	if (write(client_, "+", 1) < 1) return 0;
	return count;
}

int gdb_port::receive_udp(char* mes, int len)
{
	char buf[4096];
	// get udp packet
	if (client_adr_ != NULL) delete[] (char*)(client_adr_);
	client_adr_ = (void*)(new char[sizeof(struct sockaddr)]);
	client_adr_len_ = sizeof(struct sockaddr);
	int val = recvfrom_hook(socket_, buf, sizeof(buf), 0,
							(struct sockaddr*)(client_adr_), &client_adr_len_);
	if (val < 0) return -1;
	if (val < 5) return 0;
	char* p = buf;
	// skip head
	{
		char bak = buf[val - 1];
		buf[val - 1] = '$';
		while (*(p++) != '$');
		buf[val - 1] = bak;
		if (val - (p - buf) <= 0) return 0;
	}
	char* d = mes;
	// get body
	int count = 0, sum = 0;
	while (1) {
		char c = *(p++);
		if (c == '#') break;
		if (val - (p - buf) < 2) return 0;
		if (c == '\\') c = *(p++);
		*(d++) = c;
		if (d - mes > len - 1) return 0;
		sum = (sum + c) & 0xff;
		count++;
	}
	*d = '\0';
	// get checksum
	if (val - (p - buf) < 2) return 0;
	char upper = *(p++);
	char lower = *(p++);
	if (!isxdigit(upper) || !isxdigit(lower)) return 0;
	// calc checksum, and check value
	if (sum != ((char_to_int(upper) << 4) | char_to_int(lower))) {
		return 0;
	}
	sendto(socket_, "+", 1, 0, (const struct sockaddr*)(client_adr_),
		   client_adr_len_);
	return count;
}

int gdb_port::receive(char* mes, int len)
{
	if (!is_ready()) return -1;
	return (is_tcp_mode()) ? receive_tcp(mes, len) : receive_udp(mes, len);
}

#else // !USE_SOCKET

#include <iostream>
#include "gdb_port.h"

using namespace std;

bool gdb_port::setup(void)
{
	cerr << "Sorry, you cannot use remote gdb support." << endl
		 << "You must re-install ISIS with configure option '--enable-socket'."
		 << endl;
	return false;
}

#endif // USE_SOCKET
