/* -*- C++ -*-
 *
 * <<< c_array.h >>>
 *
 * --- Fixed size array class 'c_array'
 *     Copyright (C) 1999-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef C_ARRAY_H
#define C_ARRAY_H 1

#include <cassert>
#include <cstddef>

template <class T, size_t n>
class c_array
{
private:
	typedef c_array<T, n> thisclass;
public:
	typedef T value_type;
	typedef size_t size_type;
	typedef ptrdiff_t difference_type;
	typedef T* iterator;
	typedef const T* const_iterator;
	typedef T* pointer;
	typedef const T* const_pointer;
	typedef T& reference;
	typedef const T& const_reference;
protected:
	value_type c[n];
public:
	c_array(void) {}
	inline explicit c_array(size_type, const_reference);
	inline c_array(const thisclass&);
	template <class In>
		inline c_array(In, In);
	inline void assign(size_type, const_reference);
	operator T*() { return c; }
	const_reference at(size_type i) const { assert(i < n); return c[i]; }
	reference at(size_type i) { assert(i < n); return c[i]; }
	inline const_reference operator[](size_type) const;
	inline reference operator[](size_type);
	const_iterator begin(void) const { return c; }
	iterator begin(void) { return c; }
	const_iterator end(void) const { return c + n; }
	iterator end(void) { return c + n; }
	const_reference front(void) const { return c[0]; }
	reference front(void) { return c[0]; }
	inline const_reference back(void) const { return c[n - 1]; }
	inline reference back(void) { return c[n - 1]; }
	size_type size(void) const { return n; }
	bool empty(void) const { return n == 0; }
	size_type max_size(void) const { return n; }
	size_type capacity(void) const { return n; }
	inline void swap(const thisclass&);
	inline bool equiv(const thisclass&) const;
	inline bool cmp(const thisclass&) const;
	inline bool operator==(const thisclass&) const;
	inline bool operator!=(const thisclass&) const;
	inline bool operator<(const thisclass&) const;
	// dummy functions: for vector compatible
	explicit c_array(size_type) {}
	void resize(size_type) const {}
	void reserve(size_type) const {}
};

template <class T, size_t n>
inline typename c_array<T, n>::const_reference
c_array<T, n>::operator[](c_array<T, n>::size_type i) const
{
#ifdef DEBUG
	return at(i);
#else // DEBUG
	return c[i];
#endif // DEBUG
}

template <class T, size_t n>
inline typename c_array<T, n>::reference
c_array<T, n>::operator[](c_array<T, n>::size_type i)
{
#ifdef DEBUG
	return at(i);
#else // DEBUG
	return c[i];
#endif // DEBUG
}

template <class T, size_t n>
inline c_array<T, n>::c_array(c_array<T, n>::size_type,	
							  c_array<T, n>::const_reference val)
{
	for (size_t i = 0; i < n; i++) c[i] = val;
}

template <class T, size_t n>
inline c_array<T, n>::c_array(const c_array<T, n>& a)
{
	for (size_t i = 0; i < n; i++) c[i] = a.c[i];
}

template <class T, size_t n>
template <class In>
inline c_array<T, n>::c_array(In first, In last)
{
	copy(first, last, begin());
}

template <class T, size_t n>
inline void c_array<T, n>::assign(c_array<T, n>::size_type,	
								  c_array<T, n>::const_reference val)
{
	for (size_t i = 0; i < n; i++) c[i] = val;
}

template <class T, size_t n>
inline void c_array<T, n>::swap(const c_array<T, n>& a)
{
	for (size_t i = 0; i < n; i++) {
		value_type tmp = c[i];
		c[i] = a.c[i];
		a.c[i] = tmp;
	}
}

template <class T, size_t n>
inline bool c_array<T, n>::equiv(const c_array<T, n>& a) const
{
	for (size_t i = 0; i < n; i++) {
		if (c[i] != a.c[i]) return false;
	}
	return true;
}

template <class T, size_t n>
inline bool c_array<T, n>::cmp(const c_array<T, n>& a) const
{
	for (size_t i = 0; i < n; i++) {
		if (c[i] < a.c[i]) return true;
		if (a.c[i] < c[i]) return false;
	}
	return false;
}

template <class T, size_t n>
inline bool c_array<T, n>::operator==(const c_array<T, n>& a) const
{
	return equiv(a);
}

template <class T, size_t n>
inline bool c_array<T, n>::operator!=(const c_array<T, n>& a) const
{
	return !equiv(a);
}

template <class T, size_t n>
inline bool c_array<T, n>::operator<(const c_array<T, n>& a) const
{
	return cmp(a);
}

#endif /* C_ARRAY_H */
