/* -*- C++ -*-
 *
 * <<< bitvector.h >>>
 *
 * --- Bitvector class 'bitvector'
 *     Copyright (C) 1997-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef BITVECTOR_H
#define BITVECTOR_H 1

#include <climits> // for CHAR_BIT
#include <iostream>
#include <vector>

template <class C>
class basic_bitvector
{
private:
	typedef basic_bitvector thisclass;
public:
	typedef C container_type;
	typedef bool value_type;
	typedef size_t size_type;
private:
	typedef typename C::value_type container_value_type;
	static const size_t container_value_bit
		= sizeof(container_value_type) * CHAR_BIT;
	static const container_value_type container_value_mask
		= ~container_value_type(0);
	size_t size_;
	container_type buf;
public:
	basic_bitvector(void) : size_(0), buf(0) {}
	explicit basic_bitvector(size_type n, bool x = false)
		: size_(n), buf((n + container_value_bit - 1) / container_value_bit)
	{
		if (x) {
			for (size_t i = 0; i < buf.size(); i++)
				buf[i] = container_value_mask;
		} else {
			for (size_t i = 0; i < buf.size(); i++) buf[i] = 0;
		}
	}
	basic_bitvector(const thisclass& a) : size_(a.size_), buf(a.buf) {}
	~basic_bitvector() {}
	thisclass& operator=(const thisclass& a)
		{ size_ = a.size_; buf = a.buf; return *this; }
	void input(std::istream&);
	void output(std::ostream&) const;
	size_type size(void) const { return size_; }
	void resize(size_type);
	inline bool operator[](size_t) const;
	inline bool at(size_t) const;
	inline bool test(size_t) const;
	inline void set(size_type);
	inline void reset(size_type);
	inline void flip(size_type);
	inline void set(size_type, bool);
	inline void set(void);
	inline void reset(void);
	inline void flip(void);
	inline bool equal(const thisclass&) const;
	inline thisclass operator_not(void) const;
	inline thisclass operator_and(const thisclass&) const;
	inline thisclass operator_or(const thisclass&) const;
	inline thisclass operator_xor(const thisclass&) const;
	inline thisclass& operator&=(const thisclass&);
	inline thisclass& operator|=(const thisclass&);
	inline thisclass& operator^=(const thisclass&);
	// thisclass& operator+=(const thisclass&);
	// thisclass& operator<<=(size_type);
	// thisclass& operator>>=(size_type);
};

template <class C>
inline basic_bitvector<C> operator~(const basic_bitvector<C>&);
template <class C>
inline basic_bitvector<C>
	operator&(const basic_bitvector<C>&, const basic_bitvector<C>&);
template <class C>
inline basic_bitvector<C>
	operator|(const basic_bitvector<C>&, const basic_bitvector<C>&);
template <class C>
inline basic_bitvector<C>
	operator^(const basic_bitvector<C>&, const basic_bitvector<C>&);
template <class C>
inline std::istream& operator>>(std::istream&, basic_bitvector<C>&);
template <class C>
inline std::ostream& operator<<(std::ostream&, const basic_bitvector<C>&);

class bitvector : public basic_bitvector< std::vector<unsigned int> >
{
private:
	typedef bitvector thisclass;
	typedef basic_bitvector< std::vector<unsigned int> > inherited;
public:
	typedef inherited::container_type container_type;
	typedef inherited::value_type value_type;
	typedef inherited::size_type size_type;
	bitvector(void) : inherited() {}
	explicit bitvector(size_type n, bool x = false) : inherited(n, x) {}
	bitvector(const thisclass& a) : inherited(a) {}
};

template <class C>
inline void basic_bitvector<C>::resize(basic_bitvector<C>::size_type n)
{
	if (n < size_) {
		size_ = n;
		buf.resize(n % container_value_bit);
	} else if (n > size_) {
		const size_t old_buf_size = buf.size();
		const size_t new_buf_size
			= (n + container_value_bit - 1) / container_value_bit;
		const size_t masksize = size_ % container_value_bit;
		if (masksize > 0) {
			buf[old_buf_size - 1]
				&= (~((~container_value_type(0)) << masksize));
		}
		if (new_buf_size > old_buf_size) {
			buf.resize(new_buf_size);
			for (size_t i = old_buf_size; i < new_buf_size; i++) {
				buf[i] = container_value_type(0);
			}
		}
	}
}

template <class C>
inline bool basic_bitvector<C>::operator[](basic_bitvector<C>::size_type i)
	const
{
	return (buf[i / container_value_bit]
			& (container_value_type(1) << (i % container_value_bit))) != 0;
}

template <class C>
inline bool basic_bitvector<C>::at(basic_bitvector<C>::size_type i) const
{
	return (buf.at(i / container_value_bit)
			& (container_value_type(1) << (i % container_value_bit))) != 0;
}

template <class C>
inline bool basic_bitvector<C>::test(basic_bitvector<C>::size_type i) const
{
	return operator[](i);
}

template <class C>
inline void basic_bitvector<C>::set(basic_bitvector<C>::size_type i)
{
	buf[i / container_value_bit]
		|= (container_value_type(1) << (i % container_value_bit));
}

template <class C>
inline void basic_bitvector<C>::reset(basic_bitvector<C>::size_type i)
{
	buf[i / container_value_bit]
		&= (~(container_value_type(1) << (i % container_value_bit)));
}

template <class C>
inline void basic_bitvector<C>::set(basic_bitvector<C>::size_type i, bool x)
{
	if (x) {
		set(i);
	} else {
		reset(i);
	}
}

template <class C>
inline void basic_bitvector<C>::flip(basic_bitvector<C>::size_type i)
{
	buf[i / container_value_bit]
		^= (container_value_type(1) << (i % container_value_bit));
}

template <class C>
inline void basic_bitvector<C>::set(void)
{
	for (size_t i = 0; i < buf.size(); i++) buf[i] = container_value_mask;
}

template <class C>
inline void basic_bitvector<C>::reset(void)
{
	for (size_t i = 0; i < buf.size(); i++) buf[i] = 0;
}

template <class C>
inline void basic_bitvector<C>::flip(void)
{
	for (size_t i = 0; i < buf.size(); i++) buf[i] ^= container_value_mask;
}

template <class C>
inline bool basic_bitvector<C>::equal(const basic_bitvector<C>& a) const
{
	if (size_ != a.size_) return false;
	size_t i;
	for (i = 0; i < buf.size() - 1; i++) {
		if (buf[i] != a.buf[i]) return false;
	}
	size_t masksize = size_ % container_value_bit;
	if (masksize == 0) {
		return buf[i] == a.buf[i];
	} else {
		const container_value_type mask
			= (~((~container_value_type(0)) << masksize));
		return ((buf[i] ^ a.buf[i]) & mask) == container_value_type(0);
	}
}

template <class C>
inline basic_bitvector<C> basic_bitvector<C>::operator_not(void) const
{
	thisclass tmp;
	tmp.size_ = size_;
	tmp.buf.resize(buf.size());
	for (size_t i = 0; i < buf.size(); i++) tmp.buf[i] = (~buf[i]);
	return tmp;
}

template <class C>
inline basic_bitvector<C> basic_bitvector<C>::operator_and
	(const basic_bitvector<C>& a) const
{
	basic_bitvector<C> t;
	size_t i;
	if (size_ == a.size_) {
		const size_t bsz = buf.size();
		t.size_ = size_;
		t.buf.resize(bsz);
		for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] & a.buf[i]);
	} else if (size_ > a.size_) {
		const size_t bsz = buf.size();
		const size_t masksize = (size_ % container_value_bit);
		t.size_ = size_;
		t.buf.resize(bsz);
		if (masksize == 0) {
			for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] & a.buf[i]);
		} else {
			const container_value_type mask
				= ((~container_value_type(0)) << masksize);
			for (i = 0; i < bsz - 1; i++) t.buf[i] = (buf[i] & a.buf[i]);
			t.buf[i] = (buf[i] & (a.buf[i] | mask));
		}
	} else {
		const size_t bsz = a.buf.size();
		const size_t masksize = (a.size_ % container_value_bit);
		t.size_ = a.size_;
		t.buf.resize(bsz);
		if (masksize == 0) {
			for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] & a.buf[i]);
		} else {
			const container_value_type mask
				= ((~container_value_type(0)) << masksize);
			for (i = 0; i < bsz - 1; i++) t.buf[i] = (buf[i] & a.buf[i]);
			t.buf[i] = ((buf[i] | mask) & a.buf[i]);
		}
	}
	return t;
}

template <class C>
inline basic_bitvector<C> basic_bitvector<C>::operator_or
	(const basic_bitvector<C>& a) const
{
	basic_bitvector<C> t;
	size_t i;
	if (size_ == a.size_) {
		const size_t bsz = buf.size();
		t.size_ = size_;
		t.buf.resize(bsz);
		for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] | a.buf[i]);
	} else if (size_ > a.size_) {
		const size_t bsz = buf.size();
		const size_t masksize = (size_ % container_value_bit);
		t.size_ = size_;
		t.buf.resize(bsz);
		if (masksize == 0) {
			for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] | a.buf[i]);
		} else {
			const container_value_type mask
				= (~((~container_value_type(0)) << masksize));
			for (i = 0; i < bsz - 1; i++) t.buf[i] = (buf[i] | a.buf[i]);
			t.buf[i] = (buf[i] | (a.buf[i] & mask));
		}
	} else {
		const size_t bsz = a.buf.size();
		const size_t masksize = (a.size_ % container_value_bit);
		t.size_ = a.size_;
		t.buf.resize(bsz);
		if (masksize == 0) {
			for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] | a.buf[i]);
		} else {
			const container_value_type mask
				= (~((~container_value_type(0)) << masksize));
			for (i = 0; i < bsz - 1; i++) t.buf[i] = (buf[i] | a.buf[i]);
			t.buf[i] = ((buf[i] & mask) | a.buf[i]);
		}
	}
	return t;
}

template <class C>
inline basic_bitvector<C> basic_bitvector<C>::operator_xor
	(const basic_bitvector<C>& a) const
{
	basic_bitvector<C> t;
	size_t i;
	if (size_ == a.size_) {
		const size_t bsz = buf.size();
		t.size_ = size_;
		t.buf.resize(bsz);
		for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] ^ a.buf[i]);
	} else if (size_ > a.size_) {
		const size_t bsz = buf.size();
		const size_t masksize = (size_ % container_value_bit);
		t.size_ = size_;
		t.buf.resize(bsz);
		if (masksize == 0) {
			for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] ^ a.buf[i]);
		} else {
			const container_value_type mask
				= (~((~container_value_type(0)) << masksize));
			for (i = 0; i < bsz - 1; i++) t.buf[i] = (buf[i] ^ a.buf[i]);
			t.buf[i] = (buf[i] ^ (a.buf[i] & mask));
		}
	} else {
		const size_t bsz = a.buf.size();
		const size_t masksize = (a.size_ % container_value_bit);
		t.size_ = a.size_;
		t.buf.resize(bsz);
		if (masksize == 0) {
			for (i = 0; i < bsz; i++) t.buf[i] = (buf[i] ^ a.buf[i]);
		} else {
			const container_value_type mask
				= (~((~container_value_type(0)) << masksize));
			for (i = 0; i < bsz - 1; i++) t.buf[i] = (buf[i] ^ a.buf[i]);
			t.buf[i] = ((buf[i] & mask) ^ a.buf[i]);
		}
	}
	return t;
}

template <class C>
inline basic_bitvector<C>&
	basic_bitvector<C>::operator&=(const basic_bitvector<C>& a)
{
	size_t i, masksize;
	if (size_ <= a.size_ || (masksize = (a.size_ % container_value_bit)) == 0) {
		for (i = 0; i < buf.size(); i++) buf[i] &= a.buf[i];
	} else {
		for (i = 0; i < a.buf.size() - 1; i++) buf[i] &= a.buf[i];
		buf[i] &= (a.buf[i] | ((~container_value_type(0)) << masksize));
	}
}

template <class C>
inline basic_bitvector<C>&
	basic_bitvector<C>::operator|=(const basic_bitvector<C>& a)
{
	size_t i, masksize;
	if (size_ <= a.size_ || (masksize = (a.size_ % container_value_bit)) == 0) {
		for (i = 0; i < buf.size(); i++) buf[i] |= a.buf[i];
	} else {
		for (i = 0; i < a.buf.size() - 1; i++) buf[i] |= a.buf[i];
		buf[i] |= (a.buf[i] & (~((~container_value_type(0)) << masksize)));
	}
}

template <class C>
inline basic_bitvector<C>&
	basic_bitvector<C>::operator^=(const basic_bitvector<C>& a)
{
	size_t i, masksize;
	if (size_ <= a.size_ || (masksize = (a.size_ % container_value_bit)) == 0) {
		for (i = 0; i < buf.size(); i++) buf[i] ^= a.buf[i];
	} else {
		for (i = 0; i < a.buf.size() - 1; i++) buf[i] ^= a.buf[i];
		buf[i] ^= (a.buf[i] & (~((~container_value_type(0)) << masksize)));
	}
}

template <class C>
inline void basic_bitvector<C>::input(std::istream& is)
{
}

template <class C>
inline void basic_bitvector<C>::output(std::ostream& os) const
{
	size_t count = size_;
	for (size_t i = 0; i < buf.size(); i++) {
		container_value_type t = buf[i];
		for (size_t j = 0; j < container_value_bit; j++) {
			os << (((t & 1) != 0) ? '1' : '0');
			if (--count == 0) return;
			t >>= 1;
		}
	}
}

template <class C>
inline basic_bitvector<C> operator~(const basic_bitvector<C>& a)
{
	return a.operator_not();
}

template <class C>
inline bool operator==(const basic_bitvector<C>& a, const basic_bitvector<C>& b)
{
	return a.equal(b);
}

template <class C>
inline bool operator!=(const basic_bitvector<C>& a, const basic_bitvector<C>& b)
{
	return !a.equal(b);
}

template <class C>
inline basic_bitvector<C>
	operator&(const basic_bitvector<C>& a, const basic_bitvector<C>& b)
{
	return a.operator_and(b);
}

template <class C>
inline basic_bitvector<C>
	operator|(const basic_bitvector<C>& a, const basic_bitvector<C>& b)
{
	return a.operator_or(b);
}

template <class C>
inline basic_bitvector<C>
	operator^(const basic_bitvector<C>& a, const basic_bitvector<C>& b)
{
	return a.operator_xor(b);
}

template <class C>
inline std::istream& operator>>(std::istream& is, basic_bitvector<C>& a)
{
	if (is) a.input(is);
	return is;
}

template <class C>
inline std::ostream& operator<<(std::ostream& os, const basic_bitvector<C>& a)
{
	if (os) a.output(os);
	return os;
}

#endif /* BITVECTOR_H */
