/* -*- C++ -*-
 *
 * <<< array.h >>>
 *
 * --- Variable size array class 'array'
 *     Copyright (C) 1995-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef ARRAY_H
#define ARRAY_H 1

#include <cassert>
#include <cstddef>
#include <iostream>

template <class T>
class array
{
private:
	typedef array<T> thisclass;
public:
	typedef T value_type;
	typedef size_t size_type;
	typedef ptrdiff_t difference_type;
	typedef T* iterator;
	typedef const T* const_iterator;
	typedef T* pointer;
	typedef const T* const_pointer;
	typedef T& reference;
	typedef const T& const_reference;
protected:
	value_type* c;
	size_type n;
private:
	inline static void assign(const_pointer, pointer, size_type);
public:
	array(void) : n(0) {}
	inline explicit array(size_type i) : c(new value_type[i]), n(i) {}
	inline array(const thisclass&);
	inline ~array();
	thisclass& operator=(const thisclass&);
	operator T*() { return c; }
	const_reference at(size_type i) const { assert(i < n); return c[i]; }
	reference at(size_type i) { assert(i < n); return c[i]; }
	inline const_reference operator[](size_type) const;
	inline reference operator[](size_type);
	const_iterator begin(void) const { return c; }
	iterator begin(void) { return c; }
	const_iterator end(void) const { return c + n; }
	iterator end(void) { return c + n; }
	const_reference front(void) const { return c[0]; }
	reference front(void) { return c[0]; }
	inline const_reference back(void) const { return c[n - 1]; }
	inline reference back(void) { return c[n - 1]; }
	size_type size(void) const { return n; }
	void resize(size_type);
	bool empty(void) const { return n == 0; }
	size_type max_size(void) const { return n; }
	size_type capacity(void) const { return n; }
	inline void swap(const thisclass&);
	inline bool equiv(const thisclass&) const;
	inline bool cmp(const thisclass&) const;
	inline bool operator==(const thisclass&) const;
	inline bool operator!=(const thisclass&) const;
	inline bool operator<(const thisclass&) const;
	// dummy functions: for vector compatible
	void reserve(size_type) const {}
	// old array functions
	bool check_invariant(void) const { return !(n > 0 && c == NULL); }
	void fill(const_reference);
	void input(std::istream&);
	void output(std::ostream&) const;
};

template <class T>
inline void array<T>::assign(array<T>::const_pointer src,
							 array<T>::pointer dst, array<T>::size_type x)
{
	for (size_type i = 0; i < x; i++) dst[i] = src[i];
}

template <class T>
inline typename array<T>::const_reference
array<T>::operator[](array<T>::size_type i)
	const
{
#ifdef DEBUG
	return at(i);
#else // DEBUG
	return c[i];
#endif // DEBUG
}

template <class T>
inline typename array<T>::reference
array<T>::operator[](array<T>::size_type i)
{
#ifdef DEBUG
	return at(i);
#else // DEBUG
	return c[i];
#endif // DEBUG
}

template <class T>
inline array<T>::array(const array<T>& a)
	: c(new value_type[a.n]),
	  n(a.n)
{
	assign(a.c, c, n);
}

template <class T>
inline array<T>::~array()
{
	if (n > 0) delete[] c;
}

template <class T>
array<T>& array<T>::operator=(const array<T>& a)
{
	if (&a != this) {
		if (n > 0) delete[] c;
		n = a.n;
		if (n > 0) {
			c = new value_type[n];
			assign(a.c, c, n);
		}
	}
	return *this;
}

template <class T>
void array<T>::resize(array<T>::size_type x)
{
	if (x == n) return;
	if (x > 0) {
		value_type* tmp = new value_type[x];
		if (n > 0) {
			assign(c, tmp, (x < n) ? x : n);
			delete[] c;
		}
		c = tmp;
	} else {
		delete[] c;
	}
	n = x;
}

template <class T>
inline void array<T>::swap(const array<T>& a)
{
	if (this != &a) {
		value_type* tmp = c;
		c = a.c;
		a.c = tmp;
	}
}

template <class T>
inline bool array<T>::equiv(const array<T>& a) const
{
	for (size_type i = 0; i < n; i++) {
		if (c[i] != a.c[i]) return false;
	}
	return true;
}

template <class T>
inline bool array<T>::cmp(const array<T>& a) const
{
	for (size_type i = 0; i < n; i++) {
		if (c[i] < a.c[i]) return true;
		if (a.c[i] < c[i]) return false;
	}
	return false;
}

template <class T>
inline bool array<T>::operator==(const array<T>& a) const
{
	return equiv(a);
}

template <class T>
inline bool array<T>::operator!=(const array<T>& a) const
{
	return !equiv(a);
}

template <class T>
inline bool array<T>::operator<(const array<T>& a) const
{
	return cmp(a);
}

template <class T>
inline void array<T>::fill(array<T>::const_reference a)
{
	for (size_type i = 0; i < n; i++) c[i] = a;
}

template <class T>
inline std::istream& operator>>(std::istream& is, array<T>& a)
{
	if (is) a.input(is);
	return is;
}

template <class T>
inline std::ostream& operator<<(std::ostream& os, const array<T>& a)
{
	if (os) a.output(os);
	return os;
}

template <class T>
void array<T>::input(std::istream& is)
{
	if (!is) return;
	const size_type default_size = 4096; // bytes
	size_type buf_size =
		(size() > 0) ?
			size() :
			(sizeof(value_type) < default_size) ?
				default_size / sizeof(value_type) : 1;
	thisclass tmp(buf_size);
	size_type i;
	for (i = 0; ; i++) {
		is >> tmp[i];
		if (!is) break;
		if (buf_size - 1 <= i) tmp.resize(buf_size *= 2);
	}
	if (size() > 0) delete[] c;
	n = i;
	if (n > 0) {
		c = new value_type[n];
		assign(tmp.c, c, n);
	}
}

template <class T>
void array<T>::output(std::ostream& os) const
{
	if (!os) return;
	if (size() > 0) {
		size_type i;
		for (i = 0; i < size() - 1; i++) os << (*this)[i] << ' ';
		os << (*this)[i];
	}
}

#endif /* ARRAY_H */
