/* -*- C++ -*-
 *
 * <<< r3010_inst.h >>>
 *
 * --- Copyright (C) 1996-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3010_INST_H
#define R3010_INST_H 1

#include <iostream>
#include <isis/r3000_typedef.h>

class r3010_inst
{
private:
	static const int NFGR = 32;
public:
	enum reg_idx_t {
		REG_SRC1, REG_SRC2, REG_DST
	};
	enum op_t {
		OP_COP1 = 021, OP_LWC1 = 061, OP_SWC1 = 071
	};
	enum sub_t {
		SUB_MF = 000, SUB_CF = 002, SUB_MT = 004, SUB_CT = 006,
		SUB_BC = 010
	};
	enum br_t {
		BR_BC1F = 000, BR_BC1T = 001
	};
	enum fmt_t {
		FMT_S, FMT_D, FMT_W = 4
	};
	enum func_t {
		FUNC_ADD, FUNC_SUB, FUNC_MUL, FUNC_DIV,
		FUNC_ABS = 005, FUNC_MOV, FUNC_NEG,
		FUNC_CVT_S = 040, FUNC_CVT_D, FUNC_CVT_W = 044,
		FUNC_C_F = 060, FUNC_C_UN, FUNC_C_EQ, FUNC_C_UEQ,
		FUNC_C_OLT, FUNC_C_ULT, FUNC_C_OLE, FUNC_C_ULE,
		FUNC_C_SF, FUNC_C_NGLE, FUNC_C_SEQ, FUNC_C_LT,
		FUNC_C_NGE, FUNC_C_LE, FUNC_C_NGT
	};
	static const int COND_LESS = (1<<2);
	static const int COND_EQUAL = (1<<1);
	static const int COND_UNORDERED = (1<<0);
private:
	r3000_word _inst;
	bool _have_inst;
	bool _use_reg[3];		// index is reg_idx_t
	int _regnum[3];			// index is reg_idx_t
	enum { ft, fs, fd };
	int _reg[3];			// index is enum { ft, fs, fd };
	op_t _op;
	bool _op25;
	sub_t _sub;
	br_t _br;				// not used in R3010 modules
	fmt_t _fmt;
	func_t _func;
	int _base;				// not used in R3010 modules
	unsigned int _offset;	// not used in R3010 modules
	bool _fpa;
	bool _calc;
	bool _bus_from, _bus_to;
	void disasm_cop1(std::ostream &) const;
public:
	void reset(void);
	void decode(r3000_word);
	r3010_inst(void) {}
	r3010_inst(r3000_word w) { reset(); decode(w); }
	// -----------------------------------------------
	bool have_inst(void) const { return _have_inst; }
	bool use_src(reg_idx_t idx) const { return _use_reg[idx]; }
	bool use_dst(void) const { return _use_reg[REG_DST]; }
	bool is_calc(void) const { return _calc; }
	bool is_fpa(void) const { return _fpa; }
	bool is_read(void) const { return _bus_from; }
	bool is_write(void) const { return _bus_to; }
	// -----------------------------------------------
	r3000_word inst(void) const { return _inst; }
	// bit 31..26
	op_t op(void) const { return _op; }
	// bit 25
	bool op25(void) const { return _op25; }
	// bit 24..21
	sub_t sub(void) const { return _sub; }
	br_t br(void) const { return _br; }
	fmt_t fmt(void) const { return _fmt; }
	// bit 5..0
	func_t func(void) const { return _func; }
	int src(reg_idx_t idx) const { return _regnum[idx]; }
	int dst(void) const { return _regnum[REG_DST]; }
	// bit 25..21
	int base(void) const { return _base; }
	// bit 15..0
	unsigned int offset(void) const { return _offset; }
	// ------------------------------------------------
	const char* disasm(void) const;
};

std::ostream& operator<<(std::ostream&, const r3010_inst&);

#endif /* R3010_INST_H */
