/* -*- C++ -*-
 *
 * <<< r3010_forwarder.h >>>
 *
 * --- Copyright (C) 1995-2002 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3010_FORWARDER_H
#define R3010_FORWARDER_H 1

#include <cstddef>
#include <iostream>
#include "r3010_freelist.h"

template<class T>
class r3010_forwarder {
public:
	class forward_elem {
	private:
		typedef forward_elem thisclass;
	private:
		static r3010_freelist<void*> free_forward_elem;
		thisclass* next;
		T* elem;
	public:
		forward_elem(T* e = 0, thisclass* n = 0) : next(n), elem(e) {}
		void* operator new(size_t size) {
			void *n = free_forward_elem.get();
			if (n == 0) n = ::new char[size];
			return n;
		}
		void operator delete(void *elem) { free_forward_elem.put(elem); }
		T* search(T&);
		bool del(void*, thisclass**);
		void reset(void) {
			delete elem;
			if (next) next->reset();
			delete next;
		}
		void output(std::ostream&) const;
	};
	class forward_list {
	private:
		typedef forward_list thisclass;
	private:
		static r3010_freelist<void*> free_forward_list;
		forward_list* next;
		forward_elem* list;
	public:
		forward_list(thisclass* p = 0) : next(p), list(0) {}
		void* operator new(size_t size) {
			void *n = free_forward_list.get();
			if (n == 0) n = ::new char[size];
			return n;
		}
		void operator delete(void *elem) { free_forward_list.put(elem); }
		bool is_empty() const { return !list; }
		T* search(T&);
		void* put(T*);
		bool del(void*, thisclass**);
		void reset(void) {
			if (list) list->reset();
			delete list;
			if (next) next->reset();
			delete next;
		}
		void output(std::ostream&) const;
	};
private:
	forward_list* list;
public:
	r3010_forwarder(void) : list(0) {}
	void reset(void) { if (list) list->reset(); delete list; }
	void* put(T*);
	T* get(T&);
	void del(void*);
	void step(void);
	void output(std::ostream&) const;
};

template <class T>
std::ostream& operator<<
	(std::ostream&, const typename r3010_forwarder<T>::forward_elem&);
template <class T>
std::ostream& operator<<
	(std::ostream&, const typename r3010_forwarder<T>::forward_list&);
template <class T>
std::ostream& operator<<(std::ostream&, const r3010_forwarder<T>&);

template <class T>
r3010_freelist<void*> r3010_forwarder<T>::forward_elem::free_forward_elem;
template <class T>
r3010_freelist<void*> r3010_forwarder<T>::forward_list::free_forward_list;

template <class T>
T* r3010_forwarder<T>::forward_elem::search(T& comp)
{
	if (*elem == comp) return elem;
	if (next) return next->search(comp);
	return 0;
}

template <class T>
bool r3010_forwarder<T>::forward_elem::del
	(void *id, typename r3010_forwarder<T>::forward_elem** n)
{
	if ((void*)(this) == id) {
		*n = next;
		delete elem;
		delete this;
		return true;
	} else if (next) {
		thisclass* l;
		if (next->del(id, &l)) {
			next = l;
			*n = this;
			return true;
		}
	}
	return false;
}

template <class T>
void r3010_forwarder<T>::forward_elem::output(std::ostream& os) const
{
	os << "{";
	for (const thisclass* l = this; l; l = l->next) {
		os << *l->elem << ",";
	}
	os << "}";
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class T>
std::ostream& operator<<
	(std::ostream& os, const typename r3010_forwarder<T>::forward_elem& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}

template <class T>
void* r3010_forwarder<T>::forward_list::put(T* elem)
{
	forward_elem* n = new forward_elem(elem, list);
	list = n;
	return (void*)(n);
}

template <class T>
T* r3010_forwarder<T>::forward_list::search(T& comp)
{
	if (list) {
		T* t = list->search(comp);
		if (t) return t;
	}
	if (next) return next->search(comp);
	return 0;
}

template <class T>
bool r3010_forwarder<T>::forward_list::del
	(void* id, typename r3010_forwarder<T>::forward_list** n)
{
	forward_elem* l;
	if (list && list->del(id, &l)) {
		list = l;
		if (l == 0) {
			*n = next;
			return true;
		}
	}
	else if (next) {
		forward_list* lst;
		if (next->del(id, &lst)) {
			delete next;
			next = lst;
		}
	}
	return false;
}

template <class T>
void r3010_forwarder<T>::forward_list::output(std::ostream& os) const
{
	os << "{";
	for (const thisclass* l = this; l; l = l->next) {
		l->list->output(os);
		os << ",";
	}
	os << "}";
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class T>
std::ostream& operator<<
	(std::ostream& os, const typename r3010_forwarder<T>::forward_list& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}

template <class T>
void* r3010_forwarder<T>::put(T* elem)
{
	if (list == 0) list = new forward_list;
	return list->put(elem);
}

template <class T>
void r3010_forwarder<T>::step(void)
{
	if (list && !list->is_empty()) {
		forward_list* n = new forward_list(list);
		list = n;
	}
}

template <class T>
T* r3010_forwarder<T>::get(T& comp)
{
	if (list == 0) return 0;
	return list->search(comp);
}

template <class T>
void r3010_forwarder<T>::del(void* id)
{
	forward_list* lst;
	if (list) {
		if (list->del(id, &lst)) {
			delete list;
			list = lst;
		}
	}
}

template <class T>
void r3010_forwarder<T>::output(std::ostream& os) const
{
	list->output(os);
}

template <class T>
std::ostream& operator<<(std::ostream& os, const r3010_forwarder<T>& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}

#endif /* R3010_FORWARDER_H */
