/*
 * <<< r3000_test.cc >>>
 *
 * --- Test program for r3000
 *     Copyright (C) 2000-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <iostream>
#include <isis/memory_control_unit.h>
#include <isis/multi_io_unit.h>
#include <isis/bus_error_detector.h>
#include <isis/r3010.h>
#include "mips_executable.h"
#include "r3000.h"
#include "r3000_directmap_cache.h"

using namespace std;

int main(int, char** argv)
{
	typedef r3000_word word;
	const word local_memory_address = 0;
	const word local_memory_size = 0x20000000;
	const word multi_io_area_address = 0xbfd00000;
	const word multi_io_area_size = 0x100;
	size_t freq_mag = 1, icache_size = 0x4000, dcache_size = 0x1000;
	unsigned int wait_time = 0, verbose_level = 0;
	bool icache_flag = false, dcache_flag = false, fpa_flag = false,
		 streaming_flag = true, clock_flag = false, trace_flag = false;

	// read commandline argument for simulator
	while (*++argv != NULL && **argv == '-') {
		while (*++*argv != '\0') {
			bool flag = 0;
			switch (**argv) {
			case 'c': clock_flag = true; break;
			case 'f': fpa_flag = true; break;
			case 's': streaming_flag = false; break;
			case 't': trace_flag = clock_flag = true; break;
			case 'v': verbose_level++; clock_flag = true; break;
			case 'w': wait_time = atoi(*argv + 1); flag = true; break;
			case 'd':
				dcache_flag = true;
				if (isdigit(*(*argv + 1))) {
					dcache_size = atoi(*argv + 1);
					flag = true;
				}
				break;
			case 'i':
				icache_flag = true;
				if (isdigit(*(*argv + 1))) {
					icache_size = atoi(*argv + 1);
					flag = true;
				}
				break;
			case 'm':
				freq_mag = atoi(*argv + 1);
				break;
			default: break;
			}
			if (flag) break;
		}
	}
	if (*argv == NULL) {
		cerr << "No executable file specified." << endl;
		return 1;
	}

	// create units and connect each other
	r3000 pu;
	r3000_directmap_cache inst_cache(4, 4), data_cache(4, 4); // size is dummy
	r3010 fpa;
	memory_control_unit<word> mem_ctl;
	mapped_memory<word> local_mem;
	multi_io_unit<word> multi_io;
	bus_error_detector<word> buserror_mon;
	pu.port_ref().connect(mem_ctl.port_ref());
	pu.port_ref().connect(multi_io.port_ref());
	pu.port_ref().connect(buserror_mon.port_ref());
	mem_ctl.connect_memory(local_mem);

	// setup caches, fpa, frequency
	if (icache_flag) {
		inst_cache.resize(icache_size, 0x10);
		pu.connect_instruction_cache(inst_cache);
	}
	if (dcache_flag) {
		data_cache.resize(dcache_size, 0x10);
		pu.connect_data_cache(data_cache);
	}
	if (freq_mag > 1) {
		pu.set_frequency_magnification(freq_mag);
	}
	if (fpa_flag) {
		pu.connect_coprocessor(fpa);
	}
	if (!streaming_flag) {
		pu.disable_streaming();
	}

	// setup memory map
	local_mem.set_top(local_memory_address);
	local_mem.resize(local_memory_size);
	multi_io.set_address(multi_io_area_address, multi_io_area_size);
	multi_io.set_sysinfo("local_memory_address", local_memory_address);
	multi_io.set_sysinfo("local_memory_size", local_memory_size);

	// setup memory wait
	if (wait_time > 0) {
		mem_ctl.set_read_wait(wait_time);
		mem_ctl.set_write_wait(wait_time);
	}

	// setup file table
	multi_io.set_file_table_size(16);
	multi_io.set_file_input_stream(0, cin);
	multi_io.set_file_output_stream(1, cout);
	multi_io.set_file_output_stream(2, cerr);

	// load program to memory
	{
		ifstream file(*argv);
		if (!file) {
			cerr << *argv << ": No such file or directory." << endl;
			return 1;
		}
		mips_executable obj;
		obj.load_header(file);
		if (!obj.is_valid() || !obj.load_body_to_memory(file, local_mem)) {
			cerr << *argv << ": Bad file type." << endl;
			return 1;
		}
		pu.program_counter() = obj.entry_point();
	}
	multi_io.set_commandline_argument((const char* const*)(argv));

	// show simulation conditions
	if (verbose_level >= 1) {
		cout << "processor:   r3000" << endl;
		cout << "coprocessor: " << (fpa_flag ? "yes" : "no") << endl;
		cout << "icache:      ";
		if (icache_flag) {
			cout << "0x" << hex << icache_size << dec << endl;
		} else {
			cout << "none" << endl;
		}
		cout << "dcache:      ";
		if (dcache_flag) {
			cout << "0x" << hex << dcache_size << dec << endl;
		} else {
			cout << "none" << endl;
		}
		cout << "memory:      0x" << hex << setw(8) << setfill('0')
			 << local_memory_address << " - 0x" << setw(8) << setfill('0')
			 << local_memory_address + local_memory_size - 1 << dec << endl;
		cout << "i/o unit:    0x" << hex << setw(8) << setfill('0')
			 << multi_io_area_address << " - 0x" << setw(8) << setfill('0')
			 << multi_io_area_address + multi_io_area_size - 1 << dec << endl;
		cout << "memory wait: " << wait_time << "clk" << endl;
		cout << "arguments:   \"";
		for (int i = 0; argv[i] != NULL; i++) {
			if (i != 0) cout << ' ';
			cout << argv[i];
		}
		cout << '"' << endl << endl;
	}

	// execute simulation
	while (1) {
		if (verbose_level >= 2) {
			cout << "clk:" << multi_io.timer_clock_value() << endl
				 << pu << endl << endl;
		} else if (trace_flag) {
			cout << "0x" << hex << pu.program_counter() << dec << endl;
		}
		if (multi_io.is_halt()) break;
		if (buserror_mon.is_bus_error()) {
			cout << hex
				 << "bus error(I:0x" << pu.program_counter()
				 << ", D:0x" << buserror_mon.bus_error_address() << ')'
				 << dec << endl;
			break;
		}
		pu.clock_in();
		mem_ctl.clock_in();
		multi_io.clock_in();
		buserror_mon.clock_in();
		pu.clock_out();
		mem_ctl.clock_out();
		multi_io.clock_out();
		buserror_mon.clock_out();
	}

	// show result and exit
	if (clock_flag) {
		cout << "clock: " << multi_io.timer_clock_value() << endl;
	}
	return multi_io.commandline_status();
}
