/* -*- C++ -*-
 *
 * <<< r3000_integer_unit.h >>>
 *
 * --- R3000 integer unit class 'r3000_integer_unit'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3000_INTEGER_UNIT_H
#define R3000_INTEGER_UNIT_H 1

#include <iostream>
#include <string>
#include <isis/r3000_typedef.h>
#include "r3000_instruction_buffer.h"
#include "r3000_multiply_unit.h"
#include "r3000_register_file.h"
#include "r3000_stall_buffer.h"

class r3000_memory_access_unit;
class r3000_write_buffer_unit;
class r3000_cp0;
class r3010;

class r3000_integer_unit
{
public:
	typedef r3000_word address_type;
	typedef r3000_word data_type;
	typedef signed_r3000_word signed_data_type;
	typedef unsigned_r3000_word unsigned_data_type;
	static const size_t sizeof_data_type = sizeof_r3000_word;
private:
	typedef r3000_integer_unit thisclass;
	r3000_instruction_buffer inst_buf;
	r3000_register_file gpr;
	r3000_multiply_unit mul_unit;
	r3000_stall_buffer stall_buf;
	r3000_memory_access_unit& ma_unit;
	r3000_write_buffer_unit& wb_unit;
	r3000_cp0& cp0;
	r3010* cp1;
	address_type pc_;
	bool streaming_flag_;
	bool use_streaming_;
	bool is_cacheable(address_type) const;
	bool is_cacheable_instruction(address_type) const;
	bool is_cacheable_data(address_type) const;
	// execute in run cycle, streaming
	void instruction_fetch(void);
	void register_fetch(void);
	void execute(void);
	void memory_access(void);
	void writeback(void);
	// execute in stall cycle
	void execute_instruction_cache_miss_stall(void);
	void execute_instruction_cache_miss_refill(void);
	void execute_instruction_cache_miss_fixup(void);
	void execute_instruction_cache_miss_fixup_to_streaming(void);
	void execute_uncacheable_instruction_stall(void);
	void execute_uncacheable_instruction_fixup(void);
	void execute_data_cache_miss_stall_wflush(void);
	void execute_data_cache_miss_stall(void);
	void execute_data_cache_miss_refill(void);
	void execute_data_cache_miss_fixup(void);
	void execute_uncacheable_data_stall_wflush(void);
	void execute_uncacheable_data_stall(void);
	void execute_uncacheable_data_fixup(void);
	void execute_partial_word_stores_fixup(void);
	void execute_write_busy_stall(void);
	void execute_write_busy_fixup(void);
	void execute_multiply_unit_busy_stall(void);
	void execute_multiply_unit_busy_fixup(void);
	void execute_coprocessor_ifetch_busy_stall(void);
	void execute_coprocessor_ifetch_busy_fixup(void);
	void execute_coprocessor_fetch_busy_stall(void);
	void execute_coprocessor_fetch_busy_fixup(void);
	void execute_coprocessor_output_busy_stall(void);
	void execute_coprocessor_output_busy_fixup(void);
public:
	r3000_integer_unit(r3000_memory_access_unit&, r3000_write_buffer_unit&,
		r3000_cp0&);
	r3000_integer_unit(const thisclass&, r3000_memory_access_unit&,
		r3000_write_buffer_unit&, r3000_cp0&);
	virtual ~r3000_integer_unit();
	r3000_integer_unit& operator=(const r3000_integer_unit&);
	data_type register_file(size_t i) const { return gpr[i]; }
	data_type hi(void) const { return mul_unit.hi(); }
	data_type lo(void) const { return mul_unit.lo(); }
	address_type pc(void) const { return pc_; }
	data_type& register_file(size_t i) { return gpr[i]; }
	data_type& hi(void) { return mul_unit.hi(); }
	data_type& lo(void) { return mul_unit.lo(); }
	address_type& pc(void) { return pc_; }
	bool is_run_cycle(void) const { return stall_buf.is_run_cycle(); }
	bool is_stall_cycle(void) const { return stall_buf.is_stall_cycle(); }
	bool is_internal_cause_stall_cycle(void) const
		{ return stall_buf.is_internal_cause_stall_cycle(); }
	bool is_external_cause_stall_cycle(void) const
		{ return stall_buf.is_external_cause_stall_cycle(); }
	bool is_streaming(void) const { return streaming_flag_; }
	void enable_streaming(void) { use_streaming_ = true; }
	void disable_streaming(void) { use_streaming_ = false; }
	void clock(void);
	void reset(void);
	void output(std::ostream&) const;
	bool output(std::ostream&, const std::string& s) const;
	bool is_connected_to_coprocessor(void) const { return cp1 != NULL; }
	virtual void connect_coprocessor(r3010&);
	virtual void disconnect_coprocessor(void);
};

std::ostream& operator<<(std::ostream&, const r3000_integer_unit&);

#endif /* R3000_INTEGER_UNIT_H */
