/* -*- C++ -*-
 *
 * <<< r3000_instruction.h >>>
 *
 * --- R3000 instruction class 'r3000_instruction'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3000_INSTRUCTION_H
#define R3000_INSTRUCTION_H 1

#include <cstddef>
#include <iostream>
#include <string>
#include <isis/r3000_typedef.h>

class r3000_instruction
{
public:
	enum operation_type { I_TYPE, J_TYPE, R_TYPE, CO_TYPE };
	enum opcode_type {
		LB,		LBU,	LH,		LHU,	LW,		LWL,	LWR,
		SB,		SH,		SW,		SWL,	SWR,
		ADDI,	ADDIU,	SLTI,	SLTIU,
		ANDI,	ORI,	XORI,	LUI,
		ADD,	ADDU,	SUB,	SUBU,	SLT,	SLTU,
		AND,	OR,		XOR,	NOR,
		SLL,	SRL,	SRA,	SLLV,	SRLV,	SRAV,
		MULT,	MULTU,	DIV,	DIVU,
		MFHI,	MFLO,	MTHI,	MTLO,
		J,		JAL,	JR,		JALR,
		BEQ,	BNE,	BLEZ,	BGTZ,	BLTZ,	BGEZ,	BLTZAL,	BGEZAL,
		SYSCALL,BREAK,
		LWC,	SWC,	MTC,	MFC,	CTC,	CFC,
		COP,	BCT,	BCF,
		TLBR,	TLBWI,	TLBWR,	TLBP,	RFE,
		RESERVED, UNKNOWN,
		SPECIAL_, BCOND_, COP_, BC_
	};
	typedef r3000_word data_type;
	typedef r3000_word address_type;
	typedef signed_r3000_word signed_data_type;
	typedef unsigned_r3000_word unsigned_data_type;
	static const size_t sizeof_data_type = sizeof_r3000_word;
private:
	typedef r3000_instruction thisclass;
	data_type image_;
	address_type pc_;
	opcode_type opcode_;
	operation_type operation_;
public:
	r3000_instruction(void);
	r3000_instruction(const thisclass&);
	~r3000_instruction();
	void output(std::ostream&) const;
	data_type image(void) const { return image_; }
	address_type pc(void) const { return pc_; }
	operation_type operation(void) const { return operation_; }
	opcode_type opcode(void) const { return opcode_; }
	bool is_i_type(void) const { return operation() == I_TYPE; }
	bool is_j_type(void) const { return operation() == J_TYPE; }
	bool is_r_type(void) const { return operation() == R_TYPE; }
	bool is_co_type(void) const { return operation() == CO_TYPE; }
	bool is_load(void) const
		{ return ((image() & 0xa0000000UL) == 0x80000000UL); }
	bool is_store(void) const
		{ return ((image() & 0xa0000000UL) == 0xa0000000UL); }
	bool is_load_or_store(void) const
		{ return ((image() & 0x80000000UL) == 0x80000000UL); }
	int rs(void) const { return (image() >> 21) & 0x1f; }
	int rt(void) const { return (image() >> 16) & 0x1f; }
	int rd(void) const { return (image() >> 11) & 0x1f; }
	int sa(void) const { return (image() >> 6) & 0x1f; }
	int base(void) const { return rs(); }
	data_type immediate(void) const { return image() & 0xffffUL; }
	signed_data_type signed_immediate(void) const
		{ return (image() & 0x8000UL) ?
				 (signed_data_type(immediate()) - 0x10000L) :
				 (signed_data_type(immediate())); }
	data_type target(void) const { return data_type(image() & 0x3ffffffUL); }
	int coprocessor_number(void) const { return (image() >> 26) & 3; }
	data_type break_code(void) const
		{ return data_type((image() >> 6) & 0xfffffUL); }
	address_type jump_address(void) const
		{ return address_type(((pc() + sizeof_data_type * 2) & 0xf0000000UL) |
							  (target() << 2)); }
	address_type branch_address(void) const
		{ return address_type((pc() + sizeof_data_type) +
							  (signed_immediate() << 2)); }
	std::string opcode_string(void) const;
	std::string operation_string(void) const;
	void set(address_type, data_type);
	void clear(void) { set(0, 0); }
};

std::ostream& operator<<(std::ostream&, const r3000_instruction&);

#endif /* R3000_INSTRUCTION_H */
