/* -*- C++ -*-
 *
 * <<< r3000_cp0_register_file.h >>>
 *
 * --- R3000 CP0 register file class 'r3000_cp0_register_file'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef R3000_CP0_REGISTER_FILE_H
#define R3000_CP0_REGISTER_FILE_H 1

#include <iostream>
#include <string>
#include <isis/c_array.h>
#include <isis/r3000_typedef.h>
#include <isis/register_file.h>

class r3000_cp0_register_file
	: public register_file< r3000_word, c_array<r3000_word, 32> >
{
private:
	typedef r3000_cp0_register_file thisclass;
	typedef register_file< r3000_word, c_array<r3000_word, 32> > inherited;
public:
	typedef inherited::container_type container_type;
	typedef inherited::value_type value_type;
	typedef inherited::size_type size_type;
private:
	enum {
		cause_branch_delay_mask				= 0x80000000UL,
		cause_coprocessor_error_mask		= 0x30000000UL,
		cause_hardware_interrupt_wait_mask	= 0x0000fc00UL,
		cause_software_interrupt_wait_mask	= 0x00000300UL,
		cause_exception_code_mask			= 0x0000003cUL
	};
	enum {
		sr_coprocessor_usable_mask			= 0xf0000000UL,
		sr_reversal_endianess_mask			= 0x02000000UL,
		sr_bootstrap_exception_vectors_mask	= 0x00400000UL,
		sr_tlb_shutdown_mask				= 0x00200000UL,
		sr_parity_error_mask				= 0x00100000UL,
		sr_cache_miss_error_mask			= 0x00080000UL,
		sr_parity_zero_mask					= 0x00040000UL,
		sr_swap_caches_mask					= 0x00020000UL,
		sr_isolate_caches_mask				= 0x00010000UL,
		sr_interrupt_mask					= 0x0000ff00UL,
		sr_user_mode_old_mask				= 0x00000020UL,
		sr_user_mode_previous_mask			= 0x00000010UL,
		sr_user_mode_current_mask			= 0x00000008UL,
		sr_interrupt_enable_old_mask		= 0x00000004UL,
		sr_interrupt_enable_previous_mask	= 0x00000002UL,
		sr_interrupt_enable_current_mask	= 0x00000001UL
	};
public:
	static std::string register_name(int);
	static int register_number(const std::string&);
	r3000_cp0_register_file(void);
	r3000_cp0_register_file(const thisclass&);
	virtual ~r3000_cp0_register_file();
	virtual void output(std::ostream&) const;
	value_type index   (void) const { return (*this)[ 0]; }
	value_type random  (void) const { return (*this)[ 1]; }
	value_type entrylo (void) const { return (*this)[ 2]; }
	value_type context (void) const { return (*this)[ 4]; }
	value_type badvaddr(void) const { return (*this)[ 8]; }
	value_type entryhi (void) const { return (*this)[10]; }
	value_type sr	   (void) const { return (*this)[12]; }
	value_type cause   (void) const { return (*this)[13]; }
	value_type epc	   (void) const { return (*this)[14]; }
	value_type prid	   (void) const { return (*this)[15]; }
	value_type& index	(void) { return (*this)[ 0]; }
	value_type& random	(void) { return (*this)[ 1]; }
	value_type& entrylo	(void) { return (*this)[ 2]; }
	value_type& context	(void) { return (*this)[ 4]; }
	value_type& badvaddr(void) { return (*this)[ 8]; }
	value_type& entryhi	(void) { return (*this)[10]; }
	value_type& sr		(void) { return (*this)[12]; }
	value_type& cause	(void) { return (*this)[13]; }
	value_type& epc		(void) { return (*this)[14]; }
	value_type& prid	(void) { return (*this)[15]; }
	bool is_branch_delay(void) const
		{ return (cause() & cause_branch_delay_mask) > 0; }
	int coprocessor_error_number(void) const
		{ return int((cause() & cause_coprocessor_error_mask) >> 28); }
	int hardware_interrupt_wait_mask(int i) const
		{ return int((cause() >> (10 + i)) & 1); }
	int software_interrupt_wait_mask(int i) const
		{ return int((cause() >> (8 + i)) & 1); }
	int exception_code(void) const
		{ return int((cause() & cause_exception_code_mask) >> 2); }
	bool is_coprocessor_usable(int i) const
		{ return int((sr() >> (28 + i)) & 1); }
	bool is_reversal_endianess(void) const
		{ return (sr() & sr_reversal_endianess_mask) > 0; }
	bool is_bootstrap_exception_vectors(void) const
		{ return (sr() & sr_bootstrap_exception_vectors_mask) > 0; }
	bool is_tlb_shutdown(void) const
		{ return (sr() & sr_tlb_shutdown_mask) > 0; }
	bool is_parity_error(void) const
		{ return (sr() & sr_parity_error_mask) > 0; }
	bool is_cache_miss_error(void) const
		{ return (sr() & sr_cache_miss_error_mask) > 0; }
	bool is_parity_zero(void) const
		{ return (sr() & sr_parity_zero_mask) > 0; }
	bool is_swap_caches(void) const
		{ return (sr() & sr_swap_caches_mask) > 0; }
	bool is_isolate_caches(void) const
		{ return (sr() & sr_isolate_caches_mask) > 0; }
	bool is_hardware_interrupt_mask(int i) const
		{ return int((sr() >> (10 + i)) & 1); }
	bool is_software_interrupt_mask(int i) const
		{ return int((sr() >> (8 + i)) & 1); }
	bool is_user_mode_old(void) const
		{ return (sr() & sr_user_mode_old_mask) > 0; }
	bool is_user_mode_previous(void) const
		{ return (sr() & sr_user_mode_previous_mask) > 0; }
	bool is_user_mode_current(void) const
		{ return (sr() & sr_user_mode_current_mask) > 0; }
	bool is_interrupt_enable_old(void) const
		{ return (sr() & sr_interrupt_enable_old_mask) > 0; }
	bool is_interrupt_enable_previous(void) const
		{ return (sr() & sr_interrupt_enable_previous_mask) > 0; }
	bool is_interrupt_enable_current(void) const
		{ return (sr() & sr_interrupt_enable_current_mask) > 0; }
	void set_branch_delay(void) { cause() |= cause_branch_delay_mask; }
	void reset_branch_delay(void) { cause() &= ~cause_branch_delay_mask; }
	void set_coprocessor_error_number(int i)
		{ cause() = (cause() & ~cause_coprocessor_error_mask) | (i << 28); }
	void set_exception_code(int i)
		{ cause() = (cause() & ~cause_exception_code_mask) | (i << 2); }
	void set_coprocessor_usable(int i)
		{ sr() |= (1 << (28 + i)); }
	void reset_coprocessor_usable(int i)
		{ sr() &= ~(1 << (28 + i)); }
	void set_reversal_endianess(void)
		{ sr() |= sr_reversal_endianess_mask; }
	void reset_reversal_endianess(void)
		{ sr() &= ~sr_reversal_endianess_mask; }
	void set_bootstrap_exception_vectors(void)
		{ sr() |= sr_bootstrap_exception_vectors_mask; }
	void reset_bootstrap_exception_vectors(void)
		{ sr() &= ~sr_bootstrap_exception_vectors_mask; }
	void set_tlb_shutdown(void)
		{ sr() |= sr_tlb_shutdown_mask; }
	void reset_tlb_shutdown(void)
		{ sr() &= ~sr_tlb_shutdown_mask; }
	void set_parity_error(void)
		{ sr() |= sr_parity_error_mask; }
	void reset_parity_error(void)
		{ sr() &= ~sr_parity_error_mask; }
	void set_cache_miss_error(void)
		{ sr() |= sr_cache_miss_error_mask; }
	void reset_cache_miss_error(void)
		{ sr() &= ~sr_cache_miss_error_mask; }
	void set_parity_zero(void)
		{ sr() |= sr_parity_zero_mask; }
	void reset_parity_zero(void)
		{ sr() &= ~sr_parity_zero_mask; }
	void set_swap_caches(void)
		{ sr() |= sr_swap_caches_mask; }
	void reset_swap_caches(void)
		{ sr() &= ~sr_swap_caches_mask; }
	void set_isolate_caches(void)
		{ sr() |= sr_isolate_caches_mask; }
	void reset_isolate_caches(void)
		{ sr() &= ~sr_isolate_caches_mask; }
	void set_hardware_interrupt_mask(int i)
		{ sr() |= (1 << (10 + i)); }
	void reset_hardware_interrupt_mask(int i)
		{ sr() &= ~(1 << (10 + i)); }
	void set_software_interrupt_mask(int i)
		{ sr() |= (1 << (8 + i)); }
	void reset_software_interrupt_mask(int i)
		{ sr() &= ~(1 << (8 + i)); }
	void set_user_mode_old(void)
		{ sr() |= sr_user_mode_old_mask; }
	void reset_user_mode_old(void)
		{ sr() &= ~sr_user_mode_old_mask; }
	void set_user_mode_previous(void)
		{ sr() |= sr_user_mode_previous_mask; }
	void reset_user_mode_previous(void)
		{ sr() &= ~sr_user_mode_previous_mask; }
	void set_user_mode_current(void)
		{ sr() |= sr_user_mode_current_mask; }
	void reset_user_mode_current(void)
		{ sr() &= ~sr_user_mode_current_mask; }
	void set_interrupt_enable_old(void)
		{ sr() |= sr_interrupt_enable_old_mask; }
	void reset_interrupt_enable_old(void)
		{ sr() &= ~sr_interrupt_enable_old_mask; }
	void set_interrupt_enable_previous(void)
		{ sr() |= sr_interrupt_enable_previous_mask; }
	void reset_interrupt_enable_previous(void)
		{ sr() &= ~sr_interrupt_enable_previous_mask; }
	void set_interrupt_enable_current(void)
		{ sr() |= sr_interrupt_enable_current_mask; }
	void reset_interrupt_enable_current(void)
		{ sr() &= ~sr_interrupt_enable_current_mask; }
};

#endif /* R3000_CP0_REGISTER_FILE_H */
