/* -*- C++ -*-
 *
 * <<< mips_executable.h >>>
 *
 * --- Executable file for MIPS 'mips_executable'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General
 *  Public License along with this library; if not, write to the
 *  Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#ifndef MIPS_EXECUTABLE_H
#define MIPS_EXECUTABLE_H 1

#include <cstddef>
#include <iostream>
#include <map>
#include <isis/memory_map.h>
#include <isis/r3000_typedef.h>

// #define MIPS_EXECUTABLE_DEBUG

class mips_executable
{
public:
	typedef r3000_word data_type;
	static const size_t sizeof_data_type = sizeof_r3000_word;
private:
	struct header {
		struct file_header {
			unsigned short f_magic;		// magic number
			unsigned short f_nscns;		// number of sections
			long f_timdat;				// time & date stamp
			long f_symptr;				// file pointer to symbolic header
			long f_nsyms;				// sizeof(symbolic hdr)
			unsigned short f_opthdr;	// sizeof(optional hdr)
			unsigned short f_flags;		// some flags
			bool is_mips_big_endian_object(void) const;
			size_t section_number(void) const { return f_nscns; }
			void input(std::istream&);
		};
		struct aout_header {
			short magic;				// magic number
			short vstamp;				// version stamp
			long tsize;					// text size in bytes
			long dsize;					// initialized data
			long bsize;					// uninitialized data
			long entry;					// entry point
			long text_start;			// base of text
			long data_start;			// base of data
			long bss_start;				// base of bss
			long gprmask;				// general purpose register mask
			long cprmask[4];			// co-processor register masks
			long gp_value;				// the gp value used for this object
			long entry_point(void) const { return entry; }
			void input(std::istream&);
		};
		struct section_header {
			char s_name[8];				// section name
			long s_paddr;				// physical address
			long s_vaddr;				// virtual address
			long s_size;				// section size
			long s_scnptr;				// file ptr to raw data for section
			long s_relptr;				// file ptr to relocation
			long s_lnnoptr;				// file ptr to line numbers (not used)
			unsigned short s_nreloc;	// number of relocation entries
			unsigned short s_nlnno;		// number of line numbers (not used)
			long s_flags;				// some flags
			const char* name(void) const { return s_name; }
			long phisical_address(void) const { return s_paddr; }
			long virtual_address(void) const { return s_vaddr; }
			long size(void) const { return s_size; }
			long data_pointer(void) const { return s_scnptr; }
			void input(std::istream&);
		};
		struct file_header ex_f;
		struct aout_header ex_o;
		bool is_mips_big_endian_object(void) const
			{ return ex_f.is_mips_big_endian_object(); }
		size_t section_number(void) const { return ex_f.section_number(); }
		long entry_point(void) const { return ex_o.entry_point(); }
		void input(std::istream&);
	} header_;
	header::section_header* section_header_;
	bool valid_flag;
	static const size_t sizeof_file_header = 20;
	static const size_t sizeof_aout_header = 56;
	static const size_t sizeof_section_header = 40;
	static void disassemble_instruction(std::ostream&, data_type, data_type);
public:
	template <class M>
		static void disassemble(std::ostream&, const M&,
								const typename M::address_type,
								const typename M::address_type);
	mips_executable(void);
	~mips_executable();
	bool is_valid(void) const { return valid_flag; }
	long entry_point(void) const { return header_.entry_point(); }
	void load_header(std::istream&);
	template <class M>
		bool load_body_to_memory(std::istream&, M&);
	void input(std::istream&);
};

template <class M>
bool mips_executable::load_body_to_memory(std::istream& is, M& mem)
{
	typedef typename M::address_type address_type;
	typedef typename M::size_type size_type;
	typedef typename M::char_type char_type;
	if (!is_valid()) return false;
	const size_t section_number = header_.section_number();
#ifdef MIPS_EXECUTABLE_DEBUG
	cerr << "there is " << section_number << " sections ..." << endl;
#endif // MIPS_EXECUTABLE_DEBUG
	typedef std::map<long, size_t> map_type;
	typedef map_type::const_iterator CI;
	map_type sort_table;
	for (size_t i = 0; i < section_number; i++) {
		if (section_header_[i].data_pointer() == 0) continue;
		sort_table[section_header_[i].data_pointer()] = i;
	}
	for (CI p = sort_table.begin(); p != sort_table.end(); p++) {
		size_t i = p->second;
		const address_type addr =
			address_type(section_header_[i].virtual_address());
		const size_type size = size_type(section_header_[i].size());
#ifdef MIPS_EXECUTABLE_DEBUG
		cerr << "section " << i << ", size " << size << endl;
#endif // MIPS_EXECUTABLE_DEBUG
		const size_t buf_size = 1024;
		char_type buf[buf_size];
		size_type offset = 0;
		while (offset < size) {
			size_type block_size;
			if (size - offset > buf_size) {
				is.read(buf, sizeof(buf));
				block_size = size_type(buf_size);
			} else {
				is.read(buf, (size - offset) * sizeof(char_type));
				block_size = size - offset;
			}
#ifdef MIPS_EXECUTABLE_DEBUG
			cerr << " " << setw(4) << (offset / 1024) << "k: ";
#endif // MIPS_EXECUTABLE_DEBUG
			for (size_t i = 0; i < block_size; i++) {
				mem.write_char(addr + offset + i, buf[i]);
#ifdef MIPS_EXECUTABLE_DEBUG
				if (i % 16 == 15) cerr << '.';
#endif // MIPS_EXECUTABLE_DEBUG
			}
#ifdef MIPS_EXECUTABLE_DEBUG
			cerr << endl;
#endif // MIPS_EXECUTABLE_DEBUG
			offset += block_size;
		}
#ifdef MIPS_EXECUTABLE_DEBUG
		cerr << "... done." << endl;
#endif // MIPS_EXECUTABLE_DEBUG
	}
	return true;
}

template <class M>
void mips_executable::disassemble
	(std::ostream& os, const M& mem, typename M::address_type top,
	 typename M::address_type size)
{
	using namespace std;
	typedef typename M::address_type address_type;
	typedef typename M::size_type size_type;
	const size_t sizeof_data_type = M::sizeof_data_type;
	const char fill = os.fill();
	const ios::fmtflags flags = os.flags();
	os << hex << setfill('0');
	for (address_type adr = top; adr < top + size; adr += sizeof_data_type) {
		if (!mem.is_valid(adr)) continue;
		disassemble_instruction(os, r3000_word(adr), r3000_word(mem.read(adr)));
		if (adr < top + size - sizeof_data_type) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
}

#endif /* MIPS_EXECUTABLE_H */
