/*
 * <<< mips_executable.cc >>>
 *
 * --- Executable file for MIPS 'mips_executable'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include "mips_executable.h"
#include "r3000_instruction.h"

using namespace std;

const unsigned short MIPSEBMAGIC = 0x0160;

template <class T>
static void get_big_endian(istream& is, size_t size, T& value)
{
	value = 0;
	for (size_t i = 0; i < size; i++) {
		value = (value << 8) + T(is.get());
	}
}

bool mips_executable::header::file_header::is_mips_big_endian_object(void) const
{
	return f_magic == MIPSEBMAGIC;
}

void mips_executable::header::file_header::input(istream& is)
{
	get_big_endian(is, 2, f_magic);
	get_big_endian(is, 2, f_nscns);
	get_big_endian(is, 4, f_timdat);
	get_big_endian(is, 4, f_symptr);
	get_big_endian(is, 4, f_nsyms);
	get_big_endian(is, 2, f_opthdr);
	get_big_endian(is, 2, f_flags);
}

void mips_executable::header::aout_header::input(istream& is)
{
	get_big_endian(is, 2, magic);
	get_big_endian(is, 2, vstamp);
	get_big_endian(is, 4, tsize);
	get_big_endian(is, 4, dsize);
	get_big_endian(is, 4, bsize);
	get_big_endian(is, 4, entry);
	get_big_endian(is, 4, text_start);
	get_big_endian(is, 4, data_start);
	get_big_endian(is, 4, bss_start);
	get_big_endian(is, 4, gprmask);
	get_big_endian(is, 4, cprmask[0]);
	get_big_endian(is, 4, cprmask[1]);
	get_big_endian(is, 4, cprmask[2]);
	get_big_endian(is, 4, cprmask[3]);
	get_big_endian(is, 4, gp_value);
}

void mips_executable::header::section_header::input(istream& is)
{
	for (size_t i = 0; i < sizeof(s_name); i++) s_name[i] = is.get();
	get_big_endian(is, 4, s_paddr);
	get_big_endian(is, 4, s_vaddr);
	get_big_endian(is, 4, s_size);
	get_big_endian(is, 4, s_scnptr);
	get_big_endian(is, 4, s_relptr);
	get_big_endian(is, 4, s_lnnoptr);
	get_big_endian(is, 2, s_nreloc);
	get_big_endian(is, 2, s_nlnno);
	get_big_endian(is, 4, s_flags);
}

void mips_executable::header::input(istream& is)
{
	ex_f.input(is);
	ex_o.input(is);
}

mips_executable::mips_executable(void)
	: section_header_(NULL)
{}

mips_executable::~mips_executable()
{
	if (valid_flag) delete[] section_header_;
}

void mips_executable::load_header(istream& is)
{
	if (valid_flag) delete[] section_header_;
	header_.input(is);
	const size_t section_number = header_.section_number();
	if (section_number == 0) {
		valid_flag = false;
	} else {
		long first_section_ptr = 0;
		section_header_ = new header::section_header[section_number];
		for (size_t i = 0; i < section_number; i++) {
			section_header_[i].input(is);
			long ptr = section_header_[i].data_pointer();
			if (ptr != 0 && (i == 0 || ptr < first_section_ptr)) {
				first_section_ptr = ptr;
			}
		}
		long loaded_ptr = sizeof_file_header + sizeof_aout_header
						+ sizeof_section_header * section_number;
		if (loaded_ptr < first_section_ptr) {
			is.ignore(first_section_ptr - loaded_ptr);
		}
		valid_flag = true;
	}
}

void mips_executable::disassemble_instruction
	(ostream& os, mips_executable::data_type adr,
	 mips_executable::data_type dat)
{
	r3000_instruction inst;
	inst.set(adr, dat);
	os << setw(sizeof_data_type * 2) << adr << ": "
	   << setw(sizeof_data_type * 2) << inst.image() << "  "
	   << inst;
}
