/* -*- C++ -*-
 *
 * <<< port.h >>>
 *
 * --- port class 'port'
 *     Copyright (C) 1997-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef PORT_H
#define PORT_H 1

#include <cassert>
#include <iostream>
#include <vector>
#include <isis/root_object.h>
#include <isis/packet.h>
#include <isis/asynchronous_unit.h>

class port;

class port : public root_object
{
public:
	typedef int id_type;
	typedef int resource_id_type;
	typedef int transaction_id_type;
private:
	class junction
	{
	private:
		enum {
			owned_mask = 0x01,
			req_mask = 0x02,
			rel_mask = 0x04
		};
		std::vector<int> flags_;
		std::vector<id_type> owner_, winner_id_;
		packet* pkt_buf_;
		std::vector<port*> port_list_;
		std::vector<port*> async_port_list_;
		transaction_id_type trans_id_, next_trans_id_;
	public:
		junction(void);
		~junction();
		bool check_invariant(void) const;
		size_t port_number(void) const { return port_list_.size(); }
		bool have_packet(void) const { return pkt_buf_ != 0 /* NULL */; }
		inline void put(packet*);
		packet* get(void)
			{ packet* p = pkt_buf_;  pkt_buf_ = 0 /* NULL */;  return p; }
		const packet* look(void) const { return pkt_buf_; }
		void clear(void) { delete pkt_buf_;  pkt_buf_ = 0 /* NULL */; }
		void connect(port&);
		void disconnect(port&);
		void update_list(void);
		bool is_connected(const port&) const;
		void absorb(junction&);
		void change_id(resource_id_type, resource_id_type);
		void set_number_of_arbiter(size_t);
		inline void request_ownership(resource_id_type, resource_id_type);
		inline void release_ownership(resource_id_type);
		inline void update_ownership(void);
		inline bool is_owned(resource_id_type) const;
		inline id_type owner(resource_id_type) const;
		transaction_id_type transaction_id(void) const { return trans_id_; }
		void set_transaction_id(transaction_id_type a) { trans_id_ = a; }
		inline void autoset_transaction_id(void)
		{
			trans_id_ = next_trans_id_;
			next_trans_id_ = (next_trans_id_ < 0x7fff) ?
				next_trans_id_ + 1 : 0;
		}
		void reset_transaction_id(void);
	};
	junction* junction_;
	int id_;
	asynchronous_unit* async_unit_;
	void set_id_inner(int a) { id_ = a; }
	void activate_async_unit(void) {}
public:
	port(void);
	port(const port&);
	port& operator=(const port&);
	int operator==(const port& a) const { return this == &a; }
	int operator!=(const port& a) const { return this != &a; }
	virtual ~port();
	virtual void output(std::ostream&) const;
	virtual bool check_invariant(void) const;
	bool is_connected(void) const
		{ return junction_->port_number() >= 2; }
	bool is_connected(const port& a) const
		{ return junction_->is_connected(a); }
	bool is_connected_to_asynchronous_unit(void) const
		{ return async_unit_ != 0 /* NULL */; }
	bool is_connected_to_asynchronous_unit(const asynchronous_unit& a) const
		{ return async_unit_ == &a; }
	bool have_packet(void) const { return junction_->have_packet(); }
	void clear(void) { junction_->clear(); }
	inline void put(packet*);
	inline packet* get(void);
	const packet* look(void) const { return junction_->look(); }
	void connect(port&);
	void disconnect(port&);
	void disconnect(void);
	void connect_asynchronous_unit(asynchronous_unit&);
	void disconnect_asynchronous_unit(void);
	id_type id(void) const { return id_; }
	void set_id(id_type);
	inline bool is_owned(resource_id_type = 0) const;
	inline bool is_owner(resource_id_type = 0) const;
	inline int owner(resource_id_type = 0) const;
	inline void request_ownership(resource_id_type = 0);
	inline void release_ownership(resource_id_type = 0);
	void set_number_of_arbiter(size_t a)
		{ junction_->set_number_of_arbiter(a); }
	virtual bool compete(id_type, id_type) const;
	transaction_id_type transaction_id(void) const
		{ return junction_->transaction_id(); }
	void set_transaction_id(transaction_id_type a) const
		{ junction_->set_transaction_id(a); }
	friend class junction;
};

inline void port::junction::put(packet* a)
{
	if (have_packet()) delete pkt_buf_;
	pkt_buf_ = a;
	if (a != 0 /* NULL */ && async_port_list_.size() > 0) {
		for (size_t i = 0; i < async_port_list_.size(); i++) {
			async_port_list_[i]->activate_async_unit();
		}
	}
}

inline void port::junction::request_ownership
	(port::resource_id_type a, port::resource_id_type b)
{
#ifdef DEBUG
	assert(a >= resource_id_type(0) && a < resource_id_type(flags_.size()) &&
		   b >= resource_id_type(0) && b < resource_id_type(port_list_.size()));
#endif // DEBUG
	if (flags_[a] & req_mask) {
		if (port_list_[b]->compete(owner_[a], winner_id_[a])) {
			winner_id_[a] = b;
		}
	} else {
		winner_id_[a] = b;
		flags_[a] |= req_mask;
	}
}

inline void port::junction::release_ownership
	(port::resource_id_type a)
{
#ifdef DEBUG
	assert(a >= resource_id_type(0) && a < resource_id_type(flags_.size()));
#endif // DEBUG
	if ((flags_[a] & owned_mask)) flags_[a] |= rel_mask;
}

inline void port::junction::update_ownership(void)
{
	for (size_t i = 0; i < flags_.size(); i++) {
		switch (flags_[i]) {
		case 0:
			break;
		case owned_mask:
			break;
		case req_mask:
			owner_[i] = winner_id_[i];
			flags_[i] = owned_mask;
			autoset_transaction_id();
			break;
		case (req_mask | owned_mask):
			flags_[i] = owned_mask;
			break;
		case rel_mask:
			flags_[i] = 0;
			break;
		case (rel_mask | owned_mask):
			flags_[i] = 0;
			break;
		case (rel_mask | req_mask):
			// internal error of port::junction class is detected
			assert(0);
			break; // not reached
		case (rel_mask | req_mask | owned_mask):
			owner_[i] = winner_id_[i];
			flags_[i] = owned_mask;
			autoset_transaction_id();
			break;
		}
	}
}

inline bool port::junction::is_owned(port::resource_id_type a) const
{
#ifdef DEBUG
	assert(a >= resource_id_type(0) && a < resource_id_type(flags_.size()));
#endif // DEBUG
	((port::junction*)(this))->update_ownership();
	return flags_[a] & owned_mask;
}

inline port::id_type port::junction::owner
	(port::resource_id_type a) const
{
#ifdef DEBUG
	assert(a >= resource_id_type(0) && a < resource_id_type(flags_.size()));
#endif // DEBUG
	((port::junction*)(this))->update_ownership();
	return (flags_[a] & owned_mask) ? owner_[a] : id_type(0);
}

inline void port::put(packet* a)
{
	junction_->put(a);
}

inline packet* port::get(void)
{
	return junction_->get();
}

inline bool port::is_owned(port::resource_id_type a) const
{
	return junction_->is_owned(a);
}

inline bool port::is_owner(port::resource_id_type a) const
{
	return id() == junction_->owner(a);
}

inline port::id_type port::owner(port::resource_id_type a) const
{
	return junction_->owner(a);
}

inline void port::request_ownership(port::resource_id_type a)
{
	junction_->request_ownership(a, id());
}

inline void port::release_ownership(port::resource_id_type a)
{
	junction_->release_ownership(a);
}

#endif /* PORT_H */
