/*
 * <<< port.cc >>>
 *
 * --- I/O port classes 'port'
 *     Copyright (C) 1997-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <algorithm>
#include "port.h"

using namespace std;

port::junction::junction(void)
	: flags_(1), owner_(1), winner_id_(1), pkt_buf_(NULL),
	  trans_id_(0), next_trans_id_(0)
{
	for (size_t i = 0; i < flags_.size(); i++) {
		flags_[i] = owner_[i] = 0;
	}
}

port::junction::~junction()
{
	clear();
}

bool port::junction::check_invariant(void) const
{
	for (size_t i = 0; i < port_list_.size(); i++) {
		if (port_list_[i] == NULL ||
			port_list_[i]->junction_ != this ||
			port_list_[i]->id() != id_type(i))
			return false;
	}
	return true;
}

void port::junction::connect(port& a)
{
	port_list_.push_back(&a);
	a.set_id_inner(port_list_.size() - 1);
	update_list();
}

void port::junction::disconnect(port& a)
{
	size_t i;
	for (i = 0; i < port_list_.size(); i++) {
		if (port_list_[i] == &a) break;
	}
#ifdef DEBUG
	assert(i < port_list_.size());
#endif // DEBUG
	for (; i < port_list_.size() - 1; i++) {
		port_list_[i] = port_list_[i + 1];
		port_list_[i]->set_id_inner(i);
	}
	port_list_.pop_back();
	update_list();
}

bool port::junction::is_connected(const port& a) const
{
	for (size_t i = 0; i < port_list_.size(); i++) {
		if (port_list_[i] == &a) return true;
	}
	return false;
}

void port::junction::update_list(void)
{
	size_t count = 0, i;
	for (i = 0; i < port_list_.size(); i++) {
		if (port_list_[i]->is_connected_to_asynchronous_unit()) count++;
	}
	async_port_list_.resize(count);
	for (i = count = 0; i < port_list_.size(); i++) {
		if (port_list_[i]->is_connected_to_asynchronous_unit()) {
			async_port_list_[count++] = port_list_[i];
		}
	}
	for (i = 0; i < flags_.size(); i++) {
		owner_[i] = id_type(port_list_.size()) - 1;
		flags_[i] = 0;
	}
	reset_transaction_id();
}

void port::junction::absorb(junction& a)
{
	size_t oldsize = port_list_.size();
	port_list_.resize(oldsize + a.port_number());
	for (size_t i = oldsize; i < port_list_.size(); i++) {
		port_list_[i] = a.port_list_[i - oldsize];
		port_list_[i]->set_id_inner(id_type(i));
		port_list_[i]->junction_ = this;
	}
	a.port_list_.resize(0);
	update_list();
	a.update_list();
}

void port::junction::change_id(port::resource_id_type a,
							   port::resource_id_type b)
{
	port* p = port_list_[a];
	port_list_[a] = port_list_[b];
	port_list_[b] = p;
	port_list_[a]->set_id_inner(a);
	port_list_[b]->set_id_inner(b);
}

void port::junction::set_number_of_arbiter(size_t a)
{
#ifdef DEBUG
	assert(a >= 1);
#endif // DEBUG
	flags_.resize(a);
	owner_.resize(a);
	winner_id_.resize(a);
	fill(flags_.begin(), flags_.end(), 0);
	fill(owner_.begin(), owner_.end(), id_type(0));
}

void port::junction::reset_transaction_id(void)
{
	trans_id_ = next_trans_id_ = transaction_id_type(0);
}

port::port(void)
	: junction_(new junction),
	  async_unit_(NULL)
{
	junction_->connect(*this);
}

port::port(const port&)
	: junction_(new junction),
	  async_unit_(NULL)
{
	junction_->connect(*this);
}

port& port::operator=(const port& a)
{
	if (&a != this) {
		disconnect();
	}
	return *this;
}

port::~port()
{
	disconnect();
}

void port::output(ostream& os) const
{
	os << junction_->port_number() << '-' << id();
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

bool port::check_invariant(void) const
{
	return junction_->check_invariant();
}

void port::connect(port& a)
{
	junction* p = a.junction_;
	junction_->absorb(*a.junction_);
	delete p;
}

void port::disconnect(port& a)
{
	if (!junction_->is_connected(a)) return;
	junction_->disconnect(a);
	a.junction_ = new junction;
	a.junction_->connect(a);
}

void port::disconnect(void)
{
	if (junction_->port_number() == 1) return;
	junction_->disconnect(*this);
	junction_ = new junction;
	junction_->connect(*this);
}

void port::connect_asynchronous_unit(asynchronous_unit& a)
{
	async_unit_ = &a;
	junction_->update_list();
}

void port::disconnect_asynchronous_unit(void)
{
	async_unit_ = NULL;
	junction_->update_list();
}

void port::set_id(port::id_type a)
{
	if (a < id_type(0) || a >= id_type(junction_->port_number()) || a == id())
		return;
	junction_->change_id(id(), a);
}

bool port::compete(port::id_type prev_owner, port::id_type rival) const
{
	bool flag;
	if (rival <= prev_owner) {
		flag = id() < rival || prev_owner < id();
	} else {
		// prev_owner < rival
		flag = prev_owner < id() && id() < rival;
	}
	return flag;
}
