/* -*- C++ -*-
 *
 * <<< set_associative_cache.h >>>
 *
 * --- Set-associative cache class 'set_associative_cache'
 *     Copyright (C) 1995-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef SET_ASSOCIATIVE_CACHE_H
#define SET_ASSOCIATIVE_CACHE_H 1

#include <cassert>
#include <vector>
#include <iostream>
#include <iomanip>
#include <isis/cache_line_set.h>

template <
	class line_type
>
class set_associative_cache;

template <class L>
class set_associative_cache
{
private:
	typedef set_associative_cache<L> self_type;
public:
	typedef L line_type;
	typedef cache_line_set<line_type> line_set_type;
	typedef typename line_type::address_type address_type;
	typedef typename line_type::data_type data_type;
	typedef typename line_type::size_type line_size_type;
	typedef typename line_set_type::line_set_size_type line_set_size_type;
private:
	typedef std::vector<line_set_type> container_type;
public:
	typedef typename container_type::size_type size_type;
	//
	set_associative_cache(address_type, address_type, size_type);
	set_associative_cache(const self_type&);
	virtual ~set_associative_cache();
	self_type& operator=(const self_type&);
	void swap(self_type&);
	virtual void output(std::ostream&) const;
	//
	std::size_t way_size() const
		{ return buf_[0].line_set_size(); }
	std::size_t line_size_in_word() const
		{ return buf_[0].line_size(); }
	address_type line_size() const
		{ return line_size_in_word() * sizeof(data_type); }
	address_type cache_size() const
		{ return buf_.size() * line_size() * way_size(); }
	address_type tag_address(address_type adr) const
		{ return adr & tag_mask_; }
	std::size_t line_offset(address_type adr) const
		{ return static_cast<std::size_t>(line_address_(adr)); }
	const line_set_type& cache_line_set(address_type a) const
		{ return buf_[index_address_(a)]; }
	line_set_type& cache_line_set(address_type a)
		{ return buf_[index_address_(a)]; }
	std::size_t max_lru_counter(void)
		{ return static_cast<std::size_t>(buf_[0].max_lru_counter()); }
	void set_max_lru_counter(std::size_t);
	void resize(address_type, address_type, size_type);
	void reset();
private:
	container_type buf_;
	std::size_t upper_shift_, lower_shift_;
	address_type upper_mask_, lower_mask_, tag_mask_;
	template <typename T>
	static std::size_t to_shiftbit(T x) {
		std::size_t i = 0;
		for ( ; x > 1; x >>= 1) i++;
		return i;
	}
	std::size_t index_address_(address_type adr) const {
		return static_cast<std::size_t>((adr & upper_mask_) >> upper_shift_);
	}
	std::size_t line_address_(address_type adr) const {
		return static_cast<std::size_t>((adr & lower_mask_) >> lower_shift_);
	}
	//
	// forbidden
	set_associative_cache();
};

template <class L>
std::ostream& operator<<(std::ostream&, const set_associative_cache<L>&);

template <class L>
set_associative_cache<L>::set_associative_cache(
	typename set_associative_cache<L>::address_type cache_sz,
	typename set_associative_cache<L>::address_type line_sz,
	typename set_associative_cache<L>::size_type way_sz
)
	:
	buf_(
		std::size_t(1)
			<< (to_shiftbit(cache_sz) - to_shiftbit(line_sz)
			    - to_shiftbit(way_sz)),
		line_set_type(
			std::size_t(1)
				<< to_shiftbit(way_sz),
			std::size_t(1)
				<< (to_shiftbit(line_sz) - to_shiftbit(sizeof(data_type)))
		)
	),
	upper_shift_(
		to_shiftbit(line_sz)
	),
	lower_shift_(
		to_shiftbit(sizeof(data_type))
	),
	upper_mask_(
		~(~address_type(0)
			<< (to_shiftbit(cache_sz) - to_shiftbit(way_sz))
		)
	),
	lower_mask_(
		~(~address_type(0)
			<< (upper_shift_ - lower_shift_)) << lower_shift_
	),
	tag_mask_(
		~address_type(0) << upper_shift_
	)
{
	using std::size_t;
	assert(
		cache_sz > 0 && cache_sz == (size_t(1) << to_shiftbit(cache_sz)) &&
		line_sz > 0 && line_sz == (size_t(1) << to_shiftbit(line_sz)) &&
		way_sz > 0 && way_sz == (size_t(1) << to_shiftbit(way_sz)) &&
		sizeof(data_type) == (size_t(1) << to_shiftbit(sizeof(data_type)))
	);
}

template <class L>
set_associative_cache<L>::set_associative_cache(
	const set_associative_cache<L>& a
)
	:
	buf_(a.buf_),
	upper_shift_(a.upper_shift_),
	lower_shift_(a.lower_shift_),
	upper_mask_(a.upper_mask_),
	lower_mask_(a.lower_mask_),
	tag_mask_(a.tag_mask_)
{}

template <class L>
set_associative_cache<L>::~set_associative_cache()
{}

template <class L>
set_associative_cache<L>&
set_associative_cache<L>::operator=(
	const set_associative_cache<L>& a
)
{
	if (this != &a) self_type(a).swap(*this);
	return *this;
}

template <class L>
void set_associative_cache<L>::swap(set_associative_cache<L>& a)
{
	// using std::swap;
	//
	buf_.swap(a.buf_);
	std::swap(upper_shift_, a.upper_shift_); // gcc-2.x require "std::" prefix
	std::swap(lower_shift_, a.lower_shift_);
	std::swap(upper_mask_, a.upper_mask_);
	std::swap(lower_mask_, a.lower_mask_);
	std::swap(tag_mask_, a.tag_mask_);
}

template <class L>
void set_associative_cache<L>::set_max_lru_counter(std::size_t a)
{
#if 0
	// generic code
	using namespace std; // for_each, mem_fun, bind2nd
	for_each(
		buf_.begin(), buf_.end(),
		bind2nd(mem_fun_ref(&line_set_type::set_max_lru_counter), a)
	);
#else
	// code for gcc-2.x
	typedef typename container_type::iterator IT;
	//
	const IT first = buf_.begin(), last = buf_.end();
	for (IT p = first; p != last; ++p) (*p).set_max_lru_counter(a);
#endif
}

template <class L>
void set_associative_cache<L>::resize(
	typename set_associative_cache<L>::address_type a,
	typename set_associative_cache<L>::address_type b,
	typename set_associative_cache<L>::size_type c
)
{
	using std::size_t;
	size_t t = max_lru_counter();
	self_type(a, b, c).swap(*this);
	set_max_lru_counter(t);
}

template <class L>
void set_associative_cache<L>::reset()
{
	using namespace std; // for_each, mem_fun
	for_each(buf_.begin(), buf_.end(), mem_fun_ref(&container_type::reset));
}

template <class L>
void set_associative_cache<L>::output(std::ostream& os) const
{
	using namespace std;
	if (!os) return;
	size_t width;
	{
		size_t tmp = buf_.size() - 1;
		int i = 1;
		while (tmp >= 16) tmp >>= 4, i++;
		width = i;
	}
	const char fill = os.fill();
	const ios::fmtflags flags = os.flags();
	os << hex << setfill('0');
	for (size_t i = 0; i < buf_.size(); i++) {
		for (size_t j = 0; j < buf_[0].line_set_size(); j++) {
			if (j == 0) {
				os << '[' << setw(width) << i << "] ";
			} else {
				for (size_t k = 0; k < width + 3; k++) os << ' ';
			}
			os << buf_[i][j];
			if (buf_[i][j].is_valid()) os << ' ' << buf_[i].lru_counter(j);
			if (j < buf_[0].line_set_size() - 1) os << '\n';
		}
		if (i < buf_.size() - 1) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class L>
std::ostream& operator<<(std::ostream& os, const set_associative_cache<L>& a)
{
	a.output(os);
	return os;
}

#endif // SET_ASSOCIATIVE_CACHE_H
