/*
 * <<< network_packet_receiver.h >>>
 *
 * --- Network packet receiver class 'network_packet_receiver'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef NETWORK_PACKET_RECEIVER_H
#define NETWORK_PACKET_RECEIVER_H

#include <cstddef>
#include <iostream>
#include <isis/virtual_channel.h>
#include <isis/network_packet_base.h>

template <class T>
class network_packet_receiver
{
private:
	typedef network_packet_receiver thisclass;
	typedef T packet_type;
	virtual_channel_input ch_;
	size_t count_, length_;
	packet_type* pkt_buf_;
public:
	inline network_packet_receiver(void);
	network_packet_receiver(const thisclass&);
	~network_packet_receiver();
	void output(std::ostream&) const;
	const virtual_channel_input& channel(void) const { return ch_; }
	virtual_channel_input& channel(void) { return ch_; }
	bool empty(void) const { return count_ == 0; }
	bool full(void) const { return count_ > 0 && count_ == length_; }
	bool snoop(size_t a) const { return count_ >= a; }
	packet_type* look(void) { return pkt_buf_; }
	void get(void) { count_ = 0; }
	inline void clock(void);
};

template <class T>
inline std::ostream& operator<<
	(std::ostream& os, const network_packet_receiver<T>& a)
{
	if (os) a.output(os);
	return os;
}

template <class T>
inline void network_packet_receiver<T>::clock(void)
{
	if (count_ == 0) {
		if (!ch_.empty(0)) {
			pkt_buf_ = (packet_type*)(ch_.get(0));
			length_ = pkt_buf_->length();
			count_ = 1;
		}
	} else if (count_ < length_) {
		if (!ch_.empty(0)) {
			packet* p = ch_.get(0);
			assert(p == pkt_buf_);
			count_++;
		}
	}
}

template <class T>
network_packet_receiver<T>::network_packet_receiver(void)
	: ch_(1, 2),
	  count_(0)
{}

template <class T>
network_packet_receiver<T>::network_packet_receiver(const network_packet_receiver<T>& a)
	: ch_(1, 2),
	  count_(0)
{}

template <class T>
network_packet_receiver<T>::~network_packet_receiver()
{
	if (count_ > 0) delete pkt_buf_;
}

template <class T>
void network_packet_receiver<T>::output(std::ostream& os) const
{
	os << '[';
	if (count_ == 0) {
		// nothing to do
	} else if (count_ < length_) {
		os << *pkt_buf_ << ",receiving:" << count_ << '/' << length_;
	} else {
		os << "pkt:" << *pkt_buf_ << ",received";
	}
	os << ']';
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* NETWORK_PACKET_RECEIVER_H */
