/* -*- C++ -*-
 *
 * <<< directmap_cache.h >>>
 *
 * --- Directmap cache class 'directmap_cache'
 *     Copyright (C) 1995-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef DIRECTMAP_CACHE_H
#define DIRECTMAP_CACHE_H 1

#include <cassert>
#include <algorithm>
#include <vector>
#include <iostream>
#include <iomanip>

template <
	class line_type
>
class directmap_cache;

template <class L>
class directmap_cache
{
private:
	typedef directmap_cache<L> self_type;
public:
	typedef L line_type;
	typedef typename line_type::address_type address_type;
	typedef typename line_type::data_type data_type;
private:
	typedef std::vector<line_type> container_type;
public:
	explicit directmap_cache(address_type, address_type);
	directmap_cache(const self_type&);
	virtual ~directmap_cache();
	self_type& operator=(const self_type&);
	void swap(self_type&);
	virtual void output(std::ostream&) const;
	//
	std::size_t line_size_in_word() const
		{ return buf_[0].size(); }
	address_type line_size() const
		{ return line_size_in_word() * sizeof(data_type); }
	address_type cache_size() const
		{ return buf_.size() * line_size(); }
	address_type tag_address(address_type adr) const
		{ return adr & tag_mask_; }
	bool is_hit(address_type adr) const
		{ return buf_[index_address_(adr)].is_hit(tag_address(adr)); }
	bool is_valid(address_type adr) const
		{ return buf_[index_address_(adr)].is_valid(); }
	address_type read_address(address_type adr) const
		{ return buf_[index_address_(adr)].tag(); }
	data_type read(address_type adr) const
		{ return buf_[index_address_(adr)].data(line_address_(adr)); }
	inline void write(address_type adr, data_type dat) {
		line_type& cl(buf_[index_address_(adr)]);
		address_type a_ = tag_address(adr);
		if (!cl.is_hit(a_)) {
			cl.set_tag(a_);
			cl.set_valid();
		}
		cl.set_data(line_address_(adr), dat);
	}
	void invalidate(address_type adr)
		{ return buf_[index_address_(adr)].set_invalid(); }
	void resize(address_type, address_type);
	void reset();
private:
	container_type buf_;
	size_t upper_shift_, lower_shift_;
	address_type upper_mask_, lower_mask_, tag_mask_;
	template <typename T>
	static std::size_t to_shiftbit(T x) {
		std::size_t i = 0;
		for ( ; x > 1; x >>= 1) i++;
		return i;
	}
	std::size_t index_address_(address_type adr) const {
		return static_cast<std::size_t>((adr & upper_mask_) >> upper_shift_);
	}
	std::size_t line_address_(address_type adr) const {
		return static_cast<std::size_t>((adr & lower_mask_) >> lower_shift_);
	}
	//
	// forbidden
	directmap_cache();
};

template <class L>
std::ostream& operator<<(std::ostream&, const directmap_cache<L>&);

template <class L>
directmap_cache<L>::directmap_cache(
	directmap_cache<L>::address_type cache_sz,
	directmap_cache<L>::address_type line_sz
)
	:
	buf_(
		std::size_t(1) << (to_shiftbit(cache_sz) - to_shiftbit(line_sz)),
		line_type(
			std::size_t(1)
				<< (to_shiftbit(line_sz) - to_shiftbit(sizeof(data_type)))
		)
	),
	upper_shift_(
		to_shiftbit(line_sz)
	),
	lower_shift_(
		to_shiftbit(sizeof(data_type))
	),
	upper_mask_(
		~(~address_type(0) << to_shiftbit(cache_sz))
	),
	lower_mask_(
		~(~address_type(0)
			<< (upper_shift_ - lower_shift_)) << lower_shift_
	),
	tag_mask_(
		~address_type(0) << upper_shift_
	)
{
	using std::size_t;
	assert(
		cache_sz > 0 && cache_sz == (size_t(1) << to_shiftbit(cache_sz)) &&
		line_sz > 0 && line_sz == (size_t(1) << to_shiftbit(line_sz)) &&
		sizeof(data_type) == (size_t(1) << to_shiftbit(sizeof(data_type)))
	);
}

template <class L>
directmap_cache<L>::directmap_cache(const directmap_cache<L>& a)
	:
	buf_(a.buf_),
	upper_shift_(a.upper_shift_),
	lower_shift_(a.lower_shift_),
	upper_mask_(a.upper_mask_),
	lower_mask_(a.lower_mask_),
	tag_mask_(a.tag_mask_)
{}

template <class L>
directmap_cache<L>::~directmap_cache()
{}

template <class L>
directmap_cache<L>& directmap_cache<L>::operator=(const directmap_cache<L>& a)
{
	if (this != &a) self_type(a).swap(*this);
	return *this;
}

template <class L>
void directmap_cache<L>::swap(directmap_cache<L>& a)
{
	// using std::swap;
	//
	buf_.swap(a.buf_);
	std::swap(upper_shift_, a.upper_shift_); // gcc-2.x require "std::" prefix
	std::swap(lower_shift_, a.lower_shift_);
	std::swap(upper_mask_, a.upper_mask_);
	std::swap(lower_mask_, a.lower_mask_);
	std::swap(tag_mask_, a.tag_mask_);
}

template <class L>
void
directmap_cache<L>::resize(
	directmap_cache<L>::address_type a,
	directmap_cache<L>::address_type b
)
{
	using std::size_t;
	self_type(a, b).swap(*this);
}

template <class L>
void directmap_cache<L>::reset()
{
	using namespace std; // for_each, mem_fun
	for_each(buf_.begin(), buf_.end(), mem_fun_ref(&line_type::set_invalid));
}

template <class L>
void directmap_cache<L>::output(std::ostream& os) const
{
	using namespace std;
	if (!os) return;
	size_t width;
	{
		size_t tmp = buf_.size() - 1;
		int i = 1;
		while (tmp >= 16) tmp >>= 4, i++;
		width = i;
	}
	const char fill = os.fill();
	const ios::fmtflags flags = os.flags();
	os << hex << setfill('0');
	for (size_t i = 0; i < buf_.size(); i++) {
		os << '[' << setw(width) << i << "] " << buf_[i];
		if (i < buf_.size() - 1) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

template <class L>
std::ostream& operator<<(std::ostream& os, const directmap_cache<L>& a)
{
	a.output(os);
	return os;
}

#endif // DIRECTMAP_CACHE_H
