/*
 * <<< crossbar_test.cc >>>
 *
 * --- Test program for crossbar class
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstdlib>
#include <deque>
#include <list>
#include <vector>
#include <iostream>
#include <iomanip>
#include "crossbar.h"

using namespace std;

#define rnd() ((unsigned int)((seed = 1566083941UL * seed + 1) >> 16))
static unsigned long seed = 1;

int main(int, char** argv)
{
	// typedefs
	typedef size_t address_type;

	// set default values
	size_t in_size = 4, out_size = 4, sw_wait = 4, iter_count = 100;
	double access_rate = .1;
	bool verbose_flag = false;

	// read command-line option
	while (*++argv != NULL && **argv == '-') {
		switch (*(*argv + 1)) {
		case 'i':
			in_size = (size_t)atoi(*argv + 2);
			break;
		case 'o':
			out_size = (size_t)atoi(*argv + 2);
			break;
		case 'r':
			access_rate = (double)atof(*argv + 2);
			break;
		case 'n':
			iter_count = (size_t)atoi(*argv + 2);
			break;
		case 'w':
			sw_wait = (size_t)atoi(*argv + 2);
			break;
		case 'v':
			verbose_flag = true;
			break;
		default:
			break;
		}
	}

	// create crossbar, injection queue, lock-counters, diag-counters
	crossbar sw(in_size, out_size);
	vector< list<address_type> > inj_que(in_size);
	vector<size_t> lock_tbl(in_size);
	size_t round_master = 0;
	size_t sent_count = 0, received_count = 0;
	{
		for (size_t i = 0; i < lock_tbl.size(); i++) lock_tbl[i] = 0;
	}

	// show parameters
	cout << "--- test condition ---" << endl
		 << "input channels:  " << in_size << endl
		 << "output channels: " << out_size << endl
		 << "access wait:     " << sw_wait << endl
		 << "access rate:     " << access_rate << endl
		 << "iterations:      " << iter_count << endl << endl;

	// main loop
	for (size_t iter = 0; iter < iter_count; iter++) {
		size_t i;
		// generate packet and inject to inj_que
		for (i = 0; i < in_size; i++) {
			if ((double(rnd() & 0x7fff) / 0x8000) < access_rate) {
				inj_que[i].push_back(rnd() % out_size);
				sent_count++;
			}
		}
		// count-down lock_tbl
		for (i = 0; i < in_size; i++) {
			if (lock_tbl[i] > 0) {
				if (--lock_tbl[i] == 0) {
					sw.disconnect_input_channel(i);
					inj_que[i].pop_front();
					received_count++;
				}
			}
		}
		// transmit packet inj_que -> sw
		for (i = 0; i < in_size; i++) {
			size_t id = (round_master + i) % in_size;
			if (inj_que[id].empty()) continue;
			address_type dst = inj_que[id].front();
			if (lock_tbl[id] == 0 && !sw.is_connected_output_channel(dst)) {
				sw.connect_crosspoint(id, dst);
				lock_tbl[id] = sw_wait;
				if (i == 0) round_master = (round_master + 1) % in_size;
			}
		}
		// show current status if verbose
		if (verbose_flag) {
			cout << "--- clk:" << iter << " ---" << endl;
			cout << "round_master: " << round_master << endl;
			cout << "inj_que:" << endl;
			for (i = 0; i < in_size; i++) {
				list<address_type>::const_iterator p = inj_que[i].begin();
				deque<address_type> tmp;
				while (p != inj_que[i].end()) {
					tmp.push_front(*p);
					p++;
				}
				cout << ' ' << setw(2) << i << ":(";
				while (!tmp.empty()) {
					cout << tmp.front();
					tmp.pop_front();
					if (!tmp.empty()) cout << ',';
				}
				cout << ')' << endl;
			}
			cout << "lock_tbl:" << endl << " [";
			for (i = 0; i < in_size; i++) {
				cout << lock_tbl[i];
				if (i < in_size - 1) cout << ' ';
			}
			cout << ']' << endl;
			cout << endl;
		}
	}

	// show results
	cout << "--- results ---" << endl
		 << "sent:       " << setw(8) << sent_count << endl
		 << "received:   " << setw(8) << received_count << endl
		 << "throughput: " << setw(8)
		 << (double(received_count) / iter_count) << endl;

	// exit
	return 0;
}
