/*
 * <<< crossbar.h >>>
 *
 * --- Crossbar class 'crossbar'
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef CROSSBAR_H
#define CROSSBAR_H

#include <vector>

class crossbar
{
private:
	typedef crossbar thisclass;
	typedef std::vector<size_t> size_container_type;
	typedef std::vector<size_t> index_container_type;
	typedef std::vector<bool> flag_container_type;
	size_t in_size_, out_size_;
	size_container_type i2o_size_;
	flag_container_type o2i_flag_;
	index_container_type i2o_tbl_;
	index_container_type o2i_tbl_;
public:
	crossbar(void);
	crossbar(size_t, size_t);
	crossbar(const thisclass&);
	~crossbar() {}
	size_t input_size(void) const { return in_size_; }
	size_t output_size(void) const { return out_size_; }
	void resize(size_t, size_t);
	void clear(void);
	bool is_connected_input_channel(size_t i) { return i2o_size_[i] > 0; }
	bool is_connected_output_channel(size_t i) { return o2i_flag_[i]; }
	inline void connect_crosspoint(size_t, size_t);
	inline void disconnect_crosspoint(size_t, size_t);
	inline void disconnect_input_channel(size_t);
	inline void disconnect_output_channel(size_t);
	size_t input_to_output_size(size_t i) { return i2o_size_[i]; }
	size_t input_to_output_index(size_t i, size_t j = 0)
		{ return i2o_tbl_[i * out_size_ + j]; }
	size_t output_to_input_index(size_t i)
		{ return o2i_tbl_[i]; }
};

inline void crossbar::connect_crosspoint(size_t i, size_t j)
{
	if (!o2i_flag_[j]) {
		i2o_tbl_[i * out_size_ + i2o_size_[i]] = j;
		i2o_size_[i]++;
		o2i_flag_[j] = true;
		o2i_tbl_[j] = i;
	}
}

inline void crossbar::disconnect_crosspoint(size_t i, size_t j)
{
	if (o2i_flag_[j] && o2i_tbl_[j] == i) {
		size_t k = 0;
		if (i2o_size_[i] == 1) {
			i2o_size_[i] = 0;
		} else {
			while (i2o_tbl_[i * out_size_ + k] != j) k++;
			while (k < i2o_size_[i] - 1) {
				i2o_tbl_[i * out_size_ + k] = i2o_tbl_[i * out_size_ + k + 1];
			}
			i2o_size_[i]--;
		}
		o2i_flag_[j] = false;
	}
}

inline void crossbar::disconnect_input_channel(size_t i)
{
	if (i2o_size_[i] > 0) {
		for (size_t j = 0; j < i2o_size_[i]; j++) {
			o2i_flag_[i2o_tbl_[i * out_size_ + j]] = false;
		}
		i2o_size_[i] = 0;
	}
}

inline void crossbar::disconnect_output_channel(size_t j)
{
	if (o2i_flag_[j]) {
		size_t i = o2i_tbl_[j];
		size_t k = 0;
		if (i2o_size_[i] == 1) {
			i2o_size_[i] = 0;
		} else {
			while (i2o_tbl_[i * out_size_ + k] != j) k++;
			while (k < i2o_size_[i] - 1) {
				i2o_tbl_[i * out_size_ + k] = i2o_tbl_[i * out_size_ + k + 1];
			}
			i2o_size_[i]--;
		}
		o2i_flag_[j] = false;
	}
}

#endif /* CROSSBAR_H */
