/* -*- C++ -*-
 *
 * <<< cache_line_set.h >>>
 *
 * --- Cache line set class 'cache_line_set'
 *     Copyright (C) 1995-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef CACHE_LINE_SET_H
#define CACHE_LINE_SET_H 1

#include <cstddef>
#include <algorithm>
#include <vector>
#include <iostream>
#include "limited_counter.h"

template <
	class line_type,
	class container
		= std::vector<line_type>
>
class cache_line_set;

template <class L, class C>
class cache_line_set
{
private:
	typedef cache_line_set<L, C> self_type;
public:
	typedef L line_type;
	typedef C container_type;
	typedef typename line_type::address_type address_type;
	typedef typename line_type::data_type data_type;
	typedef limited_counter<std::size_t> counter_type;
	typedef std::vector<counter_type> counter_container_type;
	typedef typename container_type::size_type line_set_size_type;
	typedef typename line_type::size_type line_size_type;
public:
	cache_line_set(line_set_size_type, line_size_type);
	virtual ~cache_line_set();
	void swap(self_type&);
	self_type& operator=(const self_type&);
	virtual void output(std::ostream&) const;
	//
	const line_type& operator[](std::size_t i) const { return buf_[i]; }
	line_type& operator[](std::size_t i) { return buf_[i]; }
	line_set_size_type line_set_size() const { return buf_.size(); }
	line_size_type line_size() const { return buf_[0].size(); }
	void resize(line_set_size_type, line_size_type);
	int way_of_hit(address_type a) const
	{
		for (line_set_size_type i = 0; i < buf_.size(); i++) {
			if (buf_[i].is_hit(a)) return static_cast<int>(i);
		}
		return -1;
	}
	int way_of_throw() const
	{
		counter_type::value_type tmp = lru_[0];
		line_set_size_type idx = 0;
		for (line_set_size_type i = 1; tmp > 0 && i < lru_.size(); i++) {
			counter_type::value_type new_val = lru_[i];
			if (new_val < tmp) tmp = new_val, idx = i;
		}
		return static_cast<int>(idx);
	}
	int way_of_empty() const
	{
		for (line_set_size_type i = 0; i < buf_.size(); i++) {
			if (!buf_[i].is_valid()) return static_cast<int>(i);
		}
		return -1;
	}
	bool is_hit(address_type a) const
	{
		return way_of_hit(a) >= 0;
	}
	bool is_full() const
	{
		return !(way_of_empty() >= 0);
	}
	const counter_type& lru_counter(line_set_size_type a) const
		{ return lru_[a]; }
	void increment_lru_counter(line_set_size_type a)
		{ lru_[a].increment(); }
	void clear_lru_counter() {
		using namespace std; // for_each, mem_fun_ref
		for_each(lru_.begin(), lru_.end(), mem_fun_ref(&counter_type::clear));
	}
	void clear_lru_counter(line_set_size_type a)
		{ lru_[a].clear(); }
	counter_type::value_type max_lru_counter()
		{ return lru_[0].max(); }
	void set_max_lru_counter(counter_type::value_type);
	void reset();
private:
	static counter_type::value_type default_max_lru_counter_()
		{ return 1; }
	container_type buf_;
	counter_container_type lru_;
	//
	// forbidden
	cache_line_set();
};

template <class L, class C>
inline std::ostream& operator<<(std::ostream& os, const cache_line_set<L, C>& a)
{
	a.output(os);
	return os;
}

//

template <class L, class C>
cache_line_set<L, C>::cache_line_set(
	typename cache_line_set<L, C>::line_set_size_type a,
	typename cache_line_set<L, C>::line_size_type b
)
	:
	buf_(a, line_type(b)),
	lru_(a, default_max_lru_counter_())
{}

template <class L, class C>
cache_line_set<L, C>::~cache_line_set()
{}

template <class L, class C>
void cache_line_set<L, C>::swap(cache_line_set<L, C>& a)
{
	buf_.swap(a.buf_);
	lru_.swap(a.lru_);
}

template <class L, class C>
void cache_line_set<L, C>::resize(
	typename cache_line_set<L, C>::line_set_size_type a,
	typename cache_line_set<L, C>::line_size_type b
)
{
	const counter_type::value_type max_lru_counter_ = max_lru_counter();
	self_type(a, b).swap(*this);
	set_max_lru_counter(max_lru_counter_);
}

template <class L, class C>
void cache_line_set<L, C>::set_max_lru_counter
	(cache_line_set<L, C>::counter_type::value_type a)
{
#if 0
	// generic code
	using namespace std; // for_each, mem_fun_ref, bind2nd
	//
	for_each(
		lru_.begin(), lru_.end(),
		bind2nd(mem_fun_ref(&counter_type::set_max), a)
	);
#else
	// code for gcc-2.x
	typedef typename counter_container_type::iterator IT;
	//
	const IT first = lru_.begin(), last = lru_.end();
	for (IT p = first; p != last; ++p) (*p).set_max(a);
#endif
}

template <class L, class C>
void cache_line_set<L, C>::reset()
{
	using namespace std; // for_each, mem_fun_ref
	//
	for_each(
		buf_.begin(), buf_.end(),
		mem_fun_ref(&container_type::set_invalid)
	);
	for_each(
		lru_.begin(), lru_.end(),
		mem_fun_ref(&counter_container_type::reset)
	);
}

template <class L, class C>
void cache_line_set<L, C>::output(std::ostream& os) const
{
	using namespace std;
	//
	if (!os) return;
	const char fill = os.fill();
	const ios::fmtflags flags = os.flags();
	os << hex << setfill('0');
	for (line_set_size_type i = 0; i < buf_.size(); i++) {
		os << buf_[i];
		if (buf_[i].is_valid()) os << ' ' << lru_counter(i);
		if (i < buf_.size() - 1) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // CACHE_LINE_SET_H
