/* -*- C++ -*-
 *
 * <<< cache_line_base.h >>>
 *
 * --- Cache line base class 'cache_line_base'
 *     Copyright (C) 1995-2003 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef CACHE_LINE_BASE_H
#define CACHE_LINE_BASE_H 1

#include <cassert>
#include <algorithm>
#include <vector>
#include <iosfwd>

template <
	typename address_type,
	typename data_type
		= address_type,
	class container_type
		= std::vector<data_type>
>
class cache_line_base;

template <typename A, typename D, class C>
class cache_line_base
{
private:
	typedef cache_line_base<A, D, C> self_type;
	typedef C container_type;
public:
	typedef A address_type;
	typedef D data_type;
	typedef typename container_type::size_type size_type;
public:
	explicit cache_line_base(size_type, address_type = address_type(0),
	                         data_type = data_type(0));
	cache_line_base(const self_type&);
	virtual ~cache_line_base();
	void swap(self_type&);
	self_type& operator=(const self_type&);
	virtual void output(std::ostream&) const {}
	size_type size() const
		{ return buf_.size(); }
	void resize(size_type);
	data_type data(size_t i) const
		{ return buf_[i]; }
	address_type tag() const
		{ return tag_; }
	void set_data(size_t i, data_type x)
		{ buf_[i] = x; }
	void set_tag(address_type x)
		{ tag_ = x; }
	virtual bool is_valid() const = 0;
	virtual bool is_hit(address_type x) const
		{ return x == tag_ && is_valid(); }
	// obsolete interfaces: DO NOT use them
#if 0
	data_type operator[](std::size_t i) const
		{ return buf_[i]; }
	data_type& operator[](std::size_t i)
		{ return buf_[i]; }
	address_type& tag()
		{ return tag_; }
#endif
private:
	container_type buf_;
	address_type tag_;
	//
	// forbidden
	cache_line_base();
};

template <typename A, typename D, class C>
std::ostream& operator<<(std::ostream&, const cache_line_base<A, D, C>&);

template <typename A, typename D, class C>
cache_line_base<A, D, C>::cache_line_base(
	typename cache_line_base<A, D, C>::size_type n,
	typename cache_line_base<A, D, C>::address_type adr,
	typename cache_line_base<A, D, C>::data_type dat
)
	:
	buf_(n, dat),
	tag_(adr)
{
	assert(n > 0);
}

template <typename A, typename D, class C>
cache_line_base<A, D, C>::cache_line_base(const cache_line_base<A, D, C>& a)
	:
	buf_(a.buf_),
	tag_(a.tag_)
{}

template <typename A, typename D, class C>
cache_line_base<A, D, C>::~cache_line_base()
{}

template <typename A, typename D, class C>
void cache_line_base<A, D, C>::swap(cache_line_base<A, D, C>& a)
{
	using std::swap;
	buf_.swap(a.buf_);
	swap(tag_, a.tag_);
}

template <typename A, typename D, class C>
cache_line_base<A, D, C>& cache_line_base<A, D, C>::operator=(
	const cache_line_base<A, D, C>& a
)
{
	if (this != &a) self_type(a).swap(*this);
	return *this;
}

template <typename A, typename D, class C>
void cache_line_base<A, D, C>::resize(
	typename cache_line_base<A, D, C>::size_type n
)
{
	assert(n > 0);
	buf_.resize(n, data_type(0));
}

template <typename A, typename D, class C>
std::ostream& operator<<(std::ostream& os, const cache_line_base<A, D, C>& a)
{
	a.output(os);
	return os;
}

#endif // CACHE_LINE_BASE_H
