/*
 * <<< arbiter.cc >>>
 *
 * --- Arbiter class 'arbiter'
 *     Copyright (C) 1998-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifdef DEBUG
# include <cassert>
#endif // DEBUG
#include <algorithm>
#include <iostream>
#include "arbiter.h"

using namespace std;

arbiter::arbiter(arbiter::size_type a, arbiter::size_type b)
	: user_(a), req_queue_(b), can_queue_(b), rel_queue_(b)
{
	clear();
}

arbiter::~arbiter()
{}

void arbiter::set_resource_count(arbiter::size_type a)
{
	user_.resize(a);
	clear();
}

void arbiter::set_user_count(arbiter::size_type a)
{
	req_queue_.reserve(a);
	can_queue_.reserve(a);
	rel_queue_.reserve(a);
	clear();
}

void arbiter::clear(void)
{
	fill(user_.begin(), user_.end(), id_type(-1));
	req_queue_.clear();
	can_queue_.clear();
	rel_queue_.clear();
	prev_winner_ = id_type(user_count()) - 1;
}

void arbiter::update(void)
{
	// process release queue
	while (!rel_queue_.empty()) {
		id_type id = rel_queue_.front();
		rel_queue_.pop();
#ifdef DEBUG
		assert(id >= 0 && id < id_type(user_count()));
#endif // DEBUG
		for (size_type i = 0; i < resource_count(); i++) {
			if (user_[i] == id) {
				user_[i] = id_type(-1);
				break;
			}
		}
	}
	// process cancel queue
	while (!can_queue_.empty()) {
		id_type id = can_queue_.front();
		can_queue_.pop();
#ifdef DEBUG
		assert(id >= 0 && id < id_type(user_count()));
#endif // DEBUG
		size_type loop_count = req_queue_.size();
		for (size_type i = 0; i < loop_count; i++) {
			id_type me = req_queue_.front();
			req_queue_.pop();
			if (me != id) req_queue_.push(me);
		}
	}
	// count number of free resource
	size_type free_resource = 0;
	for (size_type i = 0; i < resource_count(); i++) {
		if (is_free(i)) free_resource++;
	}
	// process request queue
	if (req_queue_.size() <= free_resource) {
		// no conflict
		for (size_type i = 0;
			 i < resource_count() && !req_queue_.empty();
			 i++) {
			if (is_free(i)) {
				user_[i] = req_queue_.front();
				req_queue_.pop();
			}
		}
	} else {
		// conflict
		size_type loser_num = req_queue_.size() - free_resource;
		cyclic_queue<id_type> loser_queue_(loser_num);
		id_type grand_winner = -1;
		size_type i;
		for (i = 0; i < loser_num; i++) {
			id_type loser = req_queue_.front();
			req_queue_.pop();
			size_type loop_count = req_queue_.size();
			for (size_type j = 0; j < loop_count; j++) {
				id_type me = req_queue_.front();
				req_queue_.pop();
				id_type winner = compete(prev_winner_, loser, me);
				req_queue_.push(winner);
				if (me != winner) loser = me;
				if (grand_winner < 0) {
					grand_winner = winner;
				} else {
					grand_winner = compete(prev_winner_, grand_winner, winner);
				}
			}
			loser_queue_.push(loser);
		}
		prev_winner_ = grand_winner;
		for (i = 0; i < resource_count() && !req_queue_.empty(); i++) {
			if (is_free(i)) {
				user_[i] = req_queue_.front();
				req_queue_.pop();
			}
		}
		// restore queue
		for (i = 0; i < loser_num; i++) {
			req_queue_.push(loser_queue_.front());
			loser_queue_.pop();
		}
	}
}

int arbiter::compete(int prev_owner, int rival, int me) const
{
	bool flag;
	if (rival <= prev_owner) {
		flag = (me < rival || prev_owner < me);
	} else {
		// prev_owner < rival
		flag = (prev_owner < me && me < rival);
	}
	return flag ? me : rival;
}

void arbiter::output(ostream& os) const
{
	os << "res:(";
	for (size_type i = 0; i < resource_count(); i++) {
		if (is_free(i)) {
			os << '-';
		} else {
			os << user(i);
		}
		if (i < resource_count() - 1) os << ' ';
	}
	os << ") req:(";
	{
		cyclic_queue<id_type> tmp_queue(req_queue_);
		while (!tmp_queue.empty()) {
			os << tmp_queue.front();
			tmp_queue.pop();
			if (tmp_queue.empty()) break;
			os << ' ';
		}
	}
	os << ") can:(";
	{
		cyclic_queue<id_type> tmp_queue(can_queue_);
		while (!tmp_queue.empty()) {
			os << tmp_queue.front();
			tmp_queue.pop();
			if (tmp_queue.empty()) break;
			os << ' ';
		}
	}
	os << ") rel:(";
	{
		cyclic_queue<id_type> tmp_queue(rel_queue_);
		while (!tmp_queue.empty()) {
			os << tmp_queue.front();
			tmp_queue.pop();
			if (tmp_queue.empty()) break;
			os << ' ';
		}
	}
	os << ')';
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}
