/*
 * <<< torus_3d_escape_router.h >>>
 *
 * --- Router class for 3D-torus, with duato's escape path routing
 *     Copyright (C) 2000-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TORUS_3D_ESCAPE_ROUTER_H
#define TORUS_3D_ESCAPE_ROUTER_H

#include <isis/isis.h>

template <class P>
class torus_3d_escape_router : public deterministic_router<P>
{
private:
	typedef torus_3d_escape_router<P> thisclass;
	typedef deterministic_router<P> inherited;
public:
	typedef inherited::packet_type packet_type;
	typedef inherited::node_address_type node_address_type;
private:
	const static size_t physical_channel_size = 7;
	const static size_t virtual_channel_size = 2;
	const static size_t PLUS_X_PH_CHANNEL  = 0;
	const static size_t MINUS_X_PH_CHANNEL = 1;
	const static size_t PLUS_Y_PH_CHANNEL  = 2;
	const static size_t MINUS_Y_PH_CHANNEL = 3;
	const static size_t PLUS_Z_PH_CHANNEL  = 4;
	const static size_t MINUS_Z_PH_CHANNEL = 5;
	const static size_t COMSUMP_PH_CHANNEL = 6;
	const static size_t CH_VR_CHANNEL = 0;
	const static size_t CA_VR_CHANNEL = 1;
	enum state_type { READY, BUSY };
	size_t x_size_, y_size_, z_size_;
protected:
	virtual void routing(const packet_type&, channel_controller&);
public:
	torus_3d_escape_router(void);
	torus_3d_escape_router(const torus_3d_escape_router&);
	virtual ~torus_3d_escape_router() {}
	void set_size(size_t a, size_t b, size_t c)
		{ x_size_ = a, y_size_ = b, z_size_ = c; }
};

template <class P>
torus_3d_escape_router<P>::torus_3d_escape_router(void)
	: inherited(),
	  x_size_(0),
	  y_size_(0),
	  z_size_(0)
{
	set_input_size(physical_channel_size);
	set_output_size(physical_channel_size);
	for (size_t i = 0; i < physical_channel_size - 1; i++) {
		set_channel_size(i, virtual_channel_size);
		set_buffer_size(i, 1);
	}
	set_channel_size(COMSUMP_PH_CHANNEL, 1);
	set_buffer_size(COMSUMP_PH_CHANNEL, 1);
}

template <class P>
torus_3d_escape_router<P>::torus_3d_escape_router
	(const torus_3d_escape_router<P>& a)
	: inherited(a),
	  x_size_(a.x_size_),
	  y_size_(a.y_size_),
	  z_size_(a.z_size_)
{}

template <class P>
void torus_3d_escape_router<P>::routing
	(const torus_3d_escape_router<P>::packet_type& pkt,
	 channel_controller& ctl)
{
	if (pkt.destination() == node_address()) {
		// destination is here, transmit to PE
		ctl.set_destination(COMSUMP_PH_CHANNEL, 0);
		return;
	}
	// destination is not here, select channel with duato routing
	size_t my_x, my_y, my_z, src_x, src_y, src_z, dst_x, dst_y, dst_z,
		   next_phch, next_vrch;
	size_t x_distance, y_distance, z_distance;
	size_t ctl_phid, dirx_phch, dirx_esc_vrch, diry_phch, diry_esc_vrch,
		   dirz_phch, dirz_esc_vrch;
	my_x = node_address() % x_size_;
	my_y = (node_address() / x_size_) % y_size_;
	my_z = (node_address() / x_size_) / y_size_;
	src_x = pkt.source() % x_size_;
	src_y = (pkt.source() / x_size_) % y_size_;
	src_z = (pkt.source() / x_size_) / y_size_;
	dst_x = pkt.destination() % x_size_;
	dst_y = (pkt.destination() / x_size_) % y_size_;
	dst_z = (pkt.destination() / x_size_) / y_size_;
	ctl_phid = ctl.physical_channel_id();
	// decide x direction
	if (my_x != dst_x) {
		if (src_x < dst_x) {
			if (dst_x - src_x <= src_x + x_size_ - dst_x) {
				dirx_phch = PLUS_X_PH_CHANNEL;
				dirx_esc_vrch = CH_VR_CHANNEL;
				x_distance = dst_x - src_x;
			} else {
				dirx_phch = MINUS_X_PH_CHANNEL;
				dirx_esc_vrch = CA_VR_CHANNEL;
				x_distance = src_x + x_size_ - dst_x;
			}
		} else {
			if (dst_x + x_size_ - src_x <= src_x - dst_x) {
				dirx_phch = PLUS_X_PH_CHANNEL;
				dirx_esc_vrch = CA_VR_CHANNEL;
				x_distance = dst_x + x_size_ - src_x;
			} else {
				dirx_phch = MINUS_X_PH_CHANNEL;
				dirx_esc_vrch = CH_VR_CHANNEL;
				x_distance = src_x - dst_x;
			}
		}
	} else {
		x_distance = 0;
	}
	// decide y direction
	if (my_y != dst_y) {
		if (src_y < dst_y) {
			if (dst_y - src_y <= src_y + y_size_ - dst_y) {
				diry_phch = PLUS_Y_PH_CHANNEL;
				diry_esc_vrch = CH_VR_CHANNEL;
				y_distance = dst_y - src_y;
			} else {
				diry_phch = MINUS_Y_PH_CHANNEL;
				diry_esc_vrch = CA_VR_CHANNEL;
				y_distance = src_y + y_size_ - dst_y;
			}
		} else {
			if (dst_y + y_size_ - src_y <= src_y - dst_y) {
				diry_phch = PLUS_Y_PH_CHANNEL;
				diry_esc_vrch = CA_VR_CHANNEL;
				y_distance = dst_y + y_size_ - src_y;
			} else {
				diry_phch = MINUS_Y_PH_CHANNEL;
				diry_esc_vrch = CH_VR_CHANNEL;
				y_distance = src_y - dst_y;
			}
		}
	} else {
		y_distance = 0;
	}
	// decide z direction
	if (my_z != dst_z) {
		if (src_z < dst_z) {
			if (dst_z - src_z <= src_z + z_size_ - dst_z) {
				dirz_phch = PLUS_Z_PH_CHANNEL;
				dirz_esc_vrch = CH_VR_CHANNEL;
				z_distance = dst_z - src_z;
			} else {
				dirz_phch = MINUS_Z_PH_CHANNEL;
				dirz_esc_vrch = CA_VR_CHANNEL;
				z_distance = src_z + z_size_ - dst_z;
			}
		} else {
			if (dst_z + z_size_ - src_z <= src_z - dst_z) {
				dirz_phch = PLUS_Z_PH_CHANNEL;
				dirz_esc_vrch = CA_VR_CHANNEL;
				z_distance = dst_z + z_size_ - src_z;
			} else {
				dirz_phch = MINUS_Z_PH_CHANNEL;
				dirz_esc_vrch = CH_VR_CHANNEL;
				z_distance = src_z - dst_z;
			}
		}
	} else {
		z_distance = 0;
	}
	// select CA,CH channel (escape path)
	if (my_x != dst_x) {
		ctl.set_destination(dirx_phch, dirx_esc_vrch);
	} else if (my_y != dst_y) {
		ctl.set_destination(diry_phch, diry_esc_vrch);
	} else {
		ctl.set_destination(dirz_phch, dirz_esc_vrch);
	}
}

#endif /* TORUS_3D_ESCAPE_ROUTER_H */
