/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2001 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< cache_controller_shared.h >>>
 * shared bus side cache controller
 */

#ifndef CACHE_CONTROLLER_SHARED_H
#define CACHE_CONTROLLER_SHARED_H

#include <iostream>
#include <isis/isis.h>
#include "cache_protocol.h"
#include "bus_port_shared.h"
#include "big_word.h"

template <class Type>
class cache_controller_shared : public root_object
{
private:
	typedef synchronous_unit inherited;
	typedef cache_line_PROTOCOL<Type, Type> cache_line_t;
	typedef cache_line_set<cache_line_t> cache_line_set_t;
	typedef set_associative_cache<cache_line_t> cache_t;
	typedef big_word<Type> bus_data_t;
	typedef bus_port_shared<bus_data_t> shared_bus_port_t;

	// shared bus snoop state machine
	enum bus_snoop_state_t {
		IDLE,
		LINE_COPY_ACK_SEND,
		LINE_COPY_DECIDE_SENDER,
		LINE_COPY_CANCELED,
		LINE_COPY_TO_MEMORY_SEND,
		LINE_COPY_TO_MEMORY_WAIT,
		LINE_COPY_TRANS,
		LINE_COPY_TRANS_END,
		WAIT
	} bs_state;
		
	// processor side bus transaction state machine
	enum bus_master_state_t {
		READY,
		READ_LINE_COPY_REQ_OWNER,
		READ_LINE_COPY_SEND,
		READ_LINE_COPY_WAIT,
		READ_LINE_REL_OWNER,
		READ_FROM_CACHE_TRANS,
		READ_FROM_MEMORY_SEND,
		READ_FROM_MEMORY_WAIT,
		READ_FROM_MEMORY_TRANS,
		REPLACE_REQ_OWNER,
		REPLACE_SEND,
		REPLACE_WAIT,
		REPLACE_TRANS,
		INV_REQ_OWNER,
		INV_SEND,
		INV_REL_OWNER
	} bm_state;

	enum shared_access_t {
		READ_ACCESS,
		WRITE_ACCESS,
		SYNC_ACCESS,
		LINE_COPY_ACCESS,
		LINE_MOVE_ACCESS
	};
	shared_bus_port_t s_bus;
	cache_t* cache_buf;
	shared_access_t shared_access_type;

	// local controllerȤζĴư
	int local_read_wait_flag, local_sync_wait_flag, local_write_wait_flag,
		shared_access_wait_flag;
	Type local_read_adr_, local_read_data_;
	Type local_sync_adr_, local_sync_data_;
	Type local_write_adr_, local_write_data_;

	// illinois protocolCSʥå饤žꤹ
	arbiter* sender;
	int id_;

	// ¾
	Type access_adr;
	bus_data_t buf_adr, buf_rep, buf_dat;
	int replace_flag;
	cache_line_set_t* access_line_set;
	cache_line_t* access_line;
	int access_way;
	int wait_count;
	void case_of_cache_miss();
	
public:
	cache_controller_shared();
	cache_controller_shared(const cache_controller_shared<Type>&);
	cache_controller_shared<Type>& operator=
		(const cache_controller_shared<Type>&);
	virtual ~cache_controller_shared() {}
	virtual void output(ostream&) const;
	void bus_snoop_in();
	void bus_snoop_out();
	void bus_master_in();
	void bus_master_out();
	void reset();
	const port& port_ref() const {
		return s_bus; }
	port& port_ref() {
		return s_bus; }
	int id(void) const {
		return id_; }
	int id(int a) {
		return id_ = a; }
	// bus
	void connect(bus_port<bus_data_t>& a) {
		s_bus.connect(a); }
	void disconnect() {
		s_bus.disconnect(); }
	void disconnect(bus_port<bus_data_t>& a) {
		s_bus.disconnect(a); }
	int is_connected() const {
		return s_bus.is_connected(); }
	int is_connected(const bus_port<bus_data_t>& a) const {
		return s_bus.is_connected(a); }
	// cache buffer
	void connect_cache(cache_t& a) {
		cache_buf = &a; }
	void disconnect_cache() {
		cache_buf = NULL; }
	int is_connected_to_cache() const {
		return cache_buf != NULL; }
	// arbiter
	void connect_sender_arb(arbiter& a) {
		sender = &a; }
	void disconnect_sender_arb() {
		sender = NULL; }
	int is_connected_sender_arb() const {
		return sender != NULL; }
	// interfaces for local bus controller
	void send_local_read_request(Type a) {
		local_read_wait_flag = 1;
		local_read_adr_ = a; }
	void send_local_sync_request(Type a) {
		local_sync_wait_flag = 1;
		local_sync_adr_ = a; }
	void send_local_write_request(Type a, Type b) {
		local_write_wait_flag = 1;
		local_write_adr_ = a;
		local_write_data_ = b; }
	//
	int local_read_is_finished() const {
		return !local_read_wait_flag; }
	int local_sync_is_finished() const {
		return !local_sync_wait_flag; }
	int local_write_is_finished() const {
		return !local_write_wait_flag; }
	int shared_access_is_finished() const {
		return !shared_access_wait_flag; }
	int cache_line_is_busy(Type a) const {
		return (shared_access_wait_flag && access_adr == a); }
	//
	Type local_read_address() const {
		return local_read_adr_; }
	Type local_read_data() const {
		return local_read_data_; }
	Type local_sync_address() const {
		return local_sync_adr_; }
	Type local_sync_data() const {
		return local_sync_data_; }
	Type local_write_address() const {
		return local_write_adr_; }
	Type local_write_data() const {
		return local_write_data_; }
	Type shared_access_address() const {
		return access_adr; }
};

template <class Type>
cache_controller_shared<Type>::cache_controller_shared()
	: bs_state(IDLE),
	  bm_state(READY),
	  cache_buf(NULL),
	  local_read_wait_flag(0),
	  local_sync_wait_flag(0),
	  local_write_wait_flag(0),
	  shared_access_wait_flag(0),
	  sender(NULL),
	  id_(0)
{
}

template <class Type>
cache_controller_shared<Type>::
cache_controller_shared(const cache_controller_shared<Type>&)
	: bs_state(IDLE),
	  bm_state(READY),
	  cache_buf(NULL),
	  local_read_wait_flag(0),
	  local_sync_wait_flag(0),
	  local_write_wait_flag(0),
	  shared_access_wait_flag(0),
	  sender(NULL),
	  id_(0)
{
}

template <class Type>
cache_controller_shared<Type>& cache_controller_shared<Type>::
operator=(const cache_controller_shared<Type>& a)
{
	if (this != &a) {
		reset();
	}
	return *this;
}

template <class Type>
void cache_controller_shared<Type>::reset()
{
	bs_state = IDLE;
	bm_state = READY;
	local_read_wait_flag = local_sync_wait_flag =
	local_write_wait_flag = shared_access_wait_flag = 0;
}

template <class Type>
void cache_controller_shared<Type>::case_of_cache_miss()
{
	cache_line_set_t& cls = *access_line_set;
	int way_index = cls.way_of_empty();

	shared_access_wait_flag = 1;
	if (way_index >= 0) {
		// exist empty line
		access_line = &cls[way_index];
		access_way = way_index;
		replace_flag = 0;
		bm_state = READ_LINE_COPY_REQ_OWNER;
	} else {
		// replace
		way_index = cls.way_of_throw();
		access_line = &cls[way_index];
		access_way = way_index;
		replace_flag = 1;
		if (!access_line->is_dirty()) {
			// without writeback
			bm_state = READ_LINE_COPY_REQ_OWNER;
		} else {
			// with writeback
			bm_state = REPLACE_REQ_OWNER;
		}
	}
}

//
// bus snoop state machine at "in" clock
//
template <class Type>
void cache_controller_shared<Type>::bus_snoop_in()
{
	if (bm_state != READY) return;

	switch (bs_state) {
	case IDLE:
		//
		// Хȥ󥶥ȯƤʤнλ
		//
		if (!s_bus.is_owned()) break;
		assert(!shared_access_wait_flag);
		//
		// å饤󥳥ԡ׵
		//
		if (s_bus.is_line_copy_request()) {
			buf_adr = s_bus.address();
			buf_dat.resize(buf_adr.size());
			access_adr = cache_buf->tag_address(buf_adr[0]);
			buf_adr[0] = access_adr;
			access_line_set = &cache_buf->cache_line_set(access_adr);
			cache_line_set_t& cls = *access_line_set;
			const int way_index = cls.way_of_hit(access_adr);
			if (way_index >= 0) {
				// hit
				access_line = &cls[way_index];
				access_way = way_index;
				shared_access_type = LINE_COPY_ACCESS;
				shared_access_wait_flag = 1;
				bs_state = LINE_COPY_ACK_SEND;
			} else {
				// miss
				wait_count = 2;
				bs_state = WAIT;
			}
		}
		//
		// å饤ư׵
		//
		else if (s_bus.is_line_move_request()) {
			buf_adr = s_bus.address();
			buf_dat.resize(buf_adr.size());
			access_adr = cache_buf->tag_address(buf_adr[0]);
			buf_adr[0] = access_adr;
			access_line_set = &cache_buf->cache_line_set(access_adr);
			cache_line_set_t& cls = *access_line_set;
			const int way_index = cls.way_of_hit(access_adr);
			if (way_index >= 0) {
				// hit
				access_line = &cls[way_index];
				access_way = way_index;
				shared_access_type = LINE_MOVE_ACCESS;
				shared_access_wait_flag = 1;
				bs_state = LINE_COPY_ACK_SEND;
			} else {
				// miss
				wait_count = 2;
				bs_state = WAIT;
			}
		}
		//
		// cache line ̵׵
		//
		else if (s_bus.is_invalidation_request()) {
			buf_adr = s_bus.address();
			const Type adr = cache_buf->tag_address(buf_adr[0]);
			cache_line_set_t& cls = cache_buf->cache_line_set(adr);
			const int way_index = cls.way_of_hit(adr);
			if (way_index >= 0) {
#ifdef DEBUG
				assert(cls[way_index].is_clean_shared());
#endif // DEBUG
				cls[way_index].set_invalid();
			}
		}
		break;

	case LINE_COPY_DECIDE_SENDER:
		assert(sender != NULL);
		sender->update();
		if (sender->user(0) == id()) {
			// I'm owner.
			if (!access_line->is_dirty()) {
				// copy clean line to other cache
				bs_state = LINE_COPY_TRANS;
			} else {
				// write dirty line to memory and other cache
				bs_state = LINE_COPY_TO_MEMORY_SEND;
			}
		} else {
			// Owner is other cache.
			if (shared_access_type == LINE_MOVE_ACCESS) {
#ifdef DEBUG
				assert(access_line->is_clean_shared());
#endif // DEBUG
				access_line->set_invalid();
			}
			shared_access_wait_flag = 0;
			bs_state = LINE_COPY_CANCELED;
		}
		break;

	case LINE_COPY_TO_MEMORY_WAIT:
		if (s_bus.is_write_ack()) {
			bs_state = LINE_COPY_TRANS;
		}
		break;

	case WAIT:
		if (--wait_count == 0)
			bs_state = IDLE;
		break;

	default:
		break;
	}
}

//
// bus snoop state machine at "out" clock
//
template <class Type>
void cache_controller_shared<Type>::bus_snoop_out()
{
	switch (bs_state) {
	case LINE_COPY_ACK_SEND:
		if (shared_access_type == LINE_COPY_ACCESS) {
			s_bus.send_line_copy_ack();
		} else {
			s_bus.send_line_move_ack();
		}
		sender->request(id());
		bs_state = LINE_COPY_DECIDE_SENDER;
		break;

	case LINE_COPY_CANCELED:
		sender->cancel(id());
		bs_state = IDLE;
		break;

	case LINE_COPY_TO_MEMORY_SEND:
		s_bus.send_single_write_request(buf_adr);
		bs_state = LINE_COPY_TO_MEMORY_WAIT;
		break;

	case LINE_COPY_TRANS:
		for (size_t i = 0; i < cache_buf->line_size_in_word(); i++) {
			buf_dat[i] = (*access_line)[i];
		}
		s_bus.send_single_write_data(buf_dat);
		if (shared_access_type == LINE_COPY_ACCESS) {
			access_line->set_clean_shared();
		} else {
			access_line->set_invalid();
		}
		shared_access_wait_flag = 0;
		bs_state = LINE_COPY_TRANS_END;
		break;

	case LINE_COPY_TRANS_END:
//		s_bus.clear();
		sender->clear();				// ϸġPErelease٤
		bs_state = IDLE;
		break;

	default:
		break;
	}
}

//
// bus master state machine at "in" clock
//
template <class Type>
void cache_controller_shared<Type>::bus_master_in()
{
	if (bs_state != IDLE) return;

	switch (bm_state) {
	case READY:
		if (s_bus.is_owned()) return;	// Хȥ󥶥ʤ齪λ
		assert(!shared_access_wait_flag);
//		assert(!s_bus.is_owner());		// XXX

		//
		// ץå¦꡼׵
		//
		if (local_read_wait_flag) {
			const int line_size = cache_buf->line_size_in_word();
			buf_adr.resize(line_size);
			buf_rep.resize(line_size);
			buf_dat.resize(line_size);
			access_adr = cache_buf->tag_address(local_read_address());
			buf_adr[0] = access_adr;
			access_line_set = &cache_buf->cache_line_set(access_adr);

			shared_access_type = READ_ACCESS;
			case_of_cache_miss();
		}
		//
		// ץå¦饤׵
		//
		else if (local_write_wait_flag) {
			const int line_size = cache_buf->line_size_in_word();
			buf_adr.resize(line_size);
			buf_rep.resize(line_size);
			buf_dat.resize(line_size);
			access_adr = cache_buf->tag_address(local_write_address());
			buf_adr[0] = access_adr;
			access_line_set = &cache_buf->cache_line_set(access_adr);
			cache_line_set_t& cls = *access_line_set;

			int way_index = cls.way_of_hit(access_adr);
			if (way_index < 0) {
				// miss
				shared_access_type = WRITE_ACCESS;
				case_of_cache_miss();
			} else {
				// hit
				Type  a = local_write_address();
				Type& d = cls[way_index][cache_buf->line_offset(a)];
				if (cls[way_index].is_dirty()) {
					// dirty line
					d = local_write_data();
					cls.increment_lru_counter(way_index);
					local_write_wait_flag = 0;
				} else if (cls[way_index].is_clean_exclusive()) {
					// clean-exclusive line
					d = local_write_data();
					cls[way_index].set_dirty();
					cls.increment_lru_counter(way_index);
					local_write_wait_flag = 0;
				} else {
					// clean-shared line
					shared_access_type = WRITE_ACCESS;
					shared_access_wait_flag = 1;
					access_line = &cls[way_index];
					access_way = way_index;
					bm_state = INV_REQ_OWNER;
				}
			}
		}
		//
		// ץå¦Ʊ꡼׵
		//
		else if (local_sync_wait_flag) {
			// sync read from processor
			const int line_size = cache_buf->line_size_in_word();
			buf_adr.resize(line_size);
			buf_rep.resize(line_size);
			buf_dat.resize(line_size);
			access_adr = cache_buf->tag_address(local_sync_address());
			buf_adr[0] = access_adr;
			access_line_set = &cache_buf->cache_line_set(access_adr);
			cache_line_set_t& cls = *access_line_set;

			int way_index = cls.way_of_hit(access_adr);
			if (way_index < 0) {
				// miss
				shared_access_type = SYNC_ACCESS;
				case_of_cache_miss();
			} else {
				// hit: write to clean-shared line
#ifdef DEBUG
				assert(cls[way_index].is_clean_shared());
#endif // DEBUG
				shared_access_type = SYNC_ACCESS;
				shared_access_wait_flag = 1;
				access_line = &cls[way_index];
				access_way = way_index;
				bm_state = INV_REQ_OWNER;
			}
		}
		break;

	case READ_LINE_COPY_REQ_OWNER:
		if (s_bus.is_owner()) {
			bm_state = READ_LINE_COPY_SEND;
		} else {
			shared_access_wait_flag = 0;
			bm_state = READY;
		}
		break;

	case READ_LINE_COPY_WAIT:
		if (--wait_count > 0) break;

		access_line->tag() = access_adr;
		if (replace_flag) {
			access_line_set->clear_lru_counter();
		} else {
			access_line_set->clear_lru_counter(access_way);
		}
		switch (shared_access_type) {
		case READ_ACCESS:
			// read
			if (s_bus.is_line_copy_ack()) {
				access_line->set_clean_shared();
				bm_state = READ_FROM_CACHE_TRANS;
			} else {
				access_line->set_clean_exclusive();
				bm_state = READ_FROM_MEMORY_SEND;
			}
			break;
		case WRITE_ACCESS:
			// write
			access_line->set_dirty();
			if (s_bus.is_line_move_ack()) {
				bm_state = READ_FROM_CACHE_TRANS;
			} else {
				bm_state = READ_FROM_MEMORY_SEND;
			}
			break;
		case SYNC_ACCESS:
			// read with modify
			if (s_bus.is_line_move_ack()) {
				bm_state = READ_FROM_CACHE_TRANS;
			} else {
				bm_state = READ_FROM_MEMORY_SEND;
			}
			break;
		default:
			break;
		}
		break;

	case READ_FROM_CACHE_TRANS:
		if (!s_bus.is_write_data()) break;

		for (size_t i = 0; i < cache_buf->line_size_in_word(); i++) {
			(*access_line)[i] = s_bus.data()[i];
		}
		switch (shared_access_type) {
		case READ_ACCESS:
			local_read_wait_flag = 0;
			local_read_data_
				= (*access_line)[cache_buf->line_offset(local_read_address())];
			shared_access_wait_flag = 0;
			bm_state = READ_LINE_REL_OWNER;
			break;

		case WRITE_ACCESS:
			local_write_wait_flag = 0;
			(*access_line)[cache_buf->line_offset(local_write_address())]
				= local_write_data();
			shared_access_wait_flag = 0;
			bm_state = READ_LINE_REL_OWNER;
			break;

		case SYNC_ACCESS: {
			local_sync_wait_flag = 0;
			Type& d =
				(*access_line)[cache_buf->line_offset(local_sync_address())];
			local_sync_data_ = d;
			if (d > 0) {
				access_line->set_dirty();
				d--;
			} else {
				access_line->set_clean_exclusive();
			}
			shared_access_wait_flag = 0;
			bm_state = READ_LINE_REL_OWNER;
			break;
		}// SYNC_ACCESS

		default:
			break;
		}
		break;

	case READ_FROM_MEMORY_WAIT:
		if (s_bus.is_read_ack()) {
			if (shared_access_type == READ_ACCESS) {
				if (cache_buf->line_offset(local_read_address()) == 0) {
					local_read_wait_flag = 0;
				}
			} else if (shared_access_type == SYNC_ACCESS) {
				if (cache_buf->line_offset(local_sync_address()) == 0) {
					local_sync_wait_flag = 0;
				}
			}
			bm_state = READ_FROM_MEMORY_TRANS;
		}
		break;

	case READ_FROM_MEMORY_TRANS:
		if (!s_bus.is_read_data()) break;

		for (size_t i = 0; i < cache_buf->line_size_in_word(); i++) {
			(*access_line)[i] = s_bus.data()[i];
		}
		switch (shared_access_type) {
		case READ_ACCESS:
			local_read_wait_flag = 0;
			local_read_data_
				= (*access_line)[cache_buf->line_offset(local_read_address())];
			shared_access_wait_flag = 0;
			bm_state = READ_LINE_REL_OWNER;
			break;

		case WRITE_ACCESS:
			local_write_wait_flag = 0;
			(*access_line)[cache_buf->line_offset(local_write_address())]
				= local_write_data();
			shared_access_wait_flag = 0;
			bm_state = READ_LINE_REL_OWNER;
			break;

		case SYNC_ACCESS: {
			local_sync_wait_flag = 0;
			Type& d =
				(*access_line)[cache_buf->line_offset(local_sync_address())];
			local_sync_data_ = d;
			if (d > 0) {
				access_line->set_dirty();
				d--;
			} else {
				access_line->set_clean_exclusive();
			}
			shared_access_wait_flag = 0;
			bm_state = READ_LINE_REL_OWNER;
			break;
		} // SYNC_ACCESS

		default:
			break;
		}
		break;

	case REPLACE_REQ_OWNER:
		if (s_bus.is_owner()) {
			bm_state = REPLACE_SEND;
		} else {
			shared_access_wait_flag = 0;
			bm_state = READY;
		}
		break;

	case REPLACE_WAIT:
		buf_rep[0] = access_line->tag();
		if (s_bus.is_write_ack())
			bm_state = REPLACE_TRANS;
		break;

	case INV_REQ_OWNER:
		if (s_bus.is_owner()) {
			cache_line_t& cl = *access_line;
			cache_line_set_t& cls = *access_line_set;
			cls.increment_lru_counter(access_way);
			if (shared_access_type == WRITE_ACCESS) {
				// write
				cl.set_dirty();
				cl[cache_buf->line_offset(local_write_address())]
					= local_write_data();
				local_write_wait_flag = 0;
			} else {
				// sync read: data is always greater than 0 in this line
				Type& d = cl[cache_buf->line_offset(local_sync_address())];
#ifdef DEBUG
				assert(d > 0);
#endif // DEBUG
				cl.set_dirty();
				local_sync_data_ = d--;
				local_sync_wait_flag = 0;
			}
			shared_access_wait_flag = 0;
			bm_state = INV_SEND;
		} else {
			shared_access_wait_flag = 0;
			bm_state = READY;
		}
		break;

	default:
		break;
	}
}

//
// bus master state machine at "out" clock
//
template <class Type>
void cache_controller_shared<Type>::bus_master_out()
{
	switch (bm_state) {
	case READ_LINE_COPY_REQ_OWNER:
		s_bus.request_ownership();
		break;

	case READ_LINE_COPY_SEND:
		if (shared_access_type == READ_ACCESS) {
			// read
			s_bus.send_line_copy_request(buf_adr);
		} else {
			// write, read with modify
			s_bus.send_line_move_request(buf_adr);
		}
		wait_count = 2;
		bm_state = READ_LINE_COPY_WAIT;
		break;

	case READ_FROM_MEMORY_SEND:
		s_bus.send_single_read_request(buf_adr);
		bm_state = READ_FROM_MEMORY_WAIT;
		break;

	case READ_LINE_REL_OWNER:
		s_bus.clear();
		s_bus.release_ownership();
		bm_state = READY;
		break;

	case REPLACE_REQ_OWNER:
		s_bus.request_ownership();
		break;

	case REPLACE_SEND:
		buf_rep[0] = access_line->tag();
		s_bus.send_single_write_request(buf_rep);
		bm_state = REPLACE_WAIT;
		break;

	case REPLACE_TRANS:
		for (size_t i = 0; i < cache_buf->line_size_in_word(); i++) {
			buf_dat[i] = (*access_line)[i];
		}
		s_bus.send_single_write_data(buf_dat);
		bm_state = READ_LINE_COPY_SEND;
		break;

	case INV_REQ_OWNER:
		s_bus.request_ownership();
		break;

	case INV_SEND:
		s_bus.send_invalidation_request(buf_adr);
		bm_state = INV_REL_OWNER;
		break;

	case INV_REL_OWNER:
		s_bus.clear();
		s_bus.release_ownership();
		bm_state = READY;
		break;

	default:
		break;
	}
}

template <class Type>
void cache_controller_shared<Type>::output(ostream& os) const
{
	const long flags = os.flags();
	os << hex;
	if (bs_state == IDLE) {
		switch (bm_state) {
		case READY:
			os << "ready";
			break;
		case READ_LINE_COPY_REQ_OWNER:
			os << "read_line_copy_request_ownership(0x" << access_adr << ')';
			break;
		case READ_LINE_COPY_SEND:
			os << "read_line_copy_send(0x" << access_adr << ')';
			break;
		case READ_LINE_COPY_WAIT:
			os << "read_line_copy_wait(0x" << access_adr << ')';
			break;
		case READ_FROM_CACHE_TRANS:
			os << "read_from_cache_trans(0x" << access_adr << ')';
			break;
		case READ_FROM_MEMORY_SEND:
			os << "read_from_memory_send(0x" << access_adr << ')';
			break;
		case READ_FROM_MEMORY_WAIT:
			os << "read_from_memory_wait(0x" << access_adr << ')';
			break;
		case READ_FROM_MEMORY_TRANS:
			os << "read_from_memory_trans(0x" << access_adr << ')';
			break;
		case READ_LINE_REL_OWNER:
			os << "read_line_release_ownership(0x"
			   << access_adr << ')';
			break;
		case REPLACE_REQ_OWNER:
			os << "replace_request_ownership(0x" << access_line->tag() << ')';
			break;
		case REPLACE_SEND:
			os << "replace_send(0x" << access_line->tag() << ')';
			break;
		case REPLACE_WAIT:
			os << "replace_wait(0x" << access_line->tag() << ')';
			break;
		case REPLACE_TRANS:
			os << "replace_trans(0x" << access_line->tag() << ')';
			break;
		case INV_REQ_OWNER:
			os << "invalidation_request_ownership(0x" << access_adr << ')';
			break;
		case INV_SEND:
			os << "invalidation_send(0x" << access_adr << ')';
			break;
		case INV_REL_OWNER:
			os << "invalidation_release_ownership(0x" << access_adr << ')';
			break;
		}
	} else {
		switch (bs_state) {
		case IDLE:
			os << "ready";
			break;
		case LINE_COPY_ACK_SEND:
			os << "line_copy_ack_send(0x" << access_adr << ')';
			break;
		case LINE_COPY_DECIDE_SENDER:
			os << "line_copy_decide_sender(0x" << access_adr << ')';
			sender->update();
			if (sender->user(0) == id()) {
				os << "win  ";
			} else {
				os << "lose ";
			}
			break;
		case LINE_COPY_CANCELED:
			os << "line_copy_canceled(0x" << access_adr << ')';
			break;
		case LINE_COPY_TO_MEMORY_SEND:
			os << "line_copy_to_memory_send(0x" << access_adr << ')';
			break;
		case LINE_COPY_TO_MEMORY_WAIT:
			os << "line_copy_to_memory_wait(0x" << access_adr << ')';
			break;
		case LINE_COPY_TRANS:
			os << "line_copy_trans(0x" << access_adr << ')';
			break;
		case LINE_COPY_TRANS_END:
			os << "line_copy_trans_end(0x" << access_adr << ')';
			break;
		case WAIT:
			os << "wait(" << wait_count << ')';
			break;
		}
	}
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // CACHE_CONTROLLER_SHARED_H
