/* -*- C++ -*-
 * Copyright (c) 1995-1996 Tohru Kisuki
 *               1995-2001 Masaki Wakabayashi
 *               1998 Keisuke Inoue
 *      	 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */
/*
 * <<< cache_controller.h >>>
 * Snoop cache controller
 */

#ifndef CACHE_CONTROLLER_H
#define CACHE_CONTROLLER_H

#include <iostream>
#include <isis/isis.h>
#include "cache_protocol.h"
#include "cache_controller_local.h"
#include "cache_controller_shared.h"
#include "big_word.h"
// for verify
#include <isis/mapped_memory.h>

class arbiter;

template <class Type>
class cache_controller : public synchronous_unit
{
private:
	typedef synchronous_unit inherited;
	typedef cache_line_PROTOCOL<Type, Type> cache_line_t;
	typedef set_associative_cache<cache_line_t> cache_t;
	typedef big_word<Type> bus_size_t;
	cache_controller_local<Type> lo_ctl;
	cache_controller_shared<Type> sh_ctl;
public:
	cache_controller();
	cache_controller(const cache_controller<Type>&);
	virtual ~cache_controller() {}
	virtual void output(ostream&) const;
	inline virtual void clock_in();
	inline virtual void clock_out();
	virtual void reset();
	const port& local_bus_port() const {
		return lo_ctl.port_ref(); }
	const port& shared_bus_port() const {
		return sh_ctl.port_ref(); }
	port& local_bus_port() {
		return lo_ctl.port_ref(); }
	port& shared_bus_port() {
		return sh_ctl.port_ref(); }
	int is_shared_address(Type a) const {
		return lo_ctl.is_shared_address(a); }
	int is_sync_address(Type a) const {
		return lo_ctl.is_sync_address(a); }
	int is_valid_address(Type a) const {
		return lo_ctl.is_valid_address(a); }
	void set_shared_address(Type a, Type b) {
		lo_ctl.set_shared_address(a, b); }
	void set_sync_address(Type a, Type b) {
		lo_ctl.set_sync_address(a, b); }
	// local bus
	void connect_local_bus_port(bus_port<Type>& a) {
		lo_ctl.connect(a); }
	void disconnect_local_bus_port() {
		lo_ctl.disconnect(); }
	void disconnect_local_bus_port(bus_port<Type>& a) {
		lo_ctl.disconnect(a); }
	int is_connected_to_local_bus_port() const {
		return lo_ctl.is_connected(); }
	int is_connected_to_local_bus_port(const bus_port<Type>& a) const {
		return lo_ctl.is_connected(a); }
	// shared bus
	void connect_shared_bus_port(bus_port<bus_size_t>& a) {
		sh_ctl.connect(a); }
	void disconnect_shared_bus_port() {
		sh_ctl.disconnect(); }
	void disconnect_shared_bus_port(bus_port<bus_size_t>& a) {
		sh_ctl.disconnect(a); }
	int is_connected_to_shared_bus_port() const {
		return sh_ctl.is_connected(); }
	int is_connected_to_shared_bus_port(const bus_port<bus_size_t>& a) const {
		return sh_ctl.is_connected(a); }
	// cache
	void connect_cache(cache_t& a) {
		lo_ctl.connect_cache(a), sh_ctl.connect_cache(a); }
	void disconnect_cache() {
		lo_ctl.disconnect_cache(), sh_ctl.disconnect_cache(); }
	int is_connected_to_cache() const {
		return lo_ctl.is_connected_to_cache(); }
	int id() const {
		return sh_ctl.port_ref().id(); }
	int id(int a) {
		shared_bus_port().set_id(a); sh_ctl.port_ref().set_id(a); return a; }
	// arbiter
	void connect_sender_arb(arbiter& a) {
		sh_ctl.connect_sender_arb(a); }
	void disconnect_sender_arb() {
		sh_ctl.disconnect_sender_arb(); }
	int is_connected_sender_arb() const {
		return sh_ctl.is_connected_sender_arb(); }
	// for verify
	void connect_mem(mapped_memory<Type, Type> *sh,
					 mapped_memory<Type, Type> *sy) {
		lo_ctl.connect_mem(sh, sy); }
	int is_connected_mem() {
		return lo_ctl.is_connected_mem(); }
	void disconnect_mem() {
		lo_ctl.disconnect_mem(); }
	void connect_mem_port(bus_port<Type>& a) {
		lo_ctl.connect_mem_port(a); }
	void disconnect_mem_port() {
		lo_ctl.disconnect_mem_port(); }
	void disconnect_mem_port(bus_port<Type>& a) {
		lo_ctl.disconnect_mem_port(a); }
	int is_connected_mem_port() const {
		return lo_ctl.is_connected_mem_port(); }
	int is_connected_mem_port(const bus_port<Type>& a) const {
		return lo_ctl.is_connected_mem_port(a); }
	//
	int shared_access_is_finished() const {
		return sh_ctl.shared_access_is_finished(); }
};

template <class Type>
cache_controller<Type>::cache_controller()
	: lo_ctl(sh_ctl)
{
}

template <class Type>
cache_controller<Type>::cache_controller(const cache_controller<Type>& a)
	: lo_ctl(a.lo_ctl, sh_ctl),
	  sh_ctl(sh_ctl)
{
}

template <class Type>
void cache_controller<Type>::reset()
{
	lo_ctl.reset(), sh_ctl.reset();
}

template <class Type>
void cache_controller<Type>::clock_in()
{
	lo_ctl.clock_in();
	sh_ctl.bus_snoop_in();
	sh_ctl.bus_master_in();
}

template <class Type>
void cache_controller<Type>::clock_out()
{
	sh_ctl.bus_master_out();
	sh_ctl.bus_snoop_out();
	lo_ctl.clock_out();
}

template <class Type>
void cache_controller<Type>::output(ostream& os) const
{
	os << lo_ctl << '-' << sh_ctl;
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif // CACHE_CONTROLLER_H
