/*
 * <<< torus_2d_ecube_router.h >>>
 *
 * --- Router class for 2D-torus, with ecube routing
 *     Copyright (C) 2000 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef TORUS_2D_ECUBE_ROUTER_H
#define TORUS_2D_ECUBE_ROUTER_H

#include <isis/deterministic_router.h>

template <class P>
class torus_2d_ecube_router : public deterministic_router<P>
{
private:
	typedef torus_2d_ecube_router<P> thisclass;
	typedef deterministic_router<P> inherited;
public:
	typedef inherited::packet_type packet_type;
	typedef inherited::node_address_type node_address_type;
private:
	const static size_t physical_channel_size = 5;
	const static size_t virtual_channel_size = 2;
	const static size_t PLUS_X_PH_CHANNEL  = 0;
	const static size_t MINUS_X_PH_CHANNEL = 1;
	const static size_t PLUS_Y_PH_CHANNEL  = 2;
	const static size_t MINUS_Y_PH_CHANNEL = 3;
	const static size_t COMSUMP_PH_CHANNEL = 4;
	const static size_t NORMAL_VR_CHANNEL = 0;
	const static size_t AROUND_VR_CHANNEL = 1;
	enum state_type { READY, BUSY };
	size_t x_size_, y_size_;
protected:
	virtual void routing(const packet_type&, channel_controller&);
public:
	torus_2d_ecube_router(void);
	torus_2d_ecube_router(const torus_2d_ecube_router&);
	virtual ~torus_2d_ecube_router() {}
	void set_size(size_t a, size_t b) { x_size_ = a, y_size_ = b; }
};

template <class P>
torus_2d_ecube_router<P>::torus_2d_ecube_router(void)
	: inherited(),
	  x_size_(0),
	  y_size_(0)
{
	set_input_size(physical_channel_size);
	set_output_size(physical_channel_size);
	for (size_t i = 0; i < physical_channel_size - 1; i++) {
		set_channel_size(i, virtual_channel_size);
		set_buffer_size(i, 1);
	}
	set_channel_size(COMSUMP_PH_CHANNEL, 1);
	set_buffer_size(COMSUMP_PH_CHANNEL, 1);
}

template <class P>
torus_2d_ecube_router<P>::torus_2d_ecube_router
	(const torus_2d_ecube_router<P>& a)
	: inherited(a),
	  x_size_(a.x_size_),
	  y_size_(a.y_size_)
{}

template <class P>
void torus_2d_ecube_router<P>::routing
	(const torus_2d_ecube_router<P>::packet_type& pkt,
	 channel_controller& ctl)
{
	if (pkt.destination() == node_address()) {
		// destination is here, transmit to PE
		ctl.set_destination(COMSUMP_PH_CHANNEL, NORMAL_VR_CHANNEL);
	} else {
		// destination is not here, select channel with e-cube routing
		size_t my_x, my_y, dst_x, dst_y, curr_phch, curr_vrch, next_phch,
			   next_vrch;
		my_x = node_address() % x_size_;
		my_y = node_address() / x_size_;
		dst_x = pkt.destination() % x_size_;
		dst_y = pkt.destination() / x_size_;
		curr_phch = ctl.physical_channel_id();
		curr_vrch = ctl.virtual_channel_id();
		if (my_x != dst_x) {
			switch (curr_phch) {
			case COMSUMP_PH_CHANNEL:
				if (my_x < dst_x) {
					if (dst_x - my_x <= my_x + x_size_ - dst_x) {
						next_phch = PLUS_X_PH_CHANNEL;
						next_vrch = ((my_x != x_size_ - 1) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					} else {
						next_phch = MINUS_X_PH_CHANNEL;
						next_vrch = ((my_x != 0) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					}
				} else {
					if (dst_x + x_size_ - my_x <= my_x - dst_x) {
						next_phch = PLUS_X_PH_CHANNEL;
						next_vrch = ((my_x != x_size_ - 1) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					} else {
						next_phch = MINUS_X_PH_CHANNEL;
						next_vrch = ((my_x != 0) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					}
				}
				break;
			case MINUS_X_PH_CHANNEL:
				next_phch = PLUS_X_PH_CHANNEL;
				if (curr_vrch == NORMAL_VR_CHANNEL && my_x != x_size_ - 1) {
					next_vrch = NORMAL_VR_CHANNEL;
				} else {
					next_vrch = AROUND_VR_CHANNEL;
				}
				break;
			case PLUS_X_PH_CHANNEL:
				next_phch = MINUS_X_PH_CHANNEL;
				if (curr_vrch == NORMAL_VR_CHANNEL && my_x != 0) {
					next_vrch = NORMAL_VR_CHANNEL;
				} else {
					next_vrch = AROUND_VR_CHANNEL;
				}
				break;
			default:
				assert(false);
				break;
			}
		} else {
			switch (curr_phch) {
			case COMSUMP_PH_CHANNEL:
			case PLUS_X_PH_CHANNEL:
			case MINUS_X_PH_CHANNEL:
				if (my_y < dst_y) {
					if (dst_y - my_y <= my_y + y_size_ - dst_y) {
						next_phch = PLUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != y_size_ - 1) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					} else {
						next_phch = MINUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != 0) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					}
				} else {
					if (dst_y + y_size_ - my_y <= my_y - dst_y) {
						next_phch = PLUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != y_size_ - 1) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					} else {
						next_phch = MINUS_Y_PH_CHANNEL;
						next_vrch = ((my_y != 0) ?
							NORMAL_VR_CHANNEL : AROUND_VR_CHANNEL);
					}
				}
				break;
			case MINUS_Y_PH_CHANNEL:
				next_phch = PLUS_Y_PH_CHANNEL;
				if (curr_vrch == NORMAL_VR_CHANNEL && my_y != y_size_ - 1) {
					next_vrch = NORMAL_VR_CHANNEL;
				} else {
					next_vrch = AROUND_VR_CHANNEL;
				}
				break;
			case PLUS_Y_PH_CHANNEL:
				next_phch = MINUS_Y_PH_CHANNEL;
				if (curr_vrch == NORMAL_VR_CHANNEL && my_y != 0) {
					next_vrch = NORMAL_VR_CHANNEL;
				} else {
					next_vrch = AROUND_VR_CHANNEL;
				}
				break;
			default:
				assert(false);
				break;
			}
		}
		ctl.set_destination(next_phch, next_vrch);
	}
}

#endif /* TORUS_2D_ECUBE_ROUTER_H */
