/*
 * <<< r3010_float_wrapper.cc >>>
 *
 * --- Copyright (C) 1996-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#if HAVE_CONFIG_H
# include "config.h"
#endif /* HAVE_CONFIG_H */
#ifndef __EXTENSIONS__ /* for Solaris2.6 */
# define __EXTENSIONS__
#endif /* __EXTENSIONS__ */
#ifndef _GNU_SOURCE /* for Linux */
# define _GNU_SOURCE
#endif /* _GNU_SOURCE */
#ifndef _HPUX_SOURCE /* for HPUX */
# define _HPUX_SOURCE
#endif /* _HPUX_SOURCE */
#include <cmath>
#include "r3010_float_wrapper.h"
#include "r3010_float_wrapper_local.h"

/* setup for float copysign() */
#if R3010_SFLOAT_IS_FLOAT /* r3010_sfloat is float */
# if HAVE_COPYSIGNF
#  define sfloat_copysign copysignf
# elif HAVE_COPYSIGN
#  define sfloat_copysign copysign
# else
#  error "Can't find copysign() function."
# endif
#else /* r3010_sfloat is double */
# if HAVE_COPYSIGN
#  define sfloat_copysign copysign
# else
#  error "Can't find copysign() function."
# endif
#endif

/* setup for double copysign() */
#if R3010_DFLOAT_IS_FLOAT /* r3010_dfloat is float */
# if HAVE_COPYSIGNF
#  define dfloat_copysign copysignf
# elif HAVE_COPYSIGN
#  define dfloat_copysign copysign
# else
#  error "Can't find copysign() function."
# endif
#else /* r3010_dfloat is double */
# if HAVE_COPYSIGN
#  define dfloat_copysign copysign
# else
#  error "Can't find copysign() function."
# endif
#endif

/* setup for float ilogb() */
#if R3010_SFLOAT_IS_FLOAT /* r3010_sfloat is float */
# if HAVE_ILOGBF
#  define sfloat_ilogb ilogbf
# elif HAVE_ILOGB
#  define sfloat_ilogb ilogb
# else
#  error "Can't find ilogb() function."
# endif
#else /* r3010_sfloat is double */
# if HAVE_ILOGB
#  define sfloat_ilogb ilogb
# else
#  error "Can't find ilogb() function."
# endif
#endif

/* setup for double ilogb() */
#if R3010_DFLOAT_IS_FLOAT /* r3010_dfloat is float */
# if HAVE_ILOGBF
#  define dfloat_ilogb ilogbf
# elif HAVE_ILOGB
#  define dfloat_ilogb ilogb
# else
#  error "Can't find ilogb() function."
# endif
#else /* r3010_dfloat is double */
# if HAVE_ILOGB
#  define dfloat_ilogb ilogb
# else
#  error "Can't find ilogb() function."
# endif
#endif

/* setup for float scalbn() */
#if R3010_SFLOAT_IS_FLOAT /* r3010_sfloat is float */
# if HAVE_SCALBNF
#  define sfloat_scalbn scalbnf
# elif HAVE_SCALBN
#  define sfloat_scalbn scalbn
# elif HAVE_LDEXP
#  define sfloat_scalbn ldexp
# else
#  error "Can't find scalbn() and ldexp() function."
# endif
#else /* r3010_sfloat is double */
# if HAVE_SCALBN
#  define sfloat_scalbn scalbn
# elif HAVE_LDEXP
#  define sfloat_scalbn ldexp
# else
#  error "Can't find scalbn() and ldexp() function."
# endif
#endif

/* setup for double scalbn() */
#if R3010_DFLOAT_IS_FLOAT /* r3010_dfloat is float */
# if HAVE_SCALBNF
#  define dfloat_scalbn scalbnf
# elif HAVE_SCALBN
#  define dfloat_scalbn scalbn
# elif HAVE_LDEXP
#  define dfloat_scalbn ldexp
# else
#  error "Can't find scalbn() and ldexp() function."
# endif
#else /* r3010_dfloat is double */
# if HAVE_SCALBN
#  define dfloat_scalbn scalbn
# elif HAVE_LDEXP
#  define dfloat_scalbn ldexp
# else
#  error "Can't find scalbn() and ldexp() function."
# endif
#endif

static inline int get_sign_(r3010_sfloat a)
{
	return (sfloat_copysign(1, a) < 0) ? 1 : 0;
}

static inline int get_exp_(r3010_sfloat a)
{
	return sfloat_ilogb(a);
}

static inline void get_frac_(r3010_sfloat a, r3000_word* ret)
{
	union { r3010_sfloat f; r3000_word w; } data;
	data.f = a;
	*ret = (data.w & R3010_SFLOAT_FRAC_MASK);
}

static inline int get_sign_(r3010_dfloat a)
{
	return (dfloat_copysign(1, a) < 0) ? 1 : 0;
}

static inline int get_exp_(r3010_dfloat a)
{
	return dfloat_ilogb(a);
}

static inline void get_frac_(r3010_dfloat a, r3000_word* ret)
{
	union { r3010_dfloat f; r3000_word w[2]; } data;
	data.f = a;
#if WORDS_BIGENDIAN
	/* big-endian */
	ret[0] = (data.w[0] & R3010_DFLOAT_FRAC_MASK_HI);
	ret[1] = data.w[1];
#else
	/* little-endian */
	ret[0] = (data.w[1] & R3010_DFLOAT_FRAC_MASK_HI);
	ret[1] = data.w[0];
#endif
}

static inline r3010_sfloat make_single_(int sign, int exp, r3000_word frac)
{
	union { r3010_sfloat f; r3000_word w; } data;
	if (exp == EXP_ZERO) {
		data.f = 0;
	} else if (exp == EXP_INFINITY) {
		volatile r3010_sfloat tmp = R3010_SFLOAT_MAX;
		data.f = tmp + R3010_SFLOAT_MAX; /* infinity */
	} else {
		data.w = (frac & R3010_SFLOAT_FRAC_MASK) |
				 (R3010_SFLOAT_EXP_BASE << R3010_SFLOAT_EXP_SHIFT);
		data.f = sfloat_scalbn(data.f, exp);
	}
	return sfloat_copysign(data.f, (sign ? -1 : 1));
}

static inline r3010_dfloat make_double_
	(int sign, int exp, const r3000_word* frac)
{
	union { r3010_dfloat f; r3000_word w[2]; } data;
	if (exp == EXP_ZERO) {
		data.f = 0;
	} else if (exp == EXP_INFINITY) {
		volatile r3010_dfloat tmp = R3010_DFLOAT_MAX;
		data.f = tmp + R3010_DFLOAT_MAX; /* infinity */
	} else {
#if WORDS_BIGENDIAN
		/* big-endian */
		data.w[0] = (frac[0] & R3010_DFLOAT_FRAC_MASK_HI) |
					(R3010_DFLOAT_EXP_BASE << R3010_DFLOAT_EXP_SHIFT_HI);
		data.w[1] = frac[1];
#else
		/* little-endian */
		data.w[0] = frac[1];
		data.w[1] = (frac[0] & R3010_DFLOAT_FRAC_MASK_HI) |
					(R3010_DFLOAT_EXP_BASE << R3010_DFLOAT_EXP_SHIFT_HI);
#endif
		data.f = dfloat_scalbn(data.f, exp);
	}
	return dfloat_copysign(data.f, (sign ? -1 : 1));
}

static inline r3010_sfloat word_to_single_(r3000_word w)
{
	int sign, exp;
	r3000_word frac;
	sign = ((w & R3010_SFLOAT_SIGN_MASK) >> R3010_SFLOAT_SIGN_SHIFT);
	exp = ((w & R3010_SFLOAT_EXP_MASK) >> R3010_SFLOAT_EXP_SHIFT)
		  - R3010_SFLOAT_EXP_BASE;
	frac = ((w & R3010_SFLOAT_FRAC_MASK) >> R3010_SFLOAT_FRAC_SHIFT);
	if (exp > R3010_SFLOAT_EXP_MAX_VALUE) {
		exp = EXP_INFINITY;
	} else if (exp < R3010_SFLOAT_EXP_MIN_VALUE) {
		exp = EXP_ZERO;
	}
	return make_single_(sign, exp, frac);
}

static inline r3010_dfloat word_to_double_(const r3000_word* w)
{
	r3000_word frac[2];
	int sign, exp;
	sign = ((w[1] & R3010_DFLOAT_SIGN_MASK_HI) >> R3010_DFLOAT_SIGN_SHIFT_HI);
	exp = ((w[1] & R3010_DFLOAT_EXP_MASK_HI) >> R3010_DFLOAT_EXP_SHIFT_HI)
		  - R3010_DFLOAT_EXP_BASE;
	frac[0]
		= ((w[1] & R3010_DFLOAT_FRAC_MASK_HI) >> R3010_DFLOAT_FRAC_SHIFT_HI);
	frac[1]
		= ((w[0] & R3010_DFLOAT_FRAC_MASK_LO) >> R3010_DFLOAT_FRAC_SHIFT_LO);
	if (exp > R3010_DFLOAT_EXP_MAX_VALUE) {
		exp = EXP_INFINITY;
	} else if (exp < R3010_DFLOAT_EXP_MIN_VALUE) {
		exp = EXP_ZERO;
	}
	return make_double_(sign, exp, frac);
}

static inline r3000_word single_to_word_(r3010_sfloat f)
{
	r3000_word frac;
	int sign, exp;
	sign = get_sign_(f);
	if (f == 0.) {
		exp = EXP_ZERO;
		frac = 0;
	} else {
		exp = get_exp_(f);
		get_frac_(f, &frac);
	}
	if (exp == EXP_INFINITY) {
		exp = R3010_SFLOAT_EXP_MAX_VALUE + 1;
	} else if (exp == EXP_ZERO) {
		exp = R3010_SFLOAT_EXP_MIN_VALUE - 1;
	}
	exp += R3010_SFLOAT_EXP_BASE;
	return ((sign << R3010_SFLOAT_SIGN_SHIFT) & R3010_SFLOAT_SIGN_MASK) |
		   ((exp << R3010_SFLOAT_EXP_SHIFT) & R3010_SFLOAT_EXP_MASK) |
		   ((frac << R3010_SFLOAT_FRAC_SHIFT) & R3010_SFLOAT_FRAC_MASK);
}

static inline void double_to_word_(r3010_dfloat f, r3000_word* ret)
{
	r3000_word frac[2];
	int sign, exp;
	sign = get_sign_(f);
	if (f == 0.) {
		exp = EXP_ZERO;
		frac[0] = 0;
		frac[1] = 0;
	} else {
		exp = get_exp_(f);
		get_frac_(f, frac);
	}
	if (exp == EXP_INFINITY) {
		exp = R3010_DFLOAT_EXP_MAX_VALUE + 1;
	} else if (exp == EXP_ZERO) {
		exp = R3010_DFLOAT_EXP_MIN_VALUE - 1;
	}
	exp += R3010_DFLOAT_EXP_BASE;
	ret[0] = frac[1];
	ret[1] = ((sign << R3010_DFLOAT_SIGN_SHIFT_HI) & R3010_DFLOAT_SIGN_MASK_HI)
		   | ((exp << R3010_DFLOAT_EXP_SHIFT_HI) & R3010_DFLOAT_EXP_MASK_HI)
		   | ((frac[0] << R3010_DFLOAT_FRAC_SHIFT_HI) &
			  R3010_DFLOAT_FRAC_MASK_HI);
}

int r3010_float_wrapper::get_sign(r3010_sfloat a)
{
	return get_sign_(a);
}

int r3010_float_wrapper::get_exp(r3010_sfloat a)
{
	return get_exp_(a);
}

void r3010_float_wrapper::get_frac(r3010_sfloat a, r3000_word* b)
{
	get_frac_(a, b);
}

int r3010_float_wrapper::get_sign(r3010_dfloat a)
{
	return get_sign_(a);
}

int r3010_float_wrapper::get_exp(r3010_dfloat a)
{
	return get_exp_(a);
}

void r3010_float_wrapper::get_frac(r3010_dfloat a, r3000_word* b)
{
	return get_frac_(a, b);
}

r3010_sfloat r3010_float_wrapper::make_single(int a, int b, r3000_word c)
{
	return make_single_(a, b, c);
}

r3010_dfloat r3010_float_wrapper::make_double(int a, int b, const r3000_word* c)
{
	return make_double_(a, b, c);
}

r3010_sfloat r3010_float_wrapper::word_to_single(r3000_word a)
{
	return word_to_single_(a);
}

r3010_dfloat r3010_float_wrapper::word_to_double(const r3000_word* a)
{
	return word_to_double_(a);
}

r3000_word r3010_float_wrapper::single_to_word(r3010_sfloat a)
{
	return single_to_word_(a);
}

void r3010_float_wrapper::double_to_word(r3010_dfloat a, r3000_word* b)
{
	double_to_word_(a, b);
}

/*********************************************************************
 *																	 *
 *  Following two functions are *NOT* to convert precision.			 *
 *																	 *
 *  They are needed at the situation like:							 *
 *	 add.s $f4, $f0, $f2  ($f4 is used as single precision, but..)	 *
 *	 sub.d $f8, $f4, $f6  ($f4 is used as double precision at here!) *
 *********************************************************************/

r3010_dfloat r3010_float_wrapper::single_to_double(const r3010_sfloat* s)
{
	r3000_word w[2];
	w[0] = single_to_word_(s[0]);
	w[1] = single_to_word_(s[1]);
	return word_to_double_(w);
}

void r3010_float_wrapper::double_to_single(r3010_dfloat d, r3010_sfloat* ret)
{
	r3000_word w[2];
	double_to_word_(d, w);
	ret[0] = word_to_single_(w[0]);
	ret[1] = word_to_single_(w[1]);
}

int r3010_float_wrapper::get_topbit_index(r3000_word w)
{
	int i;
	for (i = -1; w != 0; i++) w >>= 1;
	return i;
}

#if 0
int main(void)
{
	double d, e;
	r3000_word w[2];
	int i;
	for (i = -3; i < 10; i++) {
		d = (double)i / 3.1415926536;
		r3010_double_to_word(d, w);
		r3010_word_to_double(w, e);
		cout << i
			 << ":" << d << ":"
			 << hex << setfill('0')
			 << setw(8) << w[0] << " " << setw(8) << w[1]
			 << dec << setfill(' ')
			 << ":" << e
			 << ":" << (d == e ? "EQ" : "NE") << endl;
	}
	return 0;
}
#endif
