/*
 * <<< r3000_memory_access_unit.cc >>>
 *
 * --- R3000 memory access unit class 'r3000_memory_access_unit'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#include <cstring> // needed only for SGI C++ compiler
#include <iomanip> // needed only for SGI C++ compiler
#include <iostream>
#include "r3000_bus_interface.h"
#include "r3000_memory_access_unit.h"

r3000_memory_access_unit::request_buffer_base::request_buffer_base(void)
	: address_(0), type_(SINGLE), flag_(0)
{}

r3000_memory_access_unit::read_request_buffer::read_request_buffer(void)
{}

r3000_memory_access_unit::write_request_buffer::write_request_buffer(void)
	: data_(0)
{}

void r3000_memory_access_unit::send_request(void)
{
	if (instruction_read_is_requested() && instruction_read_is_ready()) {
		bus_if.request_ownership();
		inst_read_adr_ = inst_read_buf.address();
		inst_read_buf.clear();
		state = (inst_read_buf.is_multi()) ?
			MULTI_INSTRUCTION_READ_REQ_OWNER :
			SINGLE_INSTRUCTION_READ_REQ_OWNER;
		inst_read_state = T_WAIT;
	} else if (data_write_is_requested() && data_write_is_ready()) {
		bus_if.request_ownership();
		data_write_adr_ = data_write_buf.address();
		data_write_data_ = data_write_buf.data();
		data_write_buf.clear();
		state = DATA_WRITE_REQ_OWNER;
		data_write_state = T_WAIT;
	} else if (data_read_is_requested() && data_read_is_ready()) {
		bus_if.request_ownership();
		data_read_adr_ = data_read_buf.address();
		data_read_buf.clear();
		state = (data_read_buf.is_multi()) ?
			MULTI_DATA_READ_REQ_OWNER : SINGLE_DATA_READ_REQ_OWNER;
		data_read_state = T_WAIT;
	}
}

r3000_memory_access_unit::r3000_memory_access_unit
	(r3000_bus_interface& a, r3000_cp0_register_file& b)
	: bus_if(a),
	  cp0_rf(b),
	  state(READY),
	  inst_read_state(T_READY),
	  data_read_state(T_READY),
	  data_write_state(T_READY),
	  inst_cache_(NULL),
	  data_cache_(NULL)
{}

r3000_memory_access_unit::r3000_memory_access_unit
	(const r3000_memory_access_unit& a, r3000_bus_interface& b,
	 r3000_cp0_register_file& c)
	: bus_if(b),
	  cp0_rf(c),
	  state(a.state),
	  inst_read_buf(a.inst_read_buf),
	  data_read_buf(a.data_read_buf),
	  data_write_buf(a.data_write_buf),
	  inst_read_state(a.inst_read_state),
	  data_read_state(a.data_read_state),
	  data_write_state(a.data_write_state),
	  inst_read_adr_(a.inst_read_adr_),
	  data_read_adr_(a.data_read_adr_),
	  data_write_adr_(a.data_write_adr_),
	  inst_read_data_(a.inst_read_data_),
	  data_read_data_(a.data_read_data_),
	  data_write_data_(a.data_write_data_),
	  trans_count(a.trans_count),
	  inst_cache_(NULL),
	  data_cache_(NULL)
{}

r3000_memory_access_unit::~r3000_memory_access_unit()
{}

r3000_memory_access_unit& r3000_memory_access_unit::operator=
	(const r3000_memory_access_unit& a)
{
	if (this != &a) {
		state = a.state;
		inst_read_buf = a.inst_read_buf;
		data_read_buf = a.data_read_buf;
		data_write_buf = a.data_write_buf;
		inst_read_state = a.inst_read_state;
		data_read_state = a.data_read_state;
		data_write_state = a.data_write_state;
		inst_read_adr_ = a.inst_read_adr_;
		data_read_adr_ = a.data_read_adr_;
		data_write_adr_ = a.data_write_adr_;
		inst_read_data_ = a.inst_read_data_;
		data_read_data_ = a.data_read_data_;
		data_write_data_ = a.data_write_data_;
		trans_count = a.trans_count;
	}
	return *this;
}

void r3000_memory_access_unit::connect_instruction_cache
	(r3000_directmap_cache& a)
{
	inst_cache_ = &a;
}

void r3000_memory_access_unit::connect_data_cache(r3000_directmap_cache& a)
{
	data_cache_ = &a;
}

void r3000_memory_access_unit::disconnect_instruction_cache(void)
{
	inst_cache_ = NULL;
}

void r3000_memory_access_unit::disconnect_data_cache(void)
{
	data_cache_ = NULL;
}

void r3000_memory_access_unit::clock_in(void)
{
	switch (state) {
	case SINGLE_INSTRUCTION_READ_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = SINGLE_INSTRUCTION_READ_SEND;
		}
		break;
	case SINGLE_INSTRUCTION_READ_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_single_read_grant()) {
				state = SINGLE_INSTRUCTION_READ_GRANT;
			} else if (bus_if.is_single_read_ack()) {
				state = SINGLE_INSTRUCTION_READ_ACK;
				inst_read_state = T_ACK;
			} else if (bus_if.is_single_read_nack()) {
				state = SINGLE_INSTRUCTION_READ_NACK;
			}
		}
		break;
	case SINGLE_INSTRUCTION_READ_ACK:
		if (bus_if.is_single_read_data()) {
			inst_read_data_ = bus_if.data();
			state = SINGLE_INSTRUCTION_READ_TRANS;
			inst_read_state = T_TRANS;
		}
		break;
	case SINGLE_INSTRUCTION_READ_NACK:
		state = SINGLE_INSTRUCTION_READ_REQ_OWNER;
		break;
	case MULTI_INSTRUCTION_READ_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = MULTI_INSTRUCTION_READ_SEND;
		}
		break;
	case MULTI_INSTRUCTION_READ_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_multi_read_grant()) {
				state = MULTI_INSTRUCTION_READ_GRANT;
			} else if (bus_if.is_multi_read_ack()) {
				state = MULTI_INSTRUCTION_READ_ACK;
				inst_read_state = T_ACK;
			} else if (bus_if.is_multi_read_nack()) {
				state = MULTI_INSTRUCTION_READ_NACK;
			}
		}
		break;
	case MULTI_INSTRUCTION_READ_ACK:
		if (bus_if.is_multi_read_data()) {
			inst_read_data_ = bus_if.data();
			if (is_connected_to_instruction_cache()) {
				inst_cache().write(inst_read_adr_, inst_read_data_);
			}
			state = MULTI_INSTRUCTION_READ_TRANS;
			inst_read_state = T_TRANS;
		}
		break;
	case MULTI_INSTRUCTION_READ_NACK:
		state = MULTI_INSTRUCTION_READ_REQ_OWNER;
		break;
	case MULTI_INSTRUCTION_READ_TRANS:
		inst_read_adr_ += sizeof_data_type;
		inst_read_data_ = bus_if.data();
		if (is_connected_to_instruction_cache()) {
			inst_cache().write(inst_read_adr_, inst_read_data_);
		}
		if (bus_if.is_multi_read_data_last()) {
			state = MULTI_INSTRUCTION_READ_TRANS_LAST;
			inst_read_state = T_LAST;
		}
		break;
	case SINGLE_DATA_READ_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = SINGLE_DATA_READ_SEND;
		}
		break;
	case SINGLE_DATA_READ_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_single_read_grant()) {
				state = SINGLE_DATA_READ_GRANT;
			} else if (bus_if.is_single_read_ack()) {
				state = SINGLE_DATA_READ_ACK;
				data_read_state = T_ACK;
			} else if (bus_if.is_single_read_nack()) {
				state = SINGLE_DATA_READ_NACK;
			}
		}
		break;
	case SINGLE_DATA_READ_ACK:
		if (bus_if.is_single_read_data()) {
			data_read_data_ = bus_if.data();
			state = SINGLE_DATA_READ_TRANS;
			data_read_state = T_TRANS;
		}
		break;
	case SINGLE_DATA_READ_NACK:
		state = SINGLE_DATA_READ_REQ_OWNER;
		break;
	case MULTI_DATA_READ_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = MULTI_DATA_READ_SEND;
		}
		break;
	case MULTI_DATA_READ_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_multi_read_grant()) {
				state = MULTI_DATA_READ_GRANT;
			} else if (bus_if.is_multi_read_ack()) {
				state = MULTI_DATA_READ_ACK;
				data_read_state = T_ACK;
			} else if (bus_if.is_multi_read_nack()) {
				state = MULTI_DATA_READ_NACK;
			}
		}
		break;
	case MULTI_DATA_READ_ACK:
		if (bus_if.is_multi_read_data()) {
			data_read_data_ = bus_if.data();
			if (is_connected_to_data_cache()) {
				data_cache().write(data_read_adr_, data_read_data_);
			}
			state = MULTI_DATA_READ_TRANS;
			data_read_state = T_TRANS;
		}
		break;
	case MULTI_DATA_READ_NACK:
		state = MULTI_DATA_READ_REQ_OWNER;
		break;
	case MULTI_DATA_READ_TRANS:
		data_read_adr_ += sizeof_data_type;
		data_read_data_ = bus_if.data();
		if (is_connected_to_data_cache()) {
			data_cache().write(data_read_adr_, data_read_data_);
		}
		if (bus_if.is_multi_read_data_last()) {
			state = MULTI_DATA_READ_TRANS_LAST;
			data_read_state = T_LAST;
		}
		break;
	case DATA_WRITE_REQ_OWNER:
		if (bus_if.is_owner()) {
			trans_id = bus_if.transaction_id();
			state = DATA_WRITE_SEND;
		}
		break;
	case DATA_WRITE_WAIT:
		if (bus_if.transaction_id() == trans_id) {
			if (bus_if.is_single_write_grant()) {
				state = DATA_WRITE_GRANT;
			} else if (bus_if.is_single_write_ack()) {
				state = DATA_WRITE_ACK;
				data_write_state = T_ACK;
			} else if (bus_if.is_single_write_nack()) {
				state = DATA_WRITE_NACK;
			}
		}
		break;
	case DATA_WRITE_NACK:
		state = DATA_WRITE_REQ_OWNER;
		break;
	default:
		break;
	}
}

void r3000_memory_access_unit::clock_out(void)
{
	switch (state) {
	case READY:
		send_request();
		break;
	case SINGLE_INSTRUCTION_READ_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case SINGLE_INSTRUCTION_READ_SEND:
		bus_if.send_single_read_request(inst_read_adr_);
		state = SINGLE_INSTRUCTION_READ_WAIT;
		break;
	case SINGLE_INSTRUCTION_READ_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = SINGLE_INSTRUCTION_READ_WAIT;
		break;
	case SINGLE_INSTRUCTION_READ_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case SINGLE_INSTRUCTION_READ_TRANS:
		bus_if.clear();
		bus_if.release_ownership();
		state = READY;
		inst_read_state = T_READY;
		break;
	case MULTI_INSTRUCTION_READ_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case MULTI_INSTRUCTION_READ_SEND:
		bus_if.send_multi_read_request(inst_read_adr_,
			inst_cache().line_size_in_word());
		state = MULTI_INSTRUCTION_READ_WAIT;
		break;
	case MULTI_INSTRUCTION_READ_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = MULTI_INSTRUCTION_READ_WAIT;
		break;
	case MULTI_INSTRUCTION_READ_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case MULTI_INSTRUCTION_READ_TRANS_LAST:
		bus_if.clear();
		bus_if.release_ownership();
		state = READY;
		inst_read_state = T_READY;
		break;
	case SINGLE_DATA_READ_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case SINGLE_DATA_READ_SEND:
		bus_if.send_single_read_request(data_read_adr_);
		state = SINGLE_DATA_READ_WAIT;
		break;
	case SINGLE_DATA_READ_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = SINGLE_DATA_READ_WAIT;
		break;
	case SINGLE_DATA_READ_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case SINGLE_DATA_READ_TRANS:
		bus_if.clear();
		bus_if.release_ownership();
		state = READY;
		data_read_state = T_READY;
		break;
	case MULTI_DATA_READ_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case MULTI_DATA_READ_SEND:
		bus_if.send_multi_read_request(data_read_adr_,
			data_cache().line_size_in_word());
		state = MULTI_DATA_READ_WAIT;
		break;
	case MULTI_DATA_READ_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = MULTI_DATA_READ_WAIT;
		break;
	case MULTI_DATA_READ_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case MULTI_DATA_READ_TRANS_LAST:
		bus_if.clear();
		bus_if.release_ownership();
		state = READY;
		data_read_state = T_READY;
		break;
	case DATA_WRITE_REQ_OWNER:
		bus_if.request_ownership();
		break;
	case DATA_WRITE_SEND:
		bus_if.send_single_write_request(data_write_adr_);
		state = DATA_WRITE_WAIT;
		break;
	case DATA_WRITE_GRANT:
		bus_if.clear();
		bus_if.release_ownership();
		state = DATA_WRITE_WAIT;
		break;
	case DATA_WRITE_ACK:
		bus_if.send_single_write_data(data_write_data_);
		if (is_connected_to_data_cache() &&
			is_data_cache_hit(data_write_adr_)) {
			data_cache().write(data_write_adr_, data_write_data_);
		}
		state = DATA_WRITE_TRANS;
		data_write_state = T_TRANS;
		break;
	case DATA_WRITE_NACK:
		bus_if.clear();
		bus_if.release_ownership();
		break;
	case DATA_WRITE_TRANS:
		bus_if.clear();
		bus_if.release_ownership();
		state = READY;
		data_write_state = T_READY;
		break;
	default:
		break;
	}
}

void r3000_memory_access_unit::reset(void)
{
	state = READY;
	inst_read_buf.clear();
	data_read_buf.clear();
	data_write_buf.clear();
}

void r3000_memory_access_unit::output(ostream& os) const
{
	switch (state) {
	case READY:
		os << "ready";
		break;
	case SINGLE_INSTRUCTION_READ_REQ_OWNER:
		os << "single_instruction_read_request_ownership";
		break;
	case SINGLE_INSTRUCTION_READ_SEND:
		os << "single_instruction_read_send_request";
		break;
	case SINGLE_INSTRUCTION_READ_WAIT:
		os << "single_instruction_read_wait";
		break;
	case SINGLE_INSTRUCTION_READ_GRANT:
		os << "single_instruction_read_grant";
		break;
	case SINGLE_INSTRUCTION_READ_ACK:
		os << "single_instruction_read_ack";
		break;
	case SINGLE_INSTRUCTION_READ_NACK:
		os << "single_instruction_read_nack";
		break;
	case SINGLE_INSTRUCTION_READ_TRANS:
		os << "single_instruction_read_trans";
		break;
	case MULTI_INSTRUCTION_READ_REQ_OWNER:
		os << "multi_instruction_read_request_ownership";
		break;
	case MULTI_INSTRUCTION_READ_SEND:
		os << "multi_instruction_read_send_request";
		break;
	case MULTI_INSTRUCTION_READ_WAIT:
		os << "multi_instruction_read_wait";
		break;
	case MULTI_INSTRUCTION_READ_GRANT:
		os << "multi_instruction_read_grant";
		break;
	case MULTI_INSTRUCTION_READ_ACK:
		os << "multi_instruction_read_ack";
		break;
	case MULTI_INSTRUCTION_READ_NACK:
		os << "multi_instruction_read_nack";
		break;
	case MULTI_INSTRUCTION_READ_TRANS:
		os << "multi_instruction_read_trans";
		break;
	case MULTI_INSTRUCTION_READ_TRANS_LAST:
		os << "multi_instruction_read_trans_last";
		break;
	case SINGLE_DATA_READ_REQ_OWNER:
		os << "single_data_read_request_ownership";
		break;
	case SINGLE_DATA_READ_SEND:
		os << "single_data_read_send_request";
		break;
	case SINGLE_DATA_READ_WAIT:
		os << "single_data_read_wait";
		break;
	case SINGLE_DATA_READ_GRANT:
		os << "single_data_read_grant";
		break;
	case SINGLE_DATA_READ_ACK:
		os << "single_data_read_ack";
		break;
	case SINGLE_DATA_READ_NACK:
		os << "single_data_read_nack";
		break;
	case SINGLE_DATA_READ_TRANS:
		os << "single_data_read_trans";
		break;
	case MULTI_DATA_READ_REQ_OWNER:
		os << "multi_data_read_request_ownership";
		break;
	case MULTI_DATA_READ_SEND:
		os << "multi_data_read_send_request";
		break;
	case MULTI_DATA_READ_WAIT:
		os << "multi_data_read_wait";
		break;
	case MULTI_DATA_READ_GRANT:
		os << "multi_data_read_grant";
		break;
	case MULTI_DATA_READ_ACK:
		os << "multi_data_read_ack";
		break;
	case MULTI_DATA_READ_NACK:
		os << "multi_data_read_nack";
		break;
	case MULTI_DATA_READ_TRANS:
		os << "multi_data_read_trans";
		break;
	case MULTI_DATA_READ_TRANS_LAST:
		os << "multi_data_read_trans_last";
		break;
	case DATA_WRITE_REQ_OWNER:
		os << "data_write_request_ownership";
		break;
	case DATA_WRITE_SEND:
		os << "data_write_send_request";
		break;
	case DATA_WRITE_WAIT:
		os << "data_write_wait";
		break;
	case DATA_WRITE_GRANT:
		os << "data_write_grant";
		break;
	case DATA_WRITE_ACK:
		os << "data_write_ack";
		break;
	case DATA_WRITE_NACK:
		os << "data_write_nack";
		break;
	case DATA_WRITE_TRANS:
		os << "data_write_trans";
		break;
	}
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

ostream& operator<<(ostream& os, const r3000_memory_access_unit& a)
{
	if (os) a.output(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
	return os;
}
