/* -*- C++ -*-
 *
 * <<< sysinfo_map.h >>>
 *
 * --- Pseudo system information map class 'sysinfo_map'
 *     Copyright (C) 1999-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef SYSINFO_MAP_H
#define SYSINFO_MAP_H 1

#include <iostream>
#include <string>
#include <map>
#include <isis/root_object.h>

template <class T>
class sysinfo_map
{
private:
	class entry
	{
	private:
		typedef entry thisclass;
		enum { INTEGER, STRING } type_;
		T int_value_;
		string str_value_;
	public:
		entry(void)
			: type_(INTEGER), int_value_(T(0)) {}
		entry(const thisclass& a)
			: type_(a.type_),
			  int_value_(a.int_value_),
			  str_value_(a.str_value_) {}
		thisclass& operator=(const thisclass& a)
		{
			if (this != &a) {
				type_ = a.type_, int_value_ = a.int_value_,
				str_value_ = a.str_value_;
			}
			return *this;
		}
		void output(ostream& os) const;
		void set(T a) { type_ = INTEGER; int_value_ = a; }
		void set(const string& a) { type_ = STRING; str_value_ = a; }
		bool is_integer(void) const { return type_ == INTEGER; }
		bool is_string(void) const { return type_ == STRING; }
		T get_integer(void) const { return int_value_; }
		const string& get_string(void) const { return str_value_; }
	};
	typedef sysinfo_map<T> thisclass;
	typedef map< string, entry > container_type;
	container_type buf;
public:
	sysinfo_map(void);
	sysinfo_map(const thisclass&);
	~sysinfo_map();
	void output(ostream&) const;
	void set(const string&, T);
	void set(const string&, const string&);
	void unset(const string&);
	void clear(void);
	bool is_defined(const string& a) const
		{ return buf.count(a) > 0; }
	bool is_integer(const string& a) const
		{ return is_defined(a) ? buf.find(a)->second.is_integer() : false; }
	bool is_string(const string& a) const
		{ return is_defined(a) ? buf.find(a)->second.is_string() : false; }
	T get_integer(const string& a) const
	{
		return is_integer(a) ? buf.find(a)->second.get_integer() : T();
	}
	const string& get_string(const string& a) const
	{
		static string dummy;
		return is_string(a) ? buf.find(a)->second.get_string() : dummy;
	}
};

template <class T>
ostream& operator<<(ostream& os, const sysinfo_map<T>& a)
{
	if (os) a.output(os);
	return os;
}

template <class T>
sysinfo_map<T>::sysinfo_map(void)
{}

template <class T>
sysinfo_map<T>::sysinfo_map(const sysinfo_map<T>& a)
	: buf(a.buf)
{}

template <class T>
sysinfo_map<T>::~sysinfo_map()
{}

template <class T>
void sysinfo_map<T>::set(const string& a, T b)
{
	buf[a].set(b);
}

template <class T>
void sysinfo_map<T>::set(const string& a, const string& b)
{
	buf[a].set(b);
}

template <class T>
void sysinfo_map<T>::unset(const string& a)
{
	if (!is_defined(a)) return;
	buf.erase(buf.find(a));
}

template <class T>
void sysinfo_map<T>::clear(void)
{
	buf.clear();
}

template <class T>
void sysinfo_map<T>::entry::output(ostream& os) const
{
	if (is_integer()) {
		const long flags = os.flags();
		os << "0x" << hex << get_integer();
		os.flags(flags);
	} else {
		os << '"' << get_string() << '"';
	}
}

template <class T>
void sysinfo_map<T>::output(ostream& os) const
{
	typedef typename container_type::const_iterator CI;
	for (CI p = buf.begin(); p != buf.end(); p++) {
		if (p != buf.begin()) os << '\n';
		os << '"' << p->first << "\" -> ";
		p->second.output(os);
	}
}

#endif /* SYSINFO_MAP_H */
