/* -*- C++ -*-
 *
 * <<< memory.h >>>
 *
 * --- Memory class 'memory'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MEMORY_H
#define MEMORY_H 1

#include <algorithm>
#include <climits>
#include <hash_map>
#include <iostream>
#include <iomanip>
#include <isis/c_array.h>
#include <isis/root_object.h>
#include <isis/integer_hash_map.h>

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char>
class memory : public root_object
{
public:
	typedef A address_type;
	typedef D data_type;
	typedef Ch char_type;
	typedef A size_type;
	static const size_type sizeof_data_type = Dsz;
private:
	typedef memory<A, D, Dsz, Ch> thisclass;
	typedef root_object inherited;
	enum endian_t { BIG, LITTLE };
	static const size_type default_hash_size_ = 4093;
	static const size_type item_size = 4096;
	static const size_type item_char_size = item_size * Dsz;
	address_type size_;
	typedef c_array<data_type, item_size> data_container_type;
	integer_hash_map<address_type, data_container_type> buf;
	endian_t endian;
	static address_type to_aligned_address(address_type a)
		{ return a - (a % sizeof_data_type); }
	address_type to_base_address(address_type a) const
		{ return a - (a % item_char_size); }
	size_type to_offset(address_type a) const
		{ return size_type((a / sizeof_data_type) % item_size); }
public:
	memory(void);
	explicit memory(size_type);
	memory(const thisclass&);
	virtual ~memory();
	virtual void output(ostream&) const;
	size_type size(void) const { return size_; }
	bool is_valid(address_type a) const { return a >= 0 && a < size(); }
	bool is_big_endian(void) const { return endian == BIG; }
	bool is_little_endian(void) const { return !is_big_endian(); }
	void set_big_endian(void) { endian = BIG; }
	void set_little_endian(void) { endian = LITTLE; }
	inline data_type read(address_type) const;
	inline char_type read_char(address_type) const;
	inline void write(address_type, data_type);
	inline void write_char(address_type, char_type);
	void resize(size_type);
	void clear(void);
	void dump(ostream&) const;
};

template <class A, class D, size_t Dsz, class Ch>
memory<A, D, Dsz, Ch>::memory(void)
	: size_(size_type(0)),
	  buf(),
	  endian(BIG)
{
	buf.resize(default_hash_size_);
}

template <class A, class D, size_t Dsz, class Ch>
memory<A, D, Dsz, Ch>::memory
	(memory<A, D, Dsz, Ch>::size_type a)
	: size_(a > 0 ? to_aligned_address(a) : size_type(0)),
	  buf(),
	  endian(BIG)
{
	buf.resize(default_hash_size_);
}

template <class A, class D, size_t Dsz, class Ch>
memory<A, D, Dsz, Ch>::memory(const memory<A, D, Dsz, Ch>& a)
	: size_(a.size_),
	  buf(a.buf),
	  endian(a.endian)
{}

template <class A, class D, size_t Dsz, class Ch>
memory<A, D, Dsz, Ch>::~memory()
{}

template <class A, class D, size_t Dsz, class Ch>
inline memory<A, D, Dsz, Ch>::data_type memory<A, D, Dsz, Ch>::read
	(memory<A, D, Dsz, Ch>::address_type a) const
{
#ifdef DEBUG
	assert(is_valid(a));
#endif // DEBUG
	const address_type base = to_base_address(a);
	const data_container_type* p = buf.search(base);
	if (p != NULL) {
		return p->operator[](to_offset(a)); // for gcc-2.95.1
	} else {
		return data_type(0);
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline memory<A, D, Dsz, Ch>::char_type memory<A, D, Dsz, Ch>::read_char
	(memory<A, D, Dsz, Ch>::address_type a) const
{
	size_type shiftbit = ((is_big_endian()) ?
						  (sizeof_data_type - 1 - (a % sizeof_data_type)) :
						  (a % sizeof_data_type))
						 * sizeof(char_type) * CHAR_BIT;
	data_type dat = read(to_aligned_address(a));
	return char_type((dat >> shiftbit) &
					 ~(~0 << (sizeof(char_type) * CHAR_BIT)));
}

template <class A, class D, size_t Dsz, class Ch>
inline void memory<A, D, Dsz, Ch>::write
	(memory<A, D, Dsz, Ch>::address_type a, memory<A, D, Dsz, Ch>::data_type b)
{
#ifdef DEBUG
	assert(is_valid(a));
#endif // DEBUG
	address_type base = to_base_address(a);
	data_container_type* p = buf.search(base);
	if (p != NULL) {
		p->operator[](to_offset(a)) = b;
	} else {
		if (b == data_type(0)) return;
		data_container_type& v(buf[base]);
		fill(v.begin(), v.end(), data_type(0));
		v.operator[](to_offset(a)) = b; // for gcc-2.95.1
		if (buf.size() * 2 > buf.bucket_count()) {
			buf.resize(buf.size() * 2 - 1);
		}
	}
}

template <class A, class D, size_t Dsz, class Ch>
inline void memory<A, D, Dsz, Ch>::write_char
	(memory<A, D, Dsz, Ch>::address_type a, memory<A, D, Dsz, Ch>::char_type b)
{
	address_type adr = to_aligned_address(a);
	size_type shiftbit = ((is_big_endian()) ?
						  (sizeof_data_type - 1 - (a % sizeof_data_type)) :
						  (a % sizeof_data_type))
						 * sizeof(char_type) * CHAR_BIT;
	data_type mask = (~(~0 << (sizeof(char_type) * CHAR_BIT)) << shiftbit);
	data_type old_value = read(adr);
	data_type new_value =
		((old_value & ~mask) | ((data_type(b) << shiftbit) & mask));
	if (new_value != old_value) write(adr, new_value);
}

template <class A, class D, size_t Dsz, class Ch>
void memory<A, D, Dsz, Ch>::resize(memory<A, D, Dsz, Ch>::size_type a)
{
	size_ = to_aligned_address(a);
}

template <class A, class D, size_t Dsz, class Ch>
void memory<A, D, Dsz, Ch>::clear(void)
{
	buf.clear();
}

template <class A, class D, size_t Dsz, class Ch>
void memory<A, D, Dsz, Ch>::dump(ostream& os) const
{
	if (size() == 0) return;
	size_type width;
	{
		size_type tmp = ((79 - (sizeof(address_type) * 2 + 1)) /
						 (sizeof(data_type) * 2 + 1)) - 1;
		int i = 0;
		while (tmp > 1) tmp >>= 1, i++;
		width = (1 << i);
	}
	const char fill = os.fill();
	const long flags = os.flags();
	os << hex << setfill('0');
	size_type i = 0;
	for (address_type adr = 0; adr < size(); adr += sizeof_data_type) {
		if (i == 0) os << setw(sizeof(address_type) * 2) << adr << ':';
		os << ' ' << setw(sizeof(data_type) * 2) << read(adr);
		if (++i == width && adr + sizeof_data_type < size()) {
			i = 0;
			os << '\n';
		}
	}
	os.fill(fill);
	os.flags(flags);
}

template <class A, class D, size_t Dsz, class Ch>
void memory<A, D, Dsz, Ch>::output(ostream& os) const
{
	dump(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* MEMORY_H */
