/* -*- C++ -*-
 *
 * <<< mapped_memory.h >>>
 *
 * --- Mapped Memory class 'mapped_memory'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef MAPPED_MEMORY_H
#define MAPPED_MEMORY_H 1

#include <iostream>
#include <iomanip>
#include <isis/root_object.h>
#include <isis/memory.h>

template <class A, class D = A, size_t Dsz = sizeof(D), class Ch = char>
class mapped_memory : public root_object
{
public:
	typedef typename memory<A, D, Dsz, Ch>::address_type address_type;
	typedef typename memory<A, D, Dsz, Ch>::data_type data_type;
	typedef typename memory<A, D, Dsz, Ch>::char_type char_type;
	typedef typename memory<A, D, Dsz, Ch>::size_type size_type;
	static const size_t sizeof_data_type =
		memory<A, D, Dsz, Ch>::sizeof_data_type;
private:
	typedef mapped_memory<A, D, Dsz, Ch> thisclass;
	typedef root_object inherited;
	memory<A, D, Dsz, Ch> buf;
	address_type top_;
	static address_type to_aligned_address(address_type a)
		{ return a - (a % Dsz); }
public:
	mapped_memory(void);
	explicit mapped_memory(address_type, size_type);
	mapped_memory(const thisclass&);
	virtual ~mapped_memory();
	virtual void output(ostream&) const;
	address_type top(void) const { return top_; }
	size_type size(void) const { return buf.size(); }
	address_type bottom(void) const { return top() + size(); }
	bool is_valid(address_type a) const { return a >= top() && a < bottom(); }
	void set_top(address_type);
	void resize(size_type);
	bool is_big_endian(void) const { return buf.is_big_endian(); }
	bool is_little_endian(void) const { return buf.is_little_endian(); }
	void set_big_endian(void) { buf.set_big_endian(); }
	void set_little_endian(void) { buf.set_little_endian(); }
	data_type read(address_type a) const { return buf.read(a - top()); }
	char_type read_char(address_type a) const
		{ return buf.read_char(a - top()); }
	void write(address_type a, data_type b)
		{ buf.write(a - top(), b); }
	void write_char(address_type a, char_type b)
		{ buf.write_char(a - top(), b); }
	void clear(void);
	void dump(ostream&) const;
	void dump(ostream&, address_type, size_type) const;
};

template <class A, class D, size_t Dsz, class Ch>
mapped_memory<A, D, Dsz, Ch>::mapped_memory(void)
	: buf(),
	  top_(address_type(0))
{}

template <class A, class D, size_t Dsz, class Ch>
mapped_memory<A, D, Dsz, Ch>::mapped_memory
	(mapped_memory<A, D, Dsz, Ch>::address_type a,
	 mapped_memory<A, D, Dsz, Ch>::size_type b)
	: buf(b),
	  top_(a > 0 ? to_aligned_address(a) : address_type(0))
{}

template <class A, class D, size_t Dsz, class Ch>
mapped_memory<A, D, Dsz, Ch>::mapped_memory
	(const mapped_memory<A, D, Dsz, Ch>& a)
	: buf(a.buf),
	  top_(a.top_)
{}

template <class A, class D, size_t Dsz, class Ch>
mapped_memory<A, D, Dsz, Ch>::~mapped_memory()
{}

template <class A, class D, size_t Dsz, class Ch>
void mapped_memory<A, D, Dsz, Ch>::set_top
	(mapped_memory<A, D, Dsz, Ch>::address_type a)
{
	top_ = (a > 0 ? to_aligned_address(a) : address_type(0));
}

template <class A, class D, size_t Dsz, class Ch>
void mapped_memory<A, D, Dsz, Ch>::resize
	(mapped_memory<A, D, Dsz, Ch>::size_type a)
{
	buf.resize(a);
}

template <class A, class D, size_t Dsz, class Ch>
void mapped_memory<A, D, Dsz, Ch>::clear(void)
{
	buf.clear();
}

template <class A, class D, size_t Dsz, class Ch>
void mapped_memory<A, D, Dsz, Ch>::dump(ostream& os) const
{
	dump(os, top(), size());
}

template <class A, class D, size_t Dsz, class Ch>
void mapped_memory<A, D, Dsz, Ch>::dump
	(ostream& os,
	 mapped_memory<A, D, Dsz, Ch>::address_type a,
	 mapped_memory<A, D, Dsz, Ch>::size_type b) const
{
	if (size() == 0) return;
	size_type width;
	{
		size_t tmp = ((79 - (sizeof(address_type) * 2 + 1)) /
					  (sizeof(data_type) * 2 + 1)) - 1;
		int i = 0;
		while (tmp > 1) tmp >>= 1, i++;
		width = (1 << i);
	}
	const size_t char_in_line = width * sizeof_data_type;
	address_type a_, b_;
	if (a % char_in_line != 0) {
		a_ = a - a % char_in_line;
		b_ = b + a % char_in_line;
	} else {
		a_ = a, b_ = b;
	}
	address_type start = (a_ >= top()) ? a_ : top(),
				 end = (a_ + b_ <= bottom()) ? (a_ + b_) : bottom();
	const char fill = os.fill();
	const long flags = os.flags();
	os << hex << setfill('0');
	size_t i = 0;
	for (address_type adr = start; adr < end; adr += sizeof_data_type) {
		if (i == 0) os << setw(sizeof(address_type) * 2) << adr << ':';
		os << ' ' << setw(sizeof(data_type) * 2) << read(adr);
		if (++i == width && adr + sizeof_data_type < end) {
			i = 0;
			os << '\n';
		}
	}
	os.fill(fill);
	os.flags(flags);
}

template <class A, class D, size_t Dsz, class Ch>
void mapped_memory<A, D, Dsz, Ch>::output(ostream& os) const
{
	dump(os);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* MAPPED_MEMORY_H */
