/* -*- C++ -*-
 *
 * <<< cache_line_set.h >>>
 *
 * --- Cache line set class 'cache_line_set'
 *     Copyright (C) 1995-2001 Amano Lab., Keio University. ---
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
 */

#ifndef CACHE_LINE_SET_H
#define CACHE_LINE_SET_H 1

#include <iostream>
#include <iomanip>
#include <vector>
#include <isis/limited_counter.h>
#include <isis/root_object.h>

template <class L>
class cache_line_set : public root_object
{
public:
	typedef vector<L> container_type;
	typedef typename L::address_type address_type;
	typedef typename L::data_type data_type;
	typedef L line_type;
	typedef limited_counter<size_t> counter_type;
	typedef typename container_type::size_type line_set_size_type;
	typedef typename line_type::size_type line_size_type;
private:
	typedef cache_line_set<L> thisclass;
	typedef root_object inherited;
	static const counter_type::value_type default_max_lru_counter = 1;
	container_type buf;
	vector< limited_counter<size_t> > lru;
public:
	cache_line_set(void);
	explicit cache_line_set(line_set_size_type, line_size_type);
	virtual ~cache_line_set();
	virtual void output(ostream&) const;
	const line_type& operator[](int i) const { return buf[i]; }
	line_type& operator[](int i) { return buf[i]; }
	line_set_size_type size(void) const { return buf.size(); }
	void resize(line_set_size_type);
	line_size_type line_size(void) const
		{ return size() > 0 ? buf[0].size() : 0; }
	void line_resize(line_size_type);
	int way_of_hit(address_type a) const
	{
		for (line_set_size_type i = 0; i < size(); i++) {
			if (buf[i].is_hit(a)) return i;
		}
		return -1;
	}
	int way_of_throw(void) const
	{
		int tmp = lru[0], idx = 0;
		for (line_set_size_type i = 1; tmp > 0 && i < size(); i++) {
			int new_val = lru[i];
			if (tmp > new_val) tmp = new_val, idx = i;
		}
		return idx;
	}
	int way_of_empty(void) const
	{
		for (line_set_size_type i = 0; i < size(); i++) {
			if (!buf[i].is_valid()) return i;
		}
		return -1;
	}
	bool is_hit(address_type a) const { return way_of_hit(a) >= 0; }
	bool is_full(void) const
	{
		for (line_set_size_type i = 0; i < size(); i++) {
			if (!buf[i].is_valid()) return 0;
		}
		return 1;
	}
	const counter_type& lru_counter(line_set_size_type a) const
		{ return lru[a]; }
	line_set_size_type increment_lru_counter(line_set_size_type a)
		{ return lru[a].increment(); }
	void clear_lru_counter(void)
		{ for (line_set_size_type i = 0; i < size(); i++) lru[i].clear(); }
	void clear_lru_counter(line_set_size_type a) { lru[a].clear(); }
	line_set_size_type max_lru_counter(void)
		{ return size() > 0 ? lru[0].max() : default_max_lru_counter; }
	void set_max_lru_counter(counter_type::value_type);
	void clear(void);
};

template <class L>
cache_line_set<L>::cache_line_set(void)
{}

template <class L>
cache_line_set<L>::cache_line_set
	(cache_line_set<L>::line_set_size_type a,
	 cache_line_set<L>::line_size_type b)
	: buf(a),
	  lru(a)
{
	line_resize(b);
	set_max_lru_counter(default_max_lru_counter);
}

template <class L>
cache_line_set<L>::~cache_line_set()
{}

template <class L>
void cache_line_set<L>::resize(cache_line_set<L>::line_set_size_type a)
{
	line_size_type linesize = line_size();
	counter_type::value_type max_lru_counter_ = max_lru_counter();
	buf.resize(a);
	lru.resize(a);
	set_max_lru_counter(max_lru_counter_);
	line_resize(linesize);
}

template <class L>
void cache_line_set<L>::line_resize(cache_line_set<L>::line_size_type a)
{
	for (size_t i = 0; i < size(); i++) buf[i].resize(a);
}

template <class L>
void cache_line_set<L>::set_max_lru_counter
	(cache_line_set<L>::counter_type::value_type a)
{
	for (line_set_size_type i = 0; i < size(); i++) lru[i].set_max(a);
}

template <class L>
void cache_line_set<L>::clear(void)
{
	for (line_set_size_type i = 0; i < size(); i++) buf[i].set_invalid();
}

template <class L>
void cache_line_set<L>::output(ostream& os) const
{
	const char fill = os.fill();
	const long flags = os.flags();
	os << hex << setfill('0');
	for (line_set_size_type i = 0; i < buf.size(); i++) {
		os << buf[i];
		if (buf[i].is_valid()) os << ' ' << lru_counter(i);
		if (i < buf.size() - 1) os << '\n';
	}
	os.fill(fill);
	os.flags(flags);
#ifdef DEBUG
	os.flush();
#endif // DEBUG
}

#endif /* CACHE_LINE_SET_H */
